"""State data models for organized collections."""

from __future__ import annotations

from datetime import datetime, timezone
from typing import Dict, List, Literal, Optional

from pydantic import BaseModel, Field


class FileRecord(BaseModel):
    """Metadata describing a file in an organized collection.

    Attributes:
        path: Relative path to the file within the collection root.
        hash: Optional hash representing the file contents.
        tags: Tags inferred during ingestion.
        categories: Explicit categories associated with the file.
        confidence: Confidence score from classification.
        last_modified: Timestamp for the last modification.
        rename_suggestion: Suggested filename (without extension) generated by classification.
        reasoning: Optional explanation accompanying the classification decision.
        needs_review: Flag indicating the file requires human attention.
        vision_caption: Caption text generated for images when vision processing is enabled.
        vision_labels: Labels derived from the captioning step.
        vision_confidence: Confidence reported by the captioning model.
        vision_reasoning: Optional reasoning accompanying the caption output.
    """

    path: str
    hash: Optional[str] = None
    tags: List[str] = Field(default_factory=list)
    categories: List[str] = Field(default_factory=list)
    confidence: Optional[float] = None
    last_modified: Optional[datetime] = None
    rename_suggestion: Optional[str] = None
    reasoning: Optional[str] = None
    needs_review: bool = False
    vision_caption: Optional[str] = None
    vision_labels: List[str] = Field(default_factory=list)
    vision_confidence: Optional[float] = None
    vision_reasoning: Optional[str] = None


class OperationEvent(BaseModel):
    """History entry describing a single organization operation.

    Attributes:
        timestamp: Moment the operation executed.
        operation: Operation type identifier (rename, move, or delete).
        source: Original path for the entry (relative to the collection when possible).
        destination: Path after applying the operation, when applicable.
        conflict_strategy: Conflict policy applied while executing the operation.
        conflict_applied: Indicates whether a conflict was encountered.
        notes: Optional freeform notes about the event.
    """

    timestamp: datetime
    operation: Literal["rename", "move", "delete"]
    source: str
    destination: Optional[str] = None
    conflict_strategy: Optional[str] = None
    conflict_applied: bool = False
    notes: List[str] = Field(default_factory=list)


class CollectionState(BaseModel):
    """Aggregate metadata for an organized directory.

    Attributes:
        root: Absolute path to the tracked directory.
        files: Mapping of relative paths to file records.
        created_at: Timestamp capturing when the collection was first tracked.
        updated_at: Timestamp capturing the latest mutation time.
    """

    root: str
    files: Dict[str, FileRecord] = Field(default_factory=dict)
    created_at: datetime = Field(default_factory=lambda: datetime.now(timezone.utc))
    updated_at: datetime = Field(default_factory=lambda: datetime.now(timezone.utc))


__all__ = ["FileRecord", "CollectionState", "OperationEvent"]
