import logging
import socket

from typing import Any
from threading import Event
from contextvars import ContextVar

from .utils.timer import Timer


class DatabaseBackend:
    connection: Any
    cursor: Any
    contextConnection: ContextVar[Any | None]
    contextAsyncConnection: ContextVar[Any | None]

    config: Any

    connectionTimeout: int
    slowDownTimeout: int = 5

    name: str
    logger: logging.Logger
    timer: ContextVar[Timer | None]

    shutdownRequested: Event

    def __init__(
        self,
        dbConfig: Any,
        connectionTimeout: int = 5,
        instanceName: str = "database_backend",
    ) -> None:
        """
        Main concept here is that in init we do not connect to database,
        so that class instances can be safely made regardless of connection statuss.

        Remember to call open() before using this class.
        Close will be called automatically when class is destroyed.
        But sometimes in async environment you should call close() proactively.
        """

        self.config = dbConfig
        self.connectionTimeout = connectionTimeout
        self.name = instanceName

        loggerName = f"{__name__}.{self.__class__.__name__}.{self.name}"
        self.logger = logging.getLogger(loggerName)
        self.timer = ContextVar(f"db_timer", default=None)

        self.connection = None
        self.cursor = None
        self.shutdownRequested = Event()
        self.contextConnection = ContextVar(f"pg_connection_{self.name}", default=None)
        self.contextAsyncConnection = ContextVar(
            f"pg_async_connection_{self.name}", default=None
        )

    def __del__(self) -> None:
        """What to do when class is destroyed"""
        self.logger.debug("Dealloc")
        self.close()

    # Context
    def __enter__(self) -> tuple[Any, Any]:
        """Context manager"""
        raise Exception("Not implemented")

    def __exit__(self, exc_type: Any, exc_value: Any, traceback: Any) -> None:
        """Context manager"""
        raise Exception("Not implemented")

    async def __aenter__(self) -> tuple[Any, Any]:
        """Context manager"""
        raise Exception("Not implemented")

    async def __aexit__(self, exc_type: Any, exc_value: Any, traceback: Any) -> None:
        """Context manager"""
        raise Exception("Not implemented")

    # Connection
    def open(self) -> None:
        """Connect to database"""
        raise Exception("Not implemented")

    async def openAsync(self) -> None:
        """Connect to database"""
        raise Exception("Not implemented")

    def close(self) -> None:
        """Close connections"""
        if self.cursor:
            self.logger.debug("Closing cursor")
            self.cursor.close()
            self.cursor = None

        if self.connection:
            self.logger.debug("Closing connection")
            self.connection.close()
            self.connection = None

    def fixSocketTimeouts(self, fd: Any):
        # Lets do some socket magic
        s = socket.fromfd(fd, socket.AF_INET, socket.SOCK_STREAM)
        # Enable sending of keep-alive messages
        s.setsockopt(socket.SOL_SOCKET, socket.SO_KEEPALIVE, 1)
        # Time the connection needs to remain idle before start sending
        # keepalive probes
        s.setsockopt(socket.IPPROTO_TCP, socket.TCP_KEEPIDLE, self.connectionTimeout)
        # Time between individual keepalive probes
        s.setsockopt(socket.IPPROTO_TCP, socket.TCP_KEEPINTVL, 1)
        # The maximum number of keepalive probes should send before dropping
        # the connection
        s.setsockopt(socket.IPPROTO_TCP, socket.TCP_KEEPCNT, 3)
        # To set timeout for an RTO you must set TCP_USER_TIMEOUT timeout
        # (in milliseconds) for socket.
        s.setsockopt(
            socket.IPPROTO_TCP, socket.TCP_USER_TIMEOUT, self.connectionTimeout * 1000
        )

    async def newConnection(
        self,
    ) -> tuple[Any, Any] | None:
        """
        Create new connection

        Used for async context manager and async connection creation

        Returns:
            tuple[Any, Any] | None: Connection and cursor
        """
        raise Exception("Not implemented")

    async def returnConnection(self, connection: Any) -> None:
        """
        Return connection to pool

        Used for async context manager and async connections return.
        For example to return connection to a pool.

        Args:
            connection (Any): Connection to return to pool
        """
        raise Exception("Not implemented")

    # Data
    def lastInsertId(self) -> int:
        """Get last inserted row id generated by auto increment"""
        raise Exception("Not implemented")

    def affectedRows(self) -> int:
        """Get affected rows count"""
        raise Exception("Not implemented")

    def commit(self) -> None:
        """Commit DB queries"""
        raise Exception("Not implemented")

    def rollback(self) -> None:
        """Rollback DB queries"""
        raise Exception("Not implemented")
