from .api import Quantity, NameQuantityPair
from .api import JOB_STATUS, STATUS_JOB
import random
import requests
import pint
from webstompy import StompListener
from tqdm.auto import tqdm
import numpy as np
import logging
import uuid
import asyncio
import json
from uuid import uuid4
from typing import Any, Tuple

logger = logging.getLogger(__name__)

q = pint.get_application_registry()


def decode(enc: dict) -> "pint.Quantity":
    """Decode a quantity encoded object

    Parameters
    ----------
    enc : dict
        The encoded object

    Returns
    -------
    pint.Quantity
        The decoded quantity object
    """

    units_tuple: Tuple[Tuple[str, int], ...] = tuple(
        (e["name"], e["exponent"]) for e in enc.get("units", ())
    )

    # magnitude can be a single value or an array represented as a list
    if len(enc["magnitude"]) != 1:
        enc_tuple: Tuple[Any, Tuple[Tuple[str, int], ...]] = (
            np.array(enc["magnitude"], dtype=np.float64).reshape(enc["shape"]),
            units_tuple,
        )
    else:
        enc_tuple = (enc["magnitude"][0], units_tuple)

    try:
        quant: "pint.Quantity" = q.Quantity.from_tuple(enc_tuple)
        quant.ito_base_units()
    except Exception as exc:
        logger.error(
            "Error decoding %s with units %s: %s",
            enc.get("magnitude"),
            enc.get("units"),
            exc,
        )
        raise

    logger.debug("convert %s -> %s", enc, quant)
    return quant


class Material(object):
    def __init__(self, name, reference, key_words=[], material_properties={}, id=None):
        """
        Parameters
        ----------
        name : str
            The name of the material
        reference : str
            The reference of the material, normally a URL or a file path
        key_words : list, optional
            A list of key_words associated with the material, by default []
        data : list of NameQuantityPair, optional
            A list of NameQuantityPair objects containing the material properties, by default []
        material_id : str, optional
            The unique identifier for the material, by default None. If None, a new UUID will be generated by mongoDB.
        """
        self.name = name
        self.reference = reference
        self.key_words = key_words
        self.material_properties = material_properties
        self.id = id

    def __repr__(self) -> str:
        return f"Material(name={self.name}, material_id={self.id})"

    def to_api(self):
        return {
            "reference": self.reference,
            "name": self.name,
            "key_words": self.key_words,
            "data": [
                NameQuantityPair(
                    "material_properties",
                    k,
                    Quantity(*self.material_properties[k].to_tuple()),
                ).to_dict()
                for k in self.material_properties.keys()
            ],
        }

    @classmethod
    def from_api(cls, db_material):
        """Load the material from the API db_material"""
        material = cls(
            name=db_material["name"],
            reference=db_material.get("reference", ""),
            key_words=db_material.get("key_words", []),
            material_properties={
                k["name"]: decode(k["value"])
                for k in db_material["data"]
                if k["section"] == "material_properties"
            },
            id=db_material.get("id", str(uuid4())),
        )
        return material


class Machine(object):
    def __init__(self, stator, rotor, winding, materials=None):

        self.stator = stator
        self.rotor = rotor
        self.winding = winding
        if materials is not None:
            self.materials = materials
        else:
            self.materials = {
                "rotor_lamination": "66018e5d1cd3bd0d3453646f",  # default M230-35A
                "rotor_magnet": "66018e5b1cd3bd0d3453646c",  # default is N35UH
                "rotor_air_L": "6602fb42c4a87c305481e8a6",
                "rotor_air_R": "6602fb42c4a87c305481e8a6",
                "rotor_banding": "6602fb42c4a87c305481e8a6",
                "stator_lamination": "66018e5d1cd3bd0d3453646f",  # default M230-35A
                "stator_slot_wedge": "6602fb7239bfdea291a25dd7",
                "stator_slot_liner": "6602fb5166d3c6adaa8ebe8c",
                "stator_slot_winding": "66018e5d1cd3bd0d34536470",
                "stator_slot_potting": "6602fd41b8e866414fe983ec",
            }

    def __repr__(self) -> str:
        return f"Machine({self.stator}, {self.rotor}, {self.winding})"

    def to_api(self):
        stator_api = [
            NameQuantityPair("stator", k, Quantity(*self.stator[k].to_tuple()))
            for k in self.stator
        ]
        rotor_api = [
            NameQuantityPair("rotor", k, Quantity(*self.rotor[k].to_tuple()))
            for k in self.rotor
        ]
        winding_api = [
            NameQuantityPair("winding", k, Quantity(*self.winding[k].to_tuple()))
            for k in self.winding
        ]
        data = []
        data.extend(list(x.to_dict() for x in stator_api))
        data.extend(list(x.to_dict() for x in rotor_api))
        data.extend(list(x.to_dict() for x in winding_api))
        return data


class Job(object):
    def __init__(
        self,
        machine: Machine,
        operating_point,
        simulation,
        mesh_reuse_series="",
        netlist=None,
        title=None,
    ):
        if title is None:
            self.title = self.generate_title()
        else:
            self.title = title
        self.type = "electromagnetic_spmbrl_fscwseg"
        self.status = 0
        self.machine = machine
        self.operating_point = operating_point
        self.simulation = simulation
        if mesh_reuse_series:
            mesh_reuse_series = mesh_reuse_series
        else:
            mesh_reuse_series = str(uuid.uuid4())
        if netlist:
            netliststr = json.dumps(netlist)
        else:
            netliststr = ""
        self._string_data = {
            "mesh_reuse_series": mesh_reuse_series,
            "netlist": netliststr,
        }

    @property
    def netlist(self):
        if self._string_data["netlist"] == "":
            return
        else:
            return json.loads(self._string_data["netlist"])

    @netlist.setter
    def netlist(self, value):
        self._string_data["netlist"] = json.dumps(value)

    @property
    def mesh_reuse_series(self):
        return self._string_data["mesh_reuse_series"]

    @mesh_reuse_series.setter
    def mesh_reuse_series(self, value):
        if not isinstance(value, str):
            raise ValueError("mesh_reuse_series must be a string")
        self._string_data["mesh_reuse_series"] = value

    def __repr__(self) -> str:
        return f"Job({self.machine}, {self.operating_point}, {self.simulation})"

    def generate_title(self):
        "gets a random title from the wordlists"
        random_offset = random.randint(500, 286797)
        response = requests.get(
            "https://github.com/taikuukaits/SimpleWordlists/raw/master/Wordlist-Adjectives-All.txt",
            headers={
                "Range": "bytes={1}-{0}".format(random_offset, random_offset - 500),
                "accept-encoding": "identity",
            },
        )
        adjective = random.choice(response.text.splitlines()[1:-1])
        random_offset = random.randint(500, 871742)
        response = requests.get(
            "https://github.com/taikuukaits/SimpleWordlists/raw/master/Wordlist-Nouns-All.txt",
            headers={
                "Range": "bytes={1}-{0}".format(random_offset, random_offset - 500),
                "accept-encoding": "identity",
            },
        )
        noun = random.choice(response.text.splitlines()[1:-1])
        title = f"{adjective}-{noun}"
        return title

    def to_api(self):
        job = {
            "status": 0,
            "title": self.title,
            "type": self.type,
            "tasks": 11,
            "data": [],
            "materials": [],
            "string_data": [],
        }

        operating_point_api = [
            NameQuantityPair(
                "operating_point", k, Quantity(*self.operating_point[k].to_tuple())
            )
            for k in self.operating_point
        ]

        simulation_api = [
            NameQuantityPair("simulation", k, Quantity(*self.simulation[k].to_tuple()))
            for k in self.simulation
        ]
        job["string_data"] = [
            {"name": name, "value": value} for name, value in self._string_data.items()
        ]

        job["data"].extend(list(x.to_dict() for x in operating_point_api))
        job["data"].extend(list(x.to_dict() for x in simulation_api))
        job["data"].extend(self.machine.to_api())
        job["materials"] = [
            {"part": key, "material_id": value}
            for key, value in self.machine.materials.items()
        ]
        return job

    def from_api(self, job_dict):
        self.title = job_dict.get("title", self.generate_title())
        self.status = job_dict.get("status", 0)
        self.type = job_dict.get("type", "electromagnetic_spmbrl_fscwseg")
        self.id = job_dict.get("id", None)
        self._string_data = {
            item["name"]: item["value"] for item in job_dict.get("string_data", [])
        }
        self.operating_point = {
            item["name"]: decode(item["value"])
            for item in job_dict.get("data", [])
            if item["section"] == "operating_point"
        }
        self.simulation = {
            item["name"]: decode(item["value"])
            for item in job_dict.get("data", [])
            if item["section"] == "simulation"
        }
        stator_data = {
            item["name"]: decode(item["value"])
            for item in job_dict.get("data", [])
            if item["section"] in ["stator"]
        }
        winding_data = {
            item["name"]: decode(item["value"])
            for item in job_dict.get("data", [])
            if item["section"] in ["winding"]
        }
        rotor_data = {
            item["name"]: decode(item["value"])
            for item in job_dict.get("data", [])
            if item["section"] in ["rotor"]
        }
        material_data = {
            thing["part"]: thing["material_id"]
            for thing in job_dict.get("materials", [])
        }
        self.machine = Machine(
            stator=stator_data,
            rotor=rotor_data,
            winding=winding_data,
            materials=material_data,
        )


class TqdmUpTo(tqdm):
    """Provides `update_to(n)` which uses `tqdm.update(delta_n)`."""

    def update_to(self, b=1, bsize=1, tsize=None):
        """
        b  : int, optional
            Number of blocks transferred so far [default: 1].
        bsize  : int, optional
            Size of each block (in tqdm units) [default: 1].
        tsize  : int, optional
            Total size (in tqdm units). If [default: None] remains unchanged.
        """
        if tsize is not None:
            self.total = tsize
        return self.update(b * bsize - self.n)  # also sets self.n = b * bsize


class ProgressListener(StompListener):
    def __init__(self, job, uid):
        self.job_id = job.id
        self.uid = uid
        self._callback_fn = None  # Initialize the callback function

    @property
    def callback_fn(self):
        return self._callback_fn

    @callback_fn.setter
    def callback_fn(self, fn):
        self._callback_fn = fn

    def on_message(self, frame):
        headers = {key.decode(): value.decode() for key, value in frame.header}
        sub_hdr = headers.get("subscription")
        dest_hdr = headers.get("destination", "")
        # accept if subscription matches OR destination is for our job topic (some brokers don't preserve subscription)
        if sub_hdr != self.uid and not dest_hdr.startswith(f"/topic/{self.job_id}"):
            logger.debug(
                "Ignoring frame: subscription=%r uid=%r destination=%r payload=%r",
                sub_hdr,
                self.uid,
                dest_hdr,
                getattr(frame, "message", None),
            )
            return

        try:
            destination = headers.get("destination", "")
            parts = destination.split(".")
            worker_name = "unknown"
            if len(parts) > 1 and parts[0] == f"/topic/{self.job_id}":
                worker_name = parts[1]

            raw = (
                frame.message.decode()
                if isinstance(frame.message, (bytes, bytearray))
                else str(frame.message)
            )

            # Support two formats:
            # 1) "<time> - <LEVEL> - <json>"
            # 2) "<json>"
            if " - " in raw:
                # split into at most 3 parts to avoid accidental extra splits
                _, _level_str, mesg_str = raw.split(" - ", 2)
                payload = mesg_str.strip()
            else:
                payload = raw.strip()

            # Log the raw message
            logger.info(f"Received message from {worker_name}: {payload}")
            # Expect valid JSON payload — try to parse and fail if not JSON
            data = json.loads(payload)
            logger.info(f"Parsed message data: {data}")
        except (ValueError, IndexError, json.JSONDecodeError) as exc:
            logger.warning(
                "Unable to process progress message: %s (%s)",
                getattr(frame, "message", frame),
                exc,
            )
            return

        # forward to callback if present
        if not self._callback_fn:
            return

        # TODO specify progress messages in a scheme. some progress payloads use 'done' / 'total'
        if isinstance(data, dict):
            if "done" in data:
                logger.info(
                    f"Progress update: done={data['done']}, total={data.get('total')}"
                )
                self._callback_fn(
                    data["done"],
                    tsize=data.get("total"),
                    worker=worker_name,
                    message_type="progress",  # Add message type flag
                )
                return
            # Server-side status codes
            if "status" in data:
                try:
                    status_val = int(data["status"])
                    logger.info(
                        f"Status message received: {status_val}, Complete threshold: {JOB_STATUS['Complete']}"
                    )
                except Exception:
                    status_val = data["status"]
                    logger.info(f"Non-integer status received: {status_val}")
                self._callback_fn(
                    status_val, tsize=None, worker=worker_name, message_type="status"
                )  # Add message type flag
                return
            # remaining percent style
            if "remaining" in data and "unit" in data:
                try:
                    remaining = float(data.get("remaining") or 0.0)
                    unit = data.get("unit", "")

                    if unit == "seconds" or unit == "second":
                        # Just update the description to show time remaining, don't use for progress
                        logger.info(f"Time remaining update: {remaining} {unit}")
                        self._callback_fn(
                            None,
                            tsize=None,
                            worker=worker_name,
                            message_type="time_remaining",
                            remaining_time=f"{remaining:.1f} {unit}",
                        )
                    else:
                        # Handle as percentage if unit is not time-based
                        done = max(0, min(100, int(round(100.0 - remaining))))
                        logger.info(
                            f"Remaining percent update: remaining={remaining}, done={done}"
                        )
                        self._callback_fn(
                            done,
                            tsize=100,
                            worker=worker_name,
                            message_type="remaining",
                        )
                except Exception as e:
                    logger.debug(
                        "Could not interpret remaining value: %s (%s)",
                        data.get("remaining"),
                        e,
                    )
                    return


async def async_job_monitor(api, my_job, connection, position):
    uid = str(uuid4())
    listener = ProgressListener(my_job, uid)
    connection.add_listener(listener)
    connection.subscribe(destination=f"/topic/{my_job.id}.*.*.progress", id=uid)

    done_event = asyncio.Event()

    with TqdmUpTo(
        total=100,
        desc=f"Job {my_job.title}",
        position=position,
        leave=False,
    ) as pbar:
        current_worker = [None]

        def progress_callback(
            b, bsize=1, tsize=None, worker=None, message_type=None, remaining_time=None
        ):
            # Only check status messages for job completion
            if message_type == "status":
                try:
                    # Only treat actual status messages as completion criteria
                    logger.info(
                        f"Checking job status: received={b}, Complete threshold={JOB_STATUS['Complete']}"
                    )
                    if isinstance(b, int) and b >= JOB_STATUS["Complete"]:
                        logger.info(
                            f"Job complete condition met: status={b} >= {JOB_STATUS['Complete']}"
                        )
                        done_event.set()
                        return
                    else:
                        logger.info(
                            f"Job not yet complete: status={b} < {JOB_STATUS['Complete']}"
                        )
                except Exception as e:
                    logger.warning(f"Could not interpret job status {b}: {e}")
                    pass

            # Handle worker changes
            if worker and worker != current_worker[0]:
                current_worker[0] = worker
                pbar.desc = f"Job {my_job.title} [{worker}]"
                if (
                    message_type != "time_remaining"
                ):  # Don't reset progress for time updates
                    pbar.n = 0
                pbar.refresh()

            # Handle time remaining message type
            if message_type == "time_remaining" and remaining_time:
                pbar.desc = (
                    f"Job {my_job.title} [{worker}] - {remaining_time} remaining"
                )
                pbar.refresh()
                return

            # Skip progress bar updates for None values
            if b is None:
                return

            # Update progress bar for regular progress updates
            pbar.update_to(b, bsize, tsize)

            # For progress messages, check if we've reached total and log it
            if message_type == "progress" and tsize is not None and b >= tsize:
                logger.info(f"Progress reached total: b={b}, tsize={tsize}")

        listener.callback_fn = progress_callback

        # kick off initial status update
        logger.info(
            f"Starting job monitor for job {my_job.id} with initial status: {JOB_STATUS['QueuedForMeshing']}"
        )
        api.update_job_status(my_job.id, JOB_STATUS["QueuedForMeshing"])

        # wait for server push or fallback to polling with timeout
        try:
            logger.info(f"Waiting for done_event with 300s timeout")
            await asyncio.wait_for(done_event.wait(), timeout=300)
            logger.info(f"done_event was set, job monitor exiting wait")
        except asyncio.TimeoutError:
            # fallback polling if no progress pushed
            logger.info(
                f"Timeout waiting for job completion event, falling back to polling"
            )
            job_status = JOB_STATUS["QueuedForMeshing"]
            while job_status < JOB_STATUS["Complete"]:
                await asyncio.sleep(5)
                try:
                    job = api.get_job(my_job.id)
                    job_status = job["status"]
                    logger.info(
                        f"Polled job status: {job_status} (Complete threshold: {JOB_STATUS['Complete']})"
                    )
                    if job_status >= JOB_STATUS["Quarantined"]:
                        logger.info(f"Job quarantined with status: {job_status}")
                        break
                except Exception as e:
                    logger.warning(
                        f"Could not get job status for {my_job.id}, will keep trying: {e}"
                    )

    # final job status
    final_job_state = api.get_job(my_job.id)
    logger.info(
        f"Final job status: {final_job_state['status']} ({STATUS_JOB[final_job_state['status']]})"
    )
    # Force set done_event to ensure we don't hang
    if not done_event.is_set():
        logger.info("Forcing done_event to be set at end of job monitor")
        done_event.set()

    return STATUS_JOB[final_job_state["status"]]
