"""Advanced code generator with formatting options."""

from __future__ import annotations

from io import StringIO
from typing import TYPE_CHECKING, Any

from yaraast.ast.strings import (
    HexAlternative,
    HexByte,
    HexJump,
    HexString,
    HexToken,
    HexWildcard,
    PlainString,
    RegexString,
    StringDefinition,
)
from yaraast.codegen.formatting import (
    BraceStyle,
    FormattingConfig,
    HexStyle,
    IndentStyle,
    StringStyle,
)
from yaraast.codegen.generator import CodeGenerator

if TYPE_CHECKING:
    from yaraast.ast.base import ASTNode, YaraFile
    from yaraast.ast.expressions import BinaryExpression, Expression, SetExpression
    from yaraast.ast.meta import Meta
    from yaraast.ast.rules import Import, Include, Rule


class AdvancedCodeGenerator(CodeGenerator):
    """Advanced code generator with configurable formatting."""

    def __init__(self, config: FormattingConfig | None = None):
        self.config = config or FormattingConfig()
        super().__init__(self.config.indent_size)
        self._string_definitions: list[tuple[str, Any]] = []

    def generate(self, node: ASTNode) -> str:
        """Generate code with advanced formatting."""
        self.buffer = StringIO()
        self.indent_level = 0
        self._string_definitions = []
        self.visit(node)
        return self.buffer.getvalue()

    def _get_indent(self) -> str:
        """Get indentation string."""
        if self.config.indent_style == IndentStyle.TABS:
            return "\t" * self.indent_level
        return " " * (self.indent_level * self.config.indent_size)

    def _write(self, text: str) -> None:
        """Write text to buffer."""
        self.buffer.write(text)

    def _writeline(self, text: str = "") -> None:
        """Write line with proper indentation."""
        if text:
            self.buffer.write(self._get_indent())
            self.buffer.write(text)
        self.buffer.write("\n")

    def _write_blank_lines(self, count: int) -> None:
        """Write blank lines."""
        for _ in range(count):
            self.buffer.write("\n")

    def visit_yara_file(self, node: YaraFile) -> str:
        """Generate code for YaraFile with formatting."""
        # Sort imports if configured
        imports = node.imports
        if self.config.sort_imports:
            imports = sorted(imports, key=lambda x: x.module)

        # Write imports
        for imp in imports:
            self.visit(imp)

        if imports:
            self._write_blank_lines(self.config.blank_lines_between_sections)

        # Write includes
        for inc in node.includes:
            self.visit(inc)

        if node.includes:
            self._write_blank_lines(self.config.blank_lines_between_sections)

        # Sort rules if configured
        rules = node.rules
        if self.config.sort_rules:
            rules = sorted(rules, key=lambda x: x.name)

        # Write rules
        for i, rule in enumerate(rules):
            if i > 0:
                self._write_blank_lines(self.config.blank_lines_between_rules)
            self.visit(rule)

        return self.buffer.getvalue()

    def visit_rule(self, node: Rule) -> str:
        """Generate code for Rule with formatting."""
        # Write modifiers
        if node.modifiers:
            self._write(" ".join(node.modifiers) + " ")

        # Write rule name
        self._write(f"rule {node.name}")

        # Write tags
        if node.tags:
            if self.config.space_before_colon:
                self._write(" ")
            self._write(":")
            if self.config.space_after_colon:
                self._write(" ")
            self._write(" ".join(node.tags))

        # Write opening brace
        if self.config.brace_style == BraceStyle.SAME_LINE:
            self._write(" {")
            self._writeline()
        elif self.config.brace_style == BraceStyle.NEW_LINE:
            self._writeline()
            self._writeline("{")
        else:  # K&R
            self._writeline()
            self._writeline("{")

        self._indent()

        # Write sections in configured order
        sections_written = 0
        for section in self.config.section_order:
            if section == "meta" and node.meta:
                if sections_written > 0:
                    self._write_blank_lines(self.config.blank_lines_between_sections)
                self._write_meta_section(node.meta)
                sections_written += 1
            elif section == "strings" and node.strings:
                if sections_written > 0:
                    self._write_blank_lines(self.config.blank_lines_between_sections)
                self._write_strings_section(node.strings)
                sections_written += 1
            elif section == "condition":
                if sections_written > 0:
                    self._write_blank_lines(self.config.blank_lines_between_sections)
                self._write_condition_section(node.condition)
                sections_written += 1

        self._dedent()
        self._write("}")

        return self.buffer.getvalue()

    def _write_meta_section(self, meta_list: list[Meta]) -> None:
        """Write meta section with formatting."""
        self._writeline("meta:")
        self._indent()

        # Sort meta if configured
        if self.config.sort_meta:
            meta_list = sorted(meta_list, key=lambda x: x.key)

        # Find max key length for alignment
        max_key_len = max(len(m.key) for m in meta_list) if meta_list else 0

        for meta in meta_list:
            if self.config.string_style == StringStyle.TABULAR:
                self._write(self._get_indent())
                self._write(meta.key.ljust(max_key_len))
                self._write(" = ")
            else:
                self._write(self._get_indent())
                self._write(f"{meta.key} = ")

            # Write value
            if isinstance(meta.value, str):
                self._write(f'"{meta.value}"')
            elif isinstance(meta.value, bool):
                self._write("true" if meta.value else "false")
            else:
                self._write(str(meta.value))

            self._writeline()

        self._dedent()

    def _write_strings_section(self, strings: list[StringDefinition]) -> None:
        """Write strings section with formatting."""
        self._writeline("strings:")
        self._indent()

        # Sort strings if configured
        if self.config.sort_strings:
            strings = sorted(strings, key=lambda x: x.identifier)

        # Collect string definitions for alignment
        if self.config.string_style in (StringStyle.ALIGNED, StringStyle.TABULAR):
            self._collect_string_definitions(strings)
            self._write_aligned_strings()
        else:
            # Compact style
            for string_def in strings:
                self.visit(string_def)
                self._writeline()

        self._dedent()

    def _collect_string_definitions(self, strings: list[StringDefinition]) -> None:
        """Collect string definitions for alignment."""
        self._string_definitions = []

        for string_def in strings:
            identifier = string_def.identifier

            if isinstance(string_def, PlainString):
                value = f'"{string_def.value}"'
            elif isinstance(string_def, HexString):
                value = self._format_hex_string(string_def)
            elif isinstance(string_def, RegexString):
                value = f"/{string_def.regex}/"
            else:
                value = ""

            modifiers = []
            for mod in string_def.modifiers:
                if mod.value is not None:
                    modifiers.append(f"{mod.name}({mod.value})")
                else:
                    modifiers.append(mod.name)

            self._string_definitions.append((identifier, value, modifiers))

    def _write_aligned_strings(self) -> None:
        """Write aligned string definitions."""
        if not self._string_definitions:
            return

        # Calculate alignment widths
        max_id_len = max(len(id) for id, _, _ in self._string_definitions)
        max_val_len = max(len(val) for _, val, _ in self._string_definitions)

        for identifier, value, modifiers in self._string_definitions:
            self._write(self._get_indent())

            if self.config.string_style == StringStyle.TABULAR:
                # Tabular alignment
                self._write(identifier.ljust(max_id_len))
                self._write(" = ")
                self._write(value.ljust(max_val_len))
            else:
                # Simple alignment
                self._write(f"{identifier} = {value}")

            # Write modifiers
            if modifiers:
                if self.config.align_string_modifiers:
                    self._write("  ")
                else:
                    self._write(" ")
                self._write(" ".join(modifiers))

            self._writeline()

    def _format_hex_string(self, node: HexString) -> str:
        """Format hex string according to style."""
        parts = []

        for token in node.tokens:
            if isinstance(token, HexByte):
                hex_val = f"{token.value:02x}"
                if self.config.hex_style == HexStyle.UPPERCASE:
                    hex_val = hex_val.upper()
                parts.append(hex_val)
            elif isinstance(token, HexWildcard):
                parts.append("??")
            elif isinstance(token, HexJump):
                if token.min_jump is None and token.max_jump is None:
                    parts.append("[-]")
                elif token.min_jump is None:
                    parts.append(f"[-{token.max_jump}]")
                elif token.max_jump is None:
                    parts.append(f"[{token.min_jump}-]")
                elif token.min_jump == token.max_jump:
                    parts.append(f"[{token.min_jump}]")
                else:
                    parts.append(f"[{token.min_jump}-{token.max_jump}]")
            elif isinstance(token, HexAlternative):
                alt_parts = []
                for alt in token.alternatives:
                    alt_str = " ".join(self._format_hex_token(t) for t in alt)
                    alt_parts.append(alt_str)
                parts.append(f"({' | '.join(alt_parts)})")
            elif hasattr(token, "high") and hasattr(token, "value"):  # HexNibble
                nibble_str = f"{token.value:X}"
                if token.high:
                    parts.append(f"{nibble_str}?")
                else:
                    parts.append(f"?{nibble_str}")

        # Apply grouping if configured
        if self.config.hex_group_size > 0:
            grouped_parts = []
            for i in range(0, len(parts), self.config.hex_group_size):
                group = parts[i : i + self.config.hex_group_size]
                grouped_parts.append("".join(group))
            hex_content = " ".join(grouped_parts)
        else:
            hex_content = " ".join(parts)

        return f"{{ {hex_content} }}"

    def _format_hex_token(self, token: HexToken) -> str:
        """Format individual hex token."""
        if isinstance(token, HexByte):
            hex_val = f"{token.value:02x}"
            if self.config.hex_style == HexStyle.UPPERCASE:
                hex_val = hex_val.upper()
            return hex_val
        if isinstance(token, HexWildcard):
            return "??"
        return ""

    def _write_condition_section(self, condition: Expression) -> None:
        """Write condition section."""
        self._writeline("condition:")
        self._indent()

        # Generate condition code
        condition_str = self._generate_condition_string(condition)

        # Handle line wrapping if needed
        if len(condition_str) > self.config.max_line_length:
            self._write_wrapped_condition(condition_str)
        else:
            self._writeline(condition_str)

        self._dedent()

    def _generate_condition_string(self, expr: Expression) -> str:
        """Generate condition string."""
        # Create a temporary generator to get the condition string
        temp_gen = CodeGenerator()
        return temp_gen.visit(expr)

    def _write_wrapped_condition(self, condition: str) -> None:
        """Write wrapped condition for long lines."""
        # Simple wrapping at operators
        # This is a simplified implementation
        self._writeline(condition)

    # Operator formatting
    def visit_binary_expression(self, node: BinaryExpression) -> str:
        """Generate binary expression with spacing."""
        left = self.visit(node.left)
        right = self.visit(node.right)

        if self.config.space_around_operators:
            self._write(f"({left} {node.operator} {right})")
        else:
            self._write(f"({left}{node.operator}{right})")

        return self.buffer.getvalue()

    def visit_set_expression(self, node: SetExpression) -> str:
        """Generate set expression with spacing."""
        elements = []
        for elem in node.elements:
            elem_str = self.visit(elem)
            elements.append(elem_str)

        if self.config.space_after_comma:
            self._write(f"({', '.join(elements)})")
        else:
            self._write(f"({','.join(elements)})")

        return self.buffer.getvalue()

    # Default visit methods (delegate to parent)
    def visit_import(self, node: Import) -> str:
        self._writeline(f'import "{node.module}"')
        return ""

    def visit_include(self, node: Include) -> str:
        self._writeline(f'include "{node.path}"')
        return ""

    def visit_plain_string(self, node: PlainString) -> str:
        if self.config.string_style == StringStyle.COMPACT:
            self._write(f'{node.identifier}="{node.value}"')
        else:
            self._write(f'{node.identifier} = "{node.value}"')

        for modifier in node.modifiers:
            self._write(" ")
            self.visit(modifier)

        return ""

    def visit_hex_string(self, node: HexString) -> str:
        hex_str = self._format_hex_string(node)

        if self.config.string_style == StringStyle.COMPACT:
            self._write(f"{node.identifier}={hex_str}")
        else:
            self._write(f"{node.identifier} = {hex_str}")

        for modifier in node.modifiers:
            self._write(" ")
            self.visit(modifier)

        return ""

    def visit_regex_string(self, node: RegexString) -> str:
        if self.config.string_style == StringStyle.COMPACT:
            self._write(f"{node.identifier}=/{node.regex}/")
        else:
            self._write(f"{node.identifier} = /{node.regex}/")

        for modifier in node.modifiers:
            self._write(" ")
            self.visit(modifier)

        return ""
