import os
import sys
import json
import importlib
from lager.supply.supply_net import LibraryMissingError, DeviceNotFoundError, SupplyBackendError

_RED = "\033[31m"
_RESET = "\033[0m"


def die(message: str, code: int = 0) -> None:
    """Print an error message and exit with the given code."""
    if code == 0:
        print(message)
    else:
        print(f"{_RED}{message}{_RESET}", file=sys.stderr)
    sys.exit(code)


def main() -> None:
    cmd_data = os.environ.get("LAGER_COMMAND_DATA")
    if not cmd_data:
        die("ERROR [usage] No command data provided to supply backend", code=64)

    try:
        command = json.loads(cmd_data)
    except Exception as exc:
        die(f"ERROR [usage] Could not parse command data: {exc}", code=64)

    action = command.get("action")
    params = command.get("params", {}) or {}
    net_name = params.pop("netname", None)

    if not action or not net_name:
        die("ERROR [usage] Missing action or net name", code=64)

    try:
        dispatcher = importlib.import_module("lager.supply.dispatcher")

        func = getattr(dispatcher, action, None)
        if func is None:
            die(f"ERROR [unexpected] Unknown supply command: {action}", code=1)

        func(net_name, **params)

    except LibraryMissingError as exc:
        die(f"ERROR [library-missing] {exc}", code=2)
    except DeviceNotFoundError as exc:
        die(f"ERROR [device-not-found] {exc}", code=3)
    except SupplyBackendError as exc:
        die(f"ERROR [backend] {exc}", code=5)
    except SystemExit:
        raise  # let deliberate exits bubble
    except Exception as exc:
        die(f"ERROR [unexpected] {exc}", code=1)


if __name__ == "__main__":
    main()