# -*- coding: utf-8 -*-
# chuk_artifacts/base.py
"""
Base class for operation modules.
Updated to work with chuk_sessions integration.
"""

from __future__ import annotations
import logging
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from .store import ArtifactStore

from .exceptions import (
    ArtifactNotFoundError,
    ArtifactCorruptedError,
    SessionError,
    ArtifactStoreError,
)
from .models import ArtifactMetadata

logger = logging.getLogger(__name__)


class BaseOperations:
    """Base class for all operation modules."""

    def __init__(self, store: "ArtifactStore"):
        # Store reference to artifact store
        self._artifact_store = store

    @property
    def bucket(self) -> str:
        return self._artifact_store.bucket

    @property
    def s3_factory(self):
        return self._artifact_store._s3_factory

    @property
    def session_factory(self):
        return self._artifact_store._session_factory

    @property
    def storage_provider_name(self) -> str:
        return self._artifact_store._storage_provider_name

    @property
    def session_provider_name(self) -> str:
        return self._artifact_store._session_provider_name

    @property
    def max_retries(self) -> int:
        return self._artifact_store.max_retries

    @property
    def session_manager(self):
        """Access to chuk_sessions SessionManager."""
        return self._artifact_store._session_manager

    def _check_closed(self):
        """Check if store is closed and raise error if so."""
        if self._artifact_store._closed:
            raise ArtifactStoreError("Store has been closed")

    async def _get_record(self, artifact_id: str) -> ArtifactMetadata:
        """
        Retrieve artifact metadata from session provider.

        This is a shared helper used by multiple operation modules.
        """
        try:
            session_ctx_mgr = self.session_factory()
            async with session_ctx_mgr as session:
                raw = await session.get(artifact_id)
        except Exception as e:
            raise SessionError(
                f"Session provider error retrieving {artifact_id}: {e}"
            ) from e

        if raw is None:
            raise ArtifactNotFoundError(f"Artifact {artifact_id} not found or expired")

        try:
            return ArtifactMetadata.model_validate_json(raw)
        except Exception as e:
            logger.error(f"Corrupted metadata for artifact {artifact_id}: {e}")
            raise ArtifactCorruptedError(
                f"Corrupted metadata for artifact {artifact_id}"
            ) from e
