from devvio_util.primitives.devv_constants import (
    kWALLET_SIG_SIZE,
    kNODE_SIG_SIZE,
    kWALLET_SIG_BUF_SIZE,
    kNODE_SIG_BUF_SIZE,
)
from devvio_util.primitives.utils import InputBuffer, is_hex


class Signature:
    """
    Signatures are wrappers for EC signatures that lie on either SECP256K1 or SECP384R1 curves
    for wallets or nodes, respectively.
    They contain a 1-byte prefix indicating address length, which is used to differentiate these two key types.
    """

    def __init__(self, sig: str or bytes or InputBuffer = None):
        """
        Initialize Signature from either a hex string, b
        """
        self._canonical = None
        self._size = None
        self.set_sig(sig)

    def set_sig(self, sig: str or bytes or InputBuffer):
        if not sig:
            raise ValueError("Invalid Signature: no bytes or string given")
        if isinstance(sig, str):
            if not is_hex(sig):
                raise ValueError("Invalid Signature: not a valid hex string")
            sig_bin = bytes.fromhex(sig)
        elif isinstance(sig, bytes):
            sig_bin = sig
        elif isinstance(sig, InputBuffer):
            sig_bin = sig.get_next_prefixed_obj()
        else:
            raise ValueError(
                f"Invalid Signature: cannot initialize from type {type(sig)}"
            )
        if not sig_bin:
            return None
        self._size = len(sig_bin)
        if self._size == kWALLET_SIG_SIZE or self._size == kNODE_SIG_SIZE:
            self._canonical = bytes([self._size]) + sig_bin
        elif self._size != (prefix_size := sig_bin[0] + 1):
            raise RuntimeError(
                f"Invalid Signature: prefix != num bytes given ({prefix_size} != {self._size})"
            )
        elif self._size == kWALLET_SIG_BUF_SIZE or self._size == kNODE_SIG_BUF_SIZE:
            self._canonical = sig_bin
            self._size -= 1
        else:
            raise ValueError(f"Invalid Signature: invalid size {self._size}")

    def __eq__(self, other) -> bool:
        """
        Compare signatures. True if signature binaries are equivalent.
        """
        return self._canonical == other.get_canonical()

    def __bool__(self) -> bool:
        """
        Evaluate as boolean, serves as isNull()
        """
        return self._canonical is not None

    def __str__(self) -> str:
        """
        Get formatted signature, including the 1-byte size prefix.
        :return: signature as hex string
        :rtype: str
        """
        return self.get_hex_str()

    def __len__(self) -> int:
        """
        Get signature length, excluding the 1-byte size prefix.
        :return: size of signature
        :rtype: int
        """
        return self.get_size()

    def get_size(self) -> int:
        """
        Get signature length, excluding the 1-byte size prefix.
        :return: size of signature
        :rtype: int
        """
        return self._size

    def get_canonical(self, legacy: bool = False) -> bytes:
        """
        Get signature binary. If legacy = True, this will exclude the 1-byte size prefix.
        :param legacy:
        :type legacy: bool
        :return: signature binary
        :rtype: bytes
        """
        if legacy:
            return self._canonical[1:]
        return self._canonical

    def get_hex_str(self) -> str:
        """
        Get formatted signature. Includes the 1-byte size prefix.
        :return: signature as hex string
        :rtype: str
        """
        if not self._canonical:
            raise Exception("Signature is not initialized!")
        return self._canonical.hex().upper()

    def is_wallet_sig(self) -> bool:
        """
        Check if signature was generated by a wallet key (on the SECP256K1 curve)
        :return: True if signature is on the SECP256K1 curve
        :rtype: bool
        """
        if not self:
            return False
        return self._size == kWALLET_SIG_SIZE

    def is_node_sig(self) -> bool:
        """
        Check if signature was generated by a node key (on the SECP384R1 curve)
        :return: True if signature is on the SECP384R1 curve
        :rtype: bool
        """
        if not self:
            return False
        return self._size == kNODE_SIG_SIZE

    @staticmethod
    def is_valid_sig_size(sig_size: int) -> bool:
        """
        Return true if the given length is a valid wallet or node signature length.
        :return: Length of corresponding signature
        :rtype: int
        """
        return sig_size in [
            kWALLET_SIG_SIZE,
            kWALLET_SIG_BUF_SIZE,
            kNODE_SIG_SIZE,
            kNODE_SIG_BUF_SIZE,
        ]
