from pydantic import BaseModel, Field
from typing import Annotated, Generic, Literal, TypeGuard
from uuid import UUID
from nexo.schemas.mixins.identity import Identifier
from nexo.types.uuid import OptUUID
from ..enums.checkup import (
    IdentifierType,
    OptCheckupStatusT,
    OptListOfCheckupStatuses,
)
from ..types.checkup import IdentifierValueType


class CheckupStatus(BaseModel, Generic[OptCheckupStatusT]):
    checkup_status: Annotated[
        OptCheckupStatusT, Field(..., description="Checkup's status")
    ]


class CheckupStatuses(BaseModel):
    checkup_statuses: Annotated[
        OptListOfCheckupStatuses, Field(None, description="Checkup's statuses")
    ] = None


class RadiologyRecordId(BaseModel):
    radiology_record_id: Annotated[
        OptUUID, Field(None, description="Radiology record's Id")
    ] = None


class CheckupIdentifier(Identifier[IdentifierType, IdentifierValueType]):
    @property
    def column_and_value(self) -> tuple[str, IdentifierValueType]:
        return self.type.column, self.value


class IdCheckupIdentifier(Identifier[Literal[IdentifierType.ID], int]):
    type: Annotated[
        Literal[IdentifierType.ID],
        Field(IdentifierType.ID, description="Identifier's type"),
    ] = IdentifierType.ID
    value: Annotated[int, Field(..., description="Identifier's value", ge=1)]


class UUIDCheckupIdentifier(Identifier[Literal[IdentifierType.UUID], UUID]):
    type: Annotated[
        Literal[IdentifierType.UUID],
        Field(IdentifierType.UUID, description="Identifier's type"),
    ] = IdentifierType.UUID


AnyCheckupIdentifier = CheckupIdentifier | IdCheckupIdentifier | UUIDCheckupIdentifier


def is_id_identifier(
    identifier: AnyCheckupIdentifier,
) -> TypeGuard[IdCheckupIdentifier]:
    return identifier.type is IdentifierType.ID and isinstance(identifier.value, int)


def is_uuid_identifier(
    identifier: AnyCheckupIdentifier,
) -> TypeGuard[UUIDCheckupIdentifier]:
    return identifier.type is IdentifierType.UUID and isinstance(identifier.value, UUID)
