import unittest
from pathlib import Path
from typing import ClassVar

import numpy as np
from numpy.testing import assert_array_equal

from neuralib.stimpy import StimpyProtocol, RiglogData
from neuralib.stimpy.stimpy_core import Stimlog


class TestProtocolParser(unittest.TestCase):
    pass


class TestProtocolParserReal(unittest.TestCase):
    PROT: ClassVar[str] = """
        # general parameters
        controller = VisualExpController
        stimulusType = gratings
        nTrials = 5
        shuffle = True
        startBlankDuration = 900
        blankDuration = 2
        endBlankDuration = 900
        texture = sqr
        mask = None
        
        # stimulus conditions
        n     dur  xc   yc   c    sf      ori    flick  width height evolveParams
        1     3    0    0    1    0.04    0      0      200    200    {'phase':['linear',1]}
        2     3    0    0    1    0.04    30     0      200    200    {'phase':['linear',1]}
        3     3    0    0    1    0.04    60     0      200    200    {'phase':['linear',1]}
        4     3    0    0    1    0.04    90     0      200    200    {'phase':['linear',1]}
        5     3    0    0    1    0.04    120    0      200    200    {'phase':['linear',1]}
        6     3    0    0    1    0.04    150    0      200    200    {'phase':['linear',1]}
        7     3    0    0    1    0.04    180    0      200    200    {'phase':['linear',1]}
        8     3    0    0    1    0.04    210    0      200    200    {'phase':['linear',1]}
        9     3    0    0    1    0.04    240    0      200    200    {'phase':['linear',1]}
        10    3    0    0    1    0.04    270    0      200    200    {'phase':['linear',1]}
        11    3    0    0    1    0.04    300    0      200    200    {'phase':['linear',1]}
        12    3    0    0    1    0.04    330    0      200    200    {'phase':['linear',1]}
        
        13    3    0    0    1    0.08    0      0      200    200    {'phase':['linear',1]}
        14    3    0    0    1    0.08    30     0      200    200    {'phase':['linear',1]}
        15    3    0    0    1    0.08    60     0      200    200    {'phase':['linear',1]}
        16    3    0    0    1    0.08    90     0      200    200    {'phase':['linear',1]}
        17    3    0    0    1    0.08    120    0      200    200    {'phase':['linear',1]}
        18    3    0    0    1    0.08    150    0      200    200    {'phase':['linear',1]}
        19    3    0    0    1    0.08    180    0      200    200    {'phase':['linear',1]}
        20    3    0    0    1    0.08    210    0      200    200    {'phase':['linear',1]}
        21    3    0    0    1    0.08    240    0      200    200    {'phase':['linear',1]}
        22    3    0    0    1    0.08    270    0      200    200    {'phase':['linear',1]}
        23    3    0    0    1    0.08    300    0      200    200    {'phase':['linear',1]}
        24    3    0    0    1    0.08    330    0      200    200    {'phase':['linear',1]}
        
        25    3    0    0    1    0.16    0      0      200    200    {'phase':['linear',1]}
        26    3    0    0    1    0.16    30     0      200    200    {'phase':['linear',1]}
        27    3    0    0    1    0.16    60     0      200    200    {'phase':['linear',1]}
        28    3    0    0    1    0.16    90     0      200    200    {'phase':['linear',1]}
        29    3    0    0    1    0.16    120    0      200    200    {'phase':['linear',1]}
        30    3    0    0    1    0.16    150    0      200    200    {'phase':['linear',1]}
        31    3    0    0    1    0.16    180    0      200    200    {'phase':['linear',1]}
        32    3    0    0    1    0.16    210    0      200    200    {'phase':['linear',1]}
        33    3    0    0    1    0.16    240    0      200    200    {'phase':['linear',1]}
        34    3    0    0    1    0.16    270    0      200    200    {'phase':['linear',1]}
        35    3    0    0    1    0.16    300    0      200    200    {'phase':['linear',1]}
        36    3    0    0    1    0.16    330    0      200    200    {'phase':['linear',1]}
        
        
        
        37    3    0    0    1    0.04    0      0      200    200    {'phase':['linear',4]}
        38    3    0    0    1    0.04    30     0      200    200    {'phase':['linear',4]}
        39    3    0    0    1    0.04    60     0      200    200    {'phase':['linear',4]}
        40    3    0    0    1    0.04    90     0      200    200    {'phase':['linear',4]}
        41    3    0    0    1    0.04    120    0      200    200    {'phase':['linear',4]}
        42    3    0    0    1    0.04    150    0      200    200    {'phase':['linear',4]}
        43    3    0    0    1    0.04    180    0      200    200    {'phase':['linear',4]}
        44    3    0    0    1    0.04    210    0      200    200    {'phase':['linear',4]}
        45    3    0    0    1    0.04    240    0      200    200    {'phase':['linear',4]}
        46    3    0    0    1    0.04    270    0      200    200    {'phase':['linear',4]}
        47    3    0    0    1    0.04    300    0      200    200    {'phase':['linear',4]}
        48    3    0    0    1    0.04    330    0      200    200    {'phase':['linear',4]}
        
        49    3    0    0    1    0.08    0      0      200    200    {'phase':['linear',4]}
        50    3    0    0    1    0.08    30     0      200    200    {'phase':['linear',4]}
        51    3    0    0    1    0.08    60     0      200    200    {'phase':['linear',4]}
        52    3    0    0    1    0.08    90     0      200    200    {'phase':['linear',4]}
        53    3    0    0    1    0.08    120    0      200    200    {'phase':['linear',4]}
        54    3    0    0    1    0.08    150    0      200    200    {'phase':['linear',4]}
        55    3    0    0    1    0.08    180    0      200    200    {'phase':['linear',4]}
        56    3    0    0    1    0.08    210    0      200    200    {'phase':['linear',4]}
        57    3    0    0    1    0.08    240    0      200    200    {'phase':['linear',4]}
        58    3    0    0    1    0.08    270    0      200    200    {'phase':['linear',4]}
        59    3    0    0    1    0.08    300    0      200    200    {'phase':['linear',4]}
        60    3    0    0    1    0.08    330    0      200    200    {'phase':['linear',4]}
        
        61    3    0    0    1    0.16    0      0      200    200    {'phase':['linear',4]}
        62    3    0    0    1    0.16    30     0      200    200    {'phase':['linear',4]}
        63    3    0    0    1    0.16    60     0      200    200    {'phase':['linear',4]}
        64    3    0    0    1    0.16    90     0      200    200    {'phase':['linear',4]}
        65    3    0    0    1    0.16    120    0      200    200    {'phase':['linear',4]}
        66    3    0    0    1    0.16    150    0      200    200    {'phase':['linear',4]}
        67    3    0    0    1    0.16    180    0      200    200    {'phase':['linear',4]}
        68    3    0    0    1    0.16    210    0      200    200    {'phase':['linear',4]}
        69    3    0    0    1    0.16    240    0      200    200    {'phase':['linear',4]}
        70    3    0    0    1    0.16    270    0      200    200    {'phase':['linear',4]}
        71    3    0    0    1    0.16    300    0      200    200    {'phase':['linear',4]}
        72    3    0    0    1    0.16    330    0      200    200    {'phase':['linear',4]}
        """

    PROTOCOL: StimpyProtocol

    @classmethod
    def setUpClass(cls):
        cls.PROTOCOL = StimpyProtocol.loads(cls.PROT)

    def test_controller_name(self):
        self.assertEqual(self.PROTOCOL.controller, 'VisualExpController')

    def test_stimulus_type(self):
        self.assertEqual(self.PROTOCOL.stimulus_type, 'gratings')

    def test_n_trials(self):
        self.assertEqual(self.PROTOCOL.n_trials, 5)

    def test_n_stimuli(self):
        self.assertEqual(self.PROTOCOL.n_stimuli, 72)

    def test_is_shuffle(self):
        self.assertEqual(self.PROTOCOL.is_shuffle, True)

    def test_start_blank_duration(self):
        self.assertEqual(self.PROTOCOL.start_blank_duration, 900)

    def test_blank_duration(self):
        self.assertEqual(self.PROTOCOL.blank_duration, 2)

    def test_end_blank_duration(self):
        self.assertEqual(self.PROTOCOL.end_blank_duration, 900)

    def test_texture(self):
        self.assertEqual(self.PROTOCOL.texture, 'sqr')

    def test_mask(self):
        self.assertEqual(self.PROTOCOL.mask, 'None')

    def test_stimulus_condition_n(self):
        assert_array_equal(self.PROTOCOL['n'], np.arange(1, self.PROTOCOL.n_stimuli + 1))

    def test_stimulus_condition_dur(self):
        self.assertTrue(np.all(self.PROTOCOL['dur'] == 3))

    def test_stimulus_condition_xc(self):
        self.assertTrue(np.all(self.PROTOCOL['xc'] == 0))

    def test_stimulus_condition_yc(self):
        self.assertTrue(np.all(self.PROTOCOL['yc'] == 0))

    def test_stimulus_condition_c(self):
        self.assertTrue(np.all(self.PROTOCOL['c'] == 1))

    def test_stimulus_condition_ori(self):
        ori = np.tile(np.linspace(0, 330, 12), 6)  # sf * tf
        assert_array_equal(self.PROTOCOL['ori'], ori)

    def test_stimulus_condition_flick(self):
        self.assertTrue(np.all(self.PROTOCOL['flick'] == 0))

    def test_stimulus_condition_width(self):
        self.assertTrue(np.all(self.PROTOCOL['width'] == 200))

    def test_stimulus_condition_height(self):
        self.assertTrue(np.all(self.PROTOCOL['height'] == 200))

    def test_stimulus_condition_sf(self):
        sf_val = [0.04, 0.08, 0.16]
        ret = ([sf_val[0]] * 12 + [sf_val[1]] * 12 + [sf_val[2]] * 12) * 2
        assert_array_equal(self.PROTOCOL['sf'], ret)

    def test_stimulus_condition_tf(self):
        tf_val = [1, 4]
        n_tf_set = len(tf_val)
        block = int(self.PROTOCOL.n_stimuli / n_tf_set)

        ret = []
        for tf in tf_val:
            ret.extend([tf] * block)

        assert_array_equal(self.PROTOCOL.tf, ret)


class TestRiglogParser(unittest.TestCase):
    RIGLOG_CONTENT = """# RIG VERSION: 0.3
# RIG GIT COMMIT HASH: 6d30281
# CODES: screen=0,imaging=1,position=2,lick=3,reward=4,lap=5,cam1=6,cam2=7,cam3=8,act0=21,act1=22
# RIG CSV: code,time received,duino time,value
# STARTED EXPERIMENT
[2, 946422, 946403.0, 3735]
[2, 946422, 946410.0, 3739]
[8, 946422, 946414.0, 21337]
[2, 946438, 946417.0, 3743]
[6, 946438, 946424.0, 28395]
[2, 946438, 946424.0, 3747]
[2, 946453, 946431.0, 3751]
[7, 946453, 946431.0, 28394]
[1, 946453, 946434.0, 28361]
[2, 946453, 946438.0, 3755]
[2, 946469, 946446.0, 3759]
[2, 946469, 946453.0, 3763]
[6, 946469, 946457.0, 28396]
[8, 946469, 946459.0, 21338]
[2, 946484, 946460.0, 3767]
[7, 946484, 946465.0, 28395]
[2, 946484, 946467.0, 3771]
[1, 946484, 946467.0, 28362]
[2, 946484, 946474.0, 3775]
[2, 946500, 946481.0, 3779]
[2, 946500, 946488.0, 3783]
[6, 946500, 946490.0, 28397]
[2, 946516, 946496.0, 3787]
[7, 946516, 946498.0, 28396]
[1, 946516, 946500.0, 28363]
[8, 946516, 946503.0, 21339]
[2, 946516, 946503.0, 3791]
[2, 946531, 946511.0, 3795]
[2, 946531, 946518.0, 3799]
[6, 946547, 946524.0, 28398]
[2, 946547, 946526.0, 3803]
[7, 946547, 946531.0, 28397]
[2, 946547, 946534.0, 3807]
[1, 946547, 946534.0, 28364]
[2, 946562, 946541.0, 3811]
[8, 946562, 946547.0, 21340]
[2, 946562, 946549.0, 3815]
[2, 946578, 946556.0, 3819]
[6, 946578, 946557.0, 28399]
[2, 946578, 946564.0, 3823]
[7, 946578, 946565.0, 28398]
[1, 946578, 946567.0, 28365]
[2, 946594, 946571.0, 3827]
[2, 946594, 946579.0, 3831]
[2, 946594, 946586.0, 3835]
[6, 946609, 946590.0, 28400]
[8, 946609, 946592.0, 21341]
[2, 946609, 946593.0, 3839]
[7, 946609, 946598.0, 28399]
[1, 946625, 946600.0, 28366]
[2, 946625, 946601.0, 3843]
[2, 946625, 946609.0, 3847]
[2, 946640, 946616.0, 3851]
[2, 946640, 946624.0, 3855]
[6, 946640, 946624.0, 28401]
[2, 946640, 946631.0, 3859]
[7, 946656, 946631.0, 28400]
[1, 946656, 946634.0, 28367]
[8, 946656, 946636.0, 21342]
[2, 946656, 946638.0, 3863]
[2, 946656, 946644.0, 3867]
[2, 946672, 946651.0, 3871]
[6, 946672, 946657.0, 28402]
[2, 946672, 946657.0, 3875]
[2, 946672, 946663.0, 3879]
[7, 946687, 946665.0, 28401]
[1, 946687, 946667.0, 28368]
[2, 946687, 946670.0, 3883]
[2, 946687, 946676.0, 3887]
[8, 946703, 946680.0, 21343]
[2, 946703, 946682.0, 3891]
[2, 946703, 946688.0, 3895]
[6, 946703, 946690.0, 28403]
[2, 946718, 946694.0, 3899]
[7, 946718, 946698.0, 28402]
[2, 946718, 946700.0, 3903]
[1, 946718, 946701.0, 28369]
[2, 946718, 946706.0, 3907]
[2, 946734, 946713.0, 3911]
[2, 946734, 946719.0, 3915]
[6, 946734, 946724.0, 28404]
[8, 946734, 946725.0, 21344]
[2, 946750, 946726.0, 3919]
[7, 946750, 946731.0, 28403]
[2, 946750, 946732.0, 3923]
[1, 946750, 946734.0, 28370]
[2, 946750, 946739.0, 3927]
[2, 946765, 946747.0, 3931]
[2, 946765, 946754.0, 3935]
[6, 946781, 946757.0, 28405]
[2, 946781, 946762.0, 3939]
[7, 946781, 946765.0, 28404]
[1, 946781, 946767.0, 28371]
[8, 946781, 946769.0, 21345]
[2, 946781, 946770.0, 3943]
[2, 946796, 946778.0, 3947]
[2, 946796, 946786.0, 3951]
[6, 946812, 946790.0, 28406]
[2, 946812, 946794.0, 3955]
[7, 946812, 946798.0, 28405]
[1, 946812, 946801.0, 28372]
[2, 946812, 946801.0, 3959]
[2, 946828, 946808.0, 3963]
[8, 946828, 946813.0, 21346]
[2, 946828, 946815.0, 3967]
[2, 946843, 946822.0, 3971]
[6, 946843, 946824.0, 28407]
[2, 946843, 946830.0, 3975]
[7, 946843, 946831.0, 28406]
[1, 946843, 946834.0, 28373]
[2, 946859, 946837.0, 3979]
[2, 946859, 946845.0, 3983]
[2, 946874, 946853.0, 3987]
[6, 946874, 946857.0, 28408]
[8, 946874, 946858.0, 21347]
[2, 946874, 946862.0, 3991]
[7, 946874, 946865.0, 28407]
[1, 946890, 946867.0, 28374]
[2, 946890, 946870.0, 3995]
[2, 946890, 946880.0, 3999]
[2, 946906, 946889.0, 4003]
[6, 946906, 946890.0, 28409]
[7, 946906, 946898.0, 28408]
[2, 946921, 946899.0, 4007]
[1, 946921, 946901.0, 28375]
[8, 946921, 946902.0, 21348]
[2, 946921, 946908.0, 4011]
[2, 946937, 946918.0, 4015]
[6, 946937, 946924.0, 28410]
[2, 946952, 946928.0, 4019]
[7, 946952, 946931.0, 28409]
[1, 946952, 946934.0, 28376]
[2, 946952, 946937.0, 4023]
[8, 946968, 946946.0, 21349]
[2, 946968, 946947.0, 4027]
[2, 946968, 946956.0, 4031]
[6, 946968, 946957.0, 28411]
[2, 946984, 946964.0, 4035]
[7, 946984, 946965.0, 28410]
[1, 946984, 946967.0, 28377]
[2, 946984, 946973.0, 4039]
[2, 946999, 946981.0, 4043]
[2, 946999, 946990.0, 4047]
[6, 947015, 946990.0, 28412]
[8, 947015, 946991.0, 21350]
[7, 947015, 946998.0, 28411]
[2, 947015, 946999.0, 4051]
[1, 947015, 947001.0, 28378]
[2, 947030, 947007.0, 4055]
[2, 947030, 947016.0, 4059]
[6, 947046, 947024.0, 28413]
[2, 947046, 947024.0, 4063]
[7, 947046, 947031.0, 28412]
[2, 947046, 947033.0, 4067]
[1, 947046, 947034.0, 28379]
[8, 947046, 947035.0, 21351]
[2, 947062, 947041.0, 4071]
[2, 947062, 947049.0, 4075]
[6, 947077, 947057.0, 28414]
[2, 947077, 947057.0, 4079]
[7, 947077, 947064.0, 28413]
[2, 947077, 947065.0, 4083]
[1, 947077, 947067.0, 28380]
[2, 947093, 947073.0, 4087]
[8, 947093, 947080.0, 21352]
[2, 947093, 947081.0, 4091]
[2, 947108, 947090.0, 4095]
[6, 947108, 947090.0, 28415]
[7, 947108, 947098.0, 28414]
[2, 947124, 947098.0, 4099]
[1, 947124, 947101.0, 28381]
[2, 947124, 947107.0, 4103]
[2, 947140, 947116.0, 4107]
[6, 947140, 947124.0, 28416]
[8, 947140, 947124.0, 21353]
[2, 947140, 947125.0, 4111]
[7, 947140, 947131.0, 28415]
[1, 947155, 947134.0, 28382]
[2, 947155, 947134.0, 4115]
[2, 947155, 947144.0, 4119]
[2, 947171, 947152.0, 4123]
[6, 947171, 947157.0, 28417]
[2, 947186, 947161.0, 4127]
[7, 947186, 947165.0, 28416]
[1, 947186, 947167.0, 28383]
[8, 947186, 947168.0, 21354]
[2, 947186, 947170.0, 4131]
[2, 947202, 947178.0, 4135]
[2, 947202, 947187.0, 4139]
[6, 947202, 947190.0, 28418]
[2, 947218, 947195.0, 4143]
[7, 947218, 947198.0, 28417]
[1, 947218, 947201.0, 28384]
[2, 947218, 947203.0, 4147]
[2, 947233, 947212.0, 4151]
[8, 947233, 947213.0, 21355]
[2, 947233, 947220.0, 4155]
[6, 947233, 947224.0, 28419]
[2, 947249, 947229.0, 4159]
[7, 947249, 947231.0, 28418]
[1, 947249, 947234.0, 28385]
[2, 947249, 947239.0, 4163]
[2, 947264, 947248.0, 4167]
[6, 947280, 947257.0, 28420]
[8, 947280, 947257.0, 21356]
[2, 947280, 947258.0, 4171]
[2, 947280, 947264.0, 4173]
[7, 947280, 947265.0, 28419]
[1, 947280, 947268.0, 28386]
[2, 947296, 947274.0, 4177]
[2, 947296, 947284.0, 4181]
[2, 947311, 947290.0, 4183]
[6, 947311, 947290.0, 28421]
[7, 947311, 947298.0, 28420]
[2, 947311, 947301.0, 4187]
[1, 947311, 947301.0, 28387]
[8, 947327, 947301.0, 21357]
[2, 947327, 947311.0, 4191]
[2, 947327, 947317.0, 4193]
[2, 947342, 947323.0, 4195]
[6, 947342, 947324.0, 28422]
[2, 947342, 947329.0, 4197]
[7, 947342, 947331.0, 28421]
[1, 947358, 947334.0, 28388]
[2, 947358, 947335.0, 4199]
[8, 947358, 947346.0, 21358]
[2, 947358, 947346.0, 4203]
[2, 947374, 947352.0, 4205]
[6, 947374, 947357.0, 28423]
[2, 947374, 947363.0, 4209]
[7, 947374, 947364.0, 28422]
[1, 947389, 947368.0, 28389]
[2, 947389, 947373.0, 4213]
[2, 947389, 947379.0, 4215]
[2, 947405, 947388.0, 4219]
[8, 947405, 947390.0, 21359]
[6, 947405, 947390.0, 28424]
[7, 947420, 947398.0, 28423]
[2, 947420, 947398.0, 4223]
[1, 947420, 947401.0, 28390]
[2, 947420, 947407.0, 4227]
[2, 947436, 947416.0, 4231]
[6, 947436, 947424.0, 28425]
[2, 947436, 947425.0, 4235]
[7, 947452, 947431.0, 28424]
[2, 947452, 947434.0, 4239]
[1, 947452, 947434.0, 28391]
[8, 947452, 947434.0, 21360]
[2, 947467, 947443.0, 4243]
[2, 947467, 947452.0, 4247]
[6, 947467, 947457.0, 28426]
[2, 947483, 947460.0, 4251]
[7, 947483, 947464.0, 28425]
[1, 947483, 947468.0, 28392]
[2, 947483, 947468.0, 4255]
[2, 947498, 947477.0, 4259]
[8, 947498, 947479.0, 21361]
[2, 947498, 947485.0, 4263]
[6, 947514, 947490.0, 28427]
[2, 947514, 947493.0, 4267]
[7, 947514, 947498.0, 28426]
[1, 947514, 947501.0, 28393]
[2, 947514, 947502.0, 4271]
[2, 947530, 947511.0, 4275]
[2, 947530, 947520.0, 4279]
[8, 947545, 947523.0, 21362]
[6, 947545, 947524.0, 28428]
[2, 947545, 947529.0, 4283]
[7, 947545, 947531.0, 28427]
[1, 947545, 947534.0, 28394]
[2, 947561, 947539.0, 4287]
[2, 947561, 947548.0, 4291]
[6, 947576, 947557.0, 28429]
[2, 947576, 947558.0, 4295]
[7, 947576, 947564.0, 28428]
[2, 947592, 947567.0, 4299]
[8, 947592, 947567.0, 21363]
[1, 947592, 947568.0, 28395]
[2, 947592, 947576.0, 4303]
[2, 947608, 947586.0, 4307]
[6, 947608, 947590.0, 28430]
[2, 947608, 947595.0, 4311]
[7, 947608, 947598.0, 28429]
[1, 947623, 947601.0, 28396]
[2, 947623, 947604.0, 4315]
[8, 947623, 947612.0, 21364]
[2, 947623, 947613.0, 4319]
[2, 947639, 947621.0, 4323]
[6, 947639, 947624.0, 28431]
[2, 947639, 947630.0, 4327]
[7, 947654, 947631.0, 28430]
[1, 947654, 947634.0, 28397]
[2, 947654, 947639.0, 4331]
[2, 947654, 947647.0, 4335]
[8, 947670, 947656.0, 21365]
[2, 947670, 947656.0, 4339]
[6, 947670, 947657.0, 28432]
[7, 947686, 947665.0, 28431]
[2, 947686, 947665.0, 4343]
[1, 947686, 947668.0, 28398]
[2, 947686, 947675.0, 4347]
[2, 947701, 947685.0, 4351]
[6, 947701, 947690.0, 28433]
[2, 947717, 947695.0, 4355]
[7, 947717, 947698.0, 28432]
[8, 947717, 947701.0, 21366]
[1, 947717, 947701.0, 28399]
[2, 947717, 947705.0, 4359]
[2, 947732, 947711.0, 4361]
[2, 947732, 947721.0, 4365]
[6, 947732, 947724.0, 28434]
[2, 947748, 947727.0, 4367]
[7, 947748, 947731.0, 28433]
[1, 947748, 947734.0, 28400]
[2, 947748, 947737.0, 4371]
[8, 947764, 947745.0, 21367]
[2, 947764, 947747.0, 4375]
[2, 947779, 947757.0, 4379]
[6, 947779, 947757.0, 28435]
[7, 947779, 947765.0, 28434]
[2, 947779, 947766.0, 4383]
[1, 947779, 947768.0, 28401]
[2, 947795, 947775.0, 4387]
[2, 947795, 947784.0, 4391]
[8, 947810, 947789.0, 21368]
[6, 947810, 947790.0, 28436]
[2, 947810, 947793.0, 4395]
[7, 947810, 947798.0, 28435]
[1, 947810, 947801.0, 28402]
[2, 947810, 947803.0, 4399]
[2, 947826, 947812.0, 4403]
[2, 947842, 947821.0, 4407]
[6, 947842, 947824.0, 28437]
[2, 947842, 947831.0, 4411]
[7, 947842, 947831.0, 28436]
[8, 947857, 947834.0, 21369]
[1, 947857, 947835.0, 28403]
[2, 947857, 947840.0, 4415]
[2, 947873, 947849.0, 4419]
[6, 947873, 947857.0, 28438]
[2, 947873, 947857.0, 4423]
[0, 947873, 947863.0, 9]
[7, 947873, 947864.0, 28437]
[2, 947888, 947866.0, 4427]
[1, 947888, 947868.0, 28404]
[2, 947888, 947875.0, 4431]
[8, 947888, 947878.0, 21370]
[2, 947904, 947883.0, 4435]
[6, 947904, 947890.0, 28439]
[2, 947904, 947892.0, 4439]
[7, 947920, 947898.0, 28438]
[2, 947920, 947901.0, 4443]
[1, 947920, 947901.0, 28405]
[2, 947920, 947909.0, 4447]
[2, 947935, 947918.0, 4451]
[8, 947935, 947922.0, 21371]
[6, 947935, 947923.0, 28440]
[2, 947935, 947926.0, 4455]
[7, 947951, 947931.0, 28439]
[2, 947951, 947934.0, 4459]
[1, 947951, 947935.0, 28406]
[2, 947951, 947942.0, 4463]
[2, 947966, 947950.0, 4467]
[6, 947966, 947957.0, 28441]
[2, 947966, 947958.0, 4471]
[7, 947982, 947964.0, 28440]
[2, 947982, 947966.0, 4475]
[8, 947982, 947967.0, 21372]
[1, 947982, 947968.0, 28407]
[2, 947982, 947974.0, 4479]
[2, 947998, 947983.0, 4483]
[6, 948013, 947990.0, 28442]
[2, 948013, 947991.0, 4487]
[7, 948013, 947998.0, 28441]
[2, 948013, 948000.0, 4491]
[1, 948013, 948001.0, 28408]
[2, 948029, 948009.0, 4495]
[8, 948029, 948011.0, 21373]
[2, 948029, 948018.0, 4499]
[6, 948044, 948023.0, 28443]
[2, 948044, 948027.0, 4503]
[7, 948044, 948031.0, 28442]
[1, 948044, 948035.0, 28409]
[2, 948060, 948038.0, 4507]
[2, 948060, 948044.0, 4509]
[2, 948060, 948050.0, 4511]
[8, 948076, 948055.0, 21374]
[6, 948076, 948057.0, 28444]
[2, 948076, 948057.0, 4513]
[2, 948076, 948064.0, 4515]
[7, 948091, 948064.0, 28443]
[1, 948091, 948068.0, 28410]
[2, 948091, 948072.0, 4517]
[2, 948091, 948081.0, 4519]
[6, 948107, 948090.0, 28445]
[2, 948107, 948090.0, 4521]
[7, 948107, 948098.0, 28444]
[2, 948122, 948100.0, 4523]
[8, 948122, 948100.0, 21375]
[1, 948122, 948101.0, 28411]
[2, 948122, 948109.0, 4525]
[2, 948138, 948120.0, 4527]
[6, 948138, 948123.0, 28446]
[7, 948154, 948131.0, 28445]
[2, 948154, 948132.0, 4529]
[1, 948154, 948135.0, 28412]
[8, 948169, 948144.0, 21376]
[2, 948169, 948148.0, 4531]
[6, 948169, 948157.0, 28447]
[7, 948185, 948164.0, 28446]
[2, 948185, 948167.0, 4533]
[1, 948185, 948168.0, 28413]
[2, 948200, 948188.0, 4535]
[8, 948200, 948188.0, 21377]
[6, 948200, 948190.0, 28448]
[7, 948216, 948198.0, 28447]
[1, 948216, 948201.0, 28414]
[2, 948216, 948208.0, 4537]
[6, 948247, 948223.0, 28449]
[7, 948247, 948231.0, 28448]
[8, 948247, 948233.0, 21378]
[1, 948247, 948235.0, 28415]
[2, 948263, 948242.0, 4539]
[6, 948278, 948257.0, 28450]
[7, 948278, 948264.0, 28449]
[1, 948278, 948268.0, 28416]
[8, 948294, 948277.0, 21379]
[6, 948310, 948290.0, 28451]
[7, 948310, 948298.0, 28450]
[1, 948325, 948301.0, 28417]
[8, 948341, 948322.0, 21380]
[6, 948341, 948323.0, 28452]
[7, 948341, 948331.0, 28451]
[1, 948356, 948335.0, 28418]
[2, 948356, 948339.0, 4541]
[6, 948372, 948357.0, 28453]
[7, 948388, 948364.0, 28452]
[8, 948388, 948366.0, 21381]
[1, 948388, 948368.0, 28419]
[2, 948388, 948369.0, 4543]
[6, 948403, 948390.0, 28454]
[2, 948419, 948393.0, 4545]
[7, 948419, 948398.0, 28453]
[1, 948419, 948402.0, 28420]
[8, 948434, 948410.0, 21382]
[2, 948434, 948416.0, 4547]
[6, 948434, 948423.0, 28455]
[7, 948450, 948431.0, 28454]
[1, 948450, 948435.0, 28421]
[2, 948450, 948438.0, 4549]
[8, 948466, 948455.0, 21383]
[6, 948466, 948457.0, 28456]
[2, 948481, 948459.0, 4551]
[7, 948481, 948464.0, 28455]
[1, 948481, 948468.0, 28422]
[2, 948497, 948481.0, 4553]
[6, 948512, 948490.0, 28457]
[7, 948512, 948498.0, 28456]
[8, 948512, 948499.0, 21384]
[1, 948512, 948502.0, 28423]
[2, 948528, 948505.0, 4555]
[6, 948544, 948523.0, 28458]
[2, 948544, 948527.0, 4557]
[7, 948544, 948531.0, 28457]
[1, 948544, 948535.0, 28424]
[8, 948559, 948543.0, 21385]
[2, 948575, 948549.0, 4559]
[5, 948575, 948554.0, 74]
[4, 948575, 948554.0, 29.0]
[6, 948575, 948557.0, 28459]
[7, 948575, 948564.0, 28458]
[1, 948590, 948568.0, 28425]
[2, 948590, 948575.0, 2]
[8, 948606, 948588.0, 21386]
[6, 948606, 948590.0, 28460]
[7, 948606, 948598.0, 28459]
[2, 948622, 948599.0, 4]
[1, 948622, 948602.0, 28426]
[2, 948637, 948618.0, 6]
[6, 948637, 948623.0, 28461]
[7, 948653, 948631.0, 28460]
[8, 948653, 948632.0, 21387]
[2, 948653, 948634.0, 8]
[1, 948653, 948635.0, 28427]
[2, 948668, 948649.0, 10]
[6, 948668, 948657.0, 28462]
[2, 948684, 948664.0, 12]
[7, 948684, 948664.0, 28461]
[3, 948684, 948665.0, 550]
[1, 948684, 948668.0, 28428]
[8, 948700, 948676.0, 21388]
[2, 948700, 948679.0, 14]
[6, 948700, 948690.0, 28463]
[2, 948700, 948692.0, 16]
[7, 948715, 948698.0, 28462]
[1, 948715, 948702.0, 28429]
[2, 948715, 948705.0, 18]
[2, 948731, 948717.0, 20]
[8, 948746, 948721.0, 21389]
[6, 948746, 948723.0, 28464]
[2, 948746, 948728.0, 22]
[7, 948746, 948731.0, 28463]
[1, 948746, 948735.0, 28430]
[2, 948762, 948738.0, 24]
[2, 948762, 948749.0, 26]
[6, 948778, 948757.0, 28465]
[2, 948778, 948760.0, 28]
[7, 948778, 948764.0, 28464]
[8, 948778, 948765.0, 21390]
[1, 948778, 948768.0, 28431]
[2, 948793, 948771.0, 30]
[2, 948793, 948782.0, 32]
[6, 948809, 948790.0, 28466]
[2, 948809, 948793.0, 34]
# STOPPED EXPERIMENT"""

    FILEPATH: ClassVar[Path] = Path('.test.riglog')
    RIG: RiglogData

    @classmethod
    def setUpClass(cls):
        with open(cls.FILEPATH, 'w') as f:
            f.write(cls.RIGLOG_CONTENT)

        cls.RIG = RiglogData(cls.FILEPATH)

    @classmethod
    def tearDownClass(cls):
        cls.FILEPATH.unlink()
        cls.FILEPATH.with_stem('.test_riglog').with_suffix('.npy').unlink()

    def test_source_version(self):
        self.assertEqual(self.RIG.log_config['source_version'], 'stimpy-bit')

    def test_version(self):
        self.assertEqual(self.RIG.log_config['version'], 0.3)

    def test_commit_hash(self):
        self.assertEqual(self.RIG.log_config['commit_hash'], '6d30281')

    def test_codes(self):
        res = {
            'screen': 0,
            'imaging': 1,
            'position': 2,
            'lick': 3,
            'reward': 4,
            'lap': 5,
            'cam1': 6,
            'cam2': 7,
            'cam3': 8,
            'act0': 21,
            'act1': 22
        }
        self.assertDictEqual(self.RIG.log_config['codes'], res)

    def test_csv_fields(self):
        res = ('code', 'time received', 'duino time', 'value')
        self.assertTupleEqual(self.RIG.log_config['fields'], res)

    def test_screen_event(self):
        assert_array_equal(self.RIG.screen_event.time, np.array([947.863]))
        assert_array_equal(self.RIG.screen_event.value, np.array([9]))

    def test_imaging_event(self):
        event = self.RIG.imaging_event
        t = np.array(
            [
                946.434, 946.467, 946.5, 946.534, 946.567, 946.6, 946.634, 946.667, 946.701,
                946.734, 946.767, 946.801, 946.834, 946.867, 946.901, 946.934, 946.967, 947.001,
                947.034, 947.067, 947.101, 947.134, 947.167, 947.201, 947.234, 947.268, 947.301,
                947.334, 947.368, 947.401, 947.434, 947.468, 947.501, 947.534, 947.568, 947.601,
                947.634, 947.668, 947.701, 947.734, 947.768, 947.801, 947.835, 947.868, 947.901,
                947.935, 947.968, 948.001, 948.035, 948.068, 948.101, 948.135, 948.168, 948.201,
                948.235, 948.268, 948.301, 948.335, 948.368, 948.402, 948.435, 948.468, 948.502,
                948.535, 948.568, 948.602, 948.635, 948.668, 948.702, 948.735, 948.768
            ]

        )

        v = np.array(
            [28361, 28362, 28363, 28364, 28365, 28366, 28367, 28368, 28369, 28370,
             28371, 28372, 28373, 28374, 28375, 28376, 28377, 28378, 28379, 28380,
             28381, 28382, 28383, 28384, 28385, 28386, 28387, 28388, 28389, 28390,
             28391, 28392, 28393, 28394, 28395, 28396, 28397, 28398, 28399, 28400,
             28401, 28402, 28403, 28404, 28405, 28406, 28407, 28408, 28409, 28410,
             28411, 28412, 28413, 28414, 28415, 28416, 28417, 28418, 28419, 28420,
             28421, 28422, 28423, 28424, 28425, 28426, 28427, 28428, 28429, 28430,
             28431]
        )

        assert_array_equal(event.time, t)
        assert_array_equal(event.value, v)

    def test_position_event(self):
        event = self.RIG.position_event
        v = [3.735e+03, 3.739e+03, 3.743e+03, 3.747e+03, 3.751e+03, 3.755e+03, 3.759e+03,
             3.763e+03, 3.767e+03, 3.771e+03, 3.775e+03, 3.779e+03, 3.783e+03, 3.787e+03,
             3.791e+03, 3.795e+03, 3.799e+03, 3.803e+03, 3.807e+03, 3.811e+03, 3.815e+03,
             3.819e+03, 3.823e+03, 3.827e+03, 3.831e+03, 3.835e+03, 3.839e+03, 3.843e+03,
             3.847e+03, 3.851e+03, 3.855e+03, 3.859e+03, 3.863e+03, 3.867e+03, 3.871e+03,
             3.875e+03, 3.879e+03, 3.883e+03, 3.887e+03, 3.891e+03, 3.895e+03, 3.899e+03,
             3.903e+03, 3.907e+03, 3.911e+03, 3.915e+03, 3.919e+03, 3.923e+03, 3.927e+03,
             3.931e+03, 3.935e+03, 3.939e+03, 3.943e+03, 3.947e+03, 3.951e+03, 3.955e+03,
             3.959e+03, 3.963e+03, 3.967e+03, 3.971e+03, 3.975e+03, 3.979e+03, 3.983e+03,
             3.987e+03, 3.991e+03, 3.995e+03, 3.999e+03, 4.003e+03, 4.007e+03, 4.011e+03,
             4.015e+03, 4.019e+03, 4.023e+03, 4.027e+03, 4.031e+03, 4.035e+03, 4.039e+03,
             4.043e+03, 4.047e+03, 4.051e+03, 4.055e+03, 4.059e+03, 4.063e+03, 4.067e+03,
             4.071e+03, 4.075e+03, 4.079e+03, 4.083e+03, 4.087e+03, 4.091e+03, 4.095e+03,
             4.099e+03, 4.103e+03, 4.107e+03, 4.111e+03, 4.115e+03, 4.119e+03, 4.123e+03,
             4.127e+03, 4.131e+03, 4.135e+03, 4.139e+03, 4.143e+03, 4.147e+03, 4.151e+03,
             4.155e+03, 4.159e+03, 4.163e+03, 4.167e+03, 4.171e+03, 4.173e+03, 4.177e+03,
             4.181e+03, 4.183e+03, 4.187e+03, 4.191e+03, 4.193e+03, 4.195e+03, 4.197e+03,
             4.199e+03, 4.203e+03, 4.205e+03, 4.209e+03, 4.213e+03, 4.215e+03, 4.219e+03,
             4.223e+03, 4.227e+03, 4.231e+03, 4.235e+03, 4.239e+03, 4.243e+03, 4.247e+03,
             4.251e+03, 4.255e+03, 4.259e+03, 4.263e+03, 4.267e+03, 4.271e+03, 4.275e+03,
             4.279e+03, 4.283e+03, 4.287e+03, 4.291e+03, 4.295e+03, 4.299e+03, 4.303e+03,
             4.307e+03, 4.311e+03, 4.315e+03, 4.319e+03, 4.323e+03, 4.327e+03, 4.331e+03,
             4.335e+03, 4.339e+03, 4.343e+03, 4.347e+03, 4.351e+03, 4.355e+03, 4.359e+03,
             4.361e+03, 4.365e+03, 4.367e+03, 4.371e+03, 4.375e+03, 4.379e+03, 4.383e+03,
             4.387e+03, 4.391e+03, 4.395e+03, 4.399e+03, 4.403e+03, 4.407e+03, 4.411e+03,
             4.415e+03, 4.419e+03, 4.423e+03, 4.427e+03, 4.431e+03, 4.435e+03, 4.439e+03,
             4.443e+03, 4.447e+03, 4.451e+03, 4.455e+03, 4.459e+03, 4.463e+03, 4.467e+03,
             4.471e+03, 4.475e+03, 4.479e+03, 4.483e+03, 4.487e+03, 4.491e+03, 4.495e+03,
             4.499e+03, 4.503e+03, 4.507e+03, 4.509e+03, 4.511e+03, 4.513e+03, 4.515e+03,
             4.517e+03, 4.519e+03, 4.521e+03, 4.523e+03, 4.525e+03, 4.527e+03, 4.529e+03,
             4.531e+03, 4.533e+03, 4.535e+03, 4.537e+03, 4.539e+03, 4.541e+03, 4.543e+03,
             4.545e+03, 4.547e+03, 4.549e+03, 4.551e+03, 4.553e+03, 4.555e+03, 4.557e+03,
             4.559e+03, 2.000e+00, 4.000e+00, 6.000e+00, 8.000e+00, 1.000e+01, 1.200e+01,
             1.400e+01, 1.600e+01, 1.800e+01, 2.000e+01, 2.200e+01, 2.400e+01, 2.600e+01,
             2.800e+01, 3.000e+01, 3.200e+01, 3.400e+01]

        t = np.array([
            946.403, 946.41, 946.417, 946.424, 946.431, 946.438, 946.446, 946.453, 946.46, 946.467, 946.474, 946.481,
            946.488, 946.496, 946.503, 946.511, 946.518, 946.526, 946.534, 946.541, 946.549, 946.556, 946.564, 946.571,
            946.579, 946.586, 946.593, 946.601, 946.609, 946.616, 946.624, 946.631, 946.638, 946.644, 946.651, 946.657,
            946.663, 946.67, 946.676, 946.682, 946.688, 946.694, 946.7, 946.706, 946.713, 946.719, 946.726, 946.732,
            946.739, 946.747, 946.754, 946.762, 946.77, 946.778, 946.786, 946.794, 946.801, 946.808, 946.815, 946.822,
            946.83, 946.837, 946.845, 946.853, 946.862, 946.87, 946.88, 946.889, 946.899, 946.908, 946.918, 946.928,
            946.937, 946.947, 946.956, 946.964, 946.973, 946.981, 946.99, 946.999, 947.007, 947.016, 947.024, 947.033,
            947.041, 947.049, 947.057, 947.065, 947.073, 947.081, 947.09, 947.098, 947.107, 947.116, 947.125, 947.134,
            947.144, 947.152, 947.161, 947.17, 947.178, 947.187, 947.195, 947.203, 947.212, 947.22, 947.229, 947.239,
            947.248, 947.258, 947.264, 947.274, 947.284, 947.29, 947.301, 947.311, 947.317, 947.323, 947.329, 947.335,
            947.346, 947.352, 947.363, 947.373, 947.379, 947.388, 947.398, 947.407, 947.416, 947.425, 947.434, 947.443,
            947.452, 947.46, 947.468, 947.477, 947.485, 947.493, 947.502, 947.511, 947.52, 947.529, 947.539, 947.548,
            947.558, 947.567, 947.576, 947.586, 947.595, 947.604, 947.613, 947.621, 947.63, 947.639, 947.647, 947.656,
            947.665, 947.675, 947.685, 947.695, 947.705, 947.711, 947.721, 947.727, 947.737, 947.747, 947.757, 947.766,
            947.775, 947.784, 947.793, 947.803, 947.812, 947.821, 947.831, 947.84, 947.849, 947.857, 947.866, 947.875,
            947.883, 947.892, 947.901, 947.909, 947.918, 947.926, 947.934, 947.942, 947.95, 947.958, 947.966, 947.974,
            947.983, 947.991, 948.0, 948.009, 948.018, 948.027, 948.038, 948.044, 948.05, 948.057, 948.064, 948.072,
            948.081, 948.09, 948.1, 948.109, 948.12, 948.132, 948.148, 948.167, 948.188, 948.208, 948.242, 948.339,
            948.369, 948.393, 948.416, 948.438, 948.459, 948.481, 948.505, 948.527, 948.549, 948.575, 948.599, 948.618,
            948.634, 948.649, 948.664, 948.679, 948.692, 948.705, 948.717, 948.728, 948.738, 948.749, 948.76, 948.771,
            948.782, 948.793
        ])

        assert_array_equal(event.value, v)
        assert_array_equal(event.time, t)

    def test_lick_event(self):
        event = self.RIG.lick_event
        assert_array_equal(event.time, np.array([948.665]))
        assert_array_equal(event.value, np.array([550]))

    def test_reward_event(self):
        event = self.RIG.reward_event
        assert_array_equal(event.time, np.array([948.554]))
        assert_array_equal(event.value, np.array([29]))

    def test_lap_event(self):
        event = self.RIG.lap_event
        assert_array_equal(event.time, np.array([948.554]))
        assert_array_equal(event.value, np.array([74]))

    def test_cam1_event(self):
        event = self.RIG.camera_event['facecam']
        t = np.array([
            946.424, 946.457, 946.49, 946.524, 946.557, 946.59, 946.624, 946.657, 946.69, 946.724, 946.757, 946.79,
            946.824, 946.857, 946.89, 946.924, 946.957, 946.99, 947.024, 947.057, 947.09, 947.124, 947.157, 947.19,
            947.224, 947.257, 947.29, 947.324, 947.357, 947.39, 947.424, 947.457, 947.49, 947.524, 947.557, 947.59,
            947.624, 947.657, 947.69, 947.724, 947.757, 947.79, 947.824, 947.857, 947.89, 947.923, 947.957, 947.99,
            948.023, 948.057, 948.09, 948.123, 948.157, 948.19, 948.223, 948.257, 948.29, 948.323, 948.357, 948.39,
            948.423, 948.457, 948.49, 948.523, 948.557, 948.59, 948.623, 948.657, 948.69, 948.723, 948.757, 948.79
        ])

        v = np.array([
            28395.0, 28396.0, 28397.0, 28398.0, 28399.0, 28400.0, 28401.0, 28402.0, 28403.0, 28404.0, 28405.0, 28406.0,
            28407.0, 28408.0, 28409.0, 28410.0, 28411.0, 28412.0, 28413.0, 28414.0, 28415.0, 28416.0, 28417.0, 28418.0,
            28419.0, 28420.0, 28421.0, 28422.0, 28423.0, 28424.0, 28425.0, 28426.0, 28427.0, 28428.0, 28429.0, 28430.0,
            28431.0, 28432.0, 28433.0, 28434.0, 28435.0, 28436.0, 28437.0, 28438.0, 28439.0, 28440.0, 28441.0, 28442.0,
            28443.0, 28444.0, 28445.0, 28446.0, 28447.0, 28448.0, 28449.0, 28450.0, 28451.0, 28452.0, 28453.0, 28454.0,
            28455.0, 28456.0, 28457.0, 28458.0, 28459.0, 28460.0, 28461.0, 28462.0, 28463.0, 28464.0, 28465.0, 28466.0
        ])

        assert_array_equal(event.value, v)
        assert_array_equal(event.time, t)

    def test_cam2_event(self):
        event = self.RIG.camera_event['eyecam']
        t = np.array([
            946.431, 946.465, 946.498, 946.531, 946.565, 946.598, 946.631, 946.665, 946.698, 946.731, 946.765, 946.798,
            946.831, 946.865, 946.898, 946.931, 946.965, 946.998, 947.031, 947.064, 947.098, 947.131, 947.165, 947.198,
            947.231, 947.265, 947.298, 947.331, 947.364, 947.398, 947.431, 947.464, 947.498, 947.531, 947.564, 947.598,
            947.631, 947.665, 947.698, 947.731, 947.765, 947.798, 947.831, 947.864, 947.898, 947.931, 947.964, 947.998,
            948.031, 948.064, 948.098, 948.131, 948.164, 948.198, 948.231, 948.264, 948.298, 948.331, 948.364, 948.398,
            948.431, 948.464, 948.498, 948.531, 948.564, 948.598, 948.631, 948.664, 948.698, 948.731, 948.764
        ])

        v = np.array([
            28394.0, 28395.0, 28396.0, 28397.0, 28398.0, 28399.0, 28400.0, 28401.0, 28402.0, 28403.0, 28404.0, 28405.0,
            28406.0, 28407.0, 28408.0, 28409.0, 28410.0, 28411.0, 28412.0, 28413.0, 28414.0, 28415.0, 28416.0, 28417.0,
            28418.0, 28419.0, 28420.0, 28421.0, 28422.0, 28423.0, 28424.0, 28425.0, 28426.0, 28427.0, 28428.0, 28429.0,
            28430.0, 28431.0, 28432.0, 28433.0, 28434.0, 28435.0, 28436.0, 28437.0, 28438.0, 28439.0, 28440.0, 28441.0,
            28442.0, 28443.0, 28444.0, 28445.0, 28446.0, 28447.0, 28448.0, 28449.0, 28450.0, 28451.0, 28452.0, 28453.0,
            28454.0, 28455.0, 28456.0, 28457.0, 28458.0, 28459.0, 28460.0, 28461.0, 28462.0, 28463.0, 28464.0
        ])

        assert_array_equal(event.value, v)
        assert_array_equal(event.time, t)

    def test_cam3_event(self):
        event = self.RIG.camera_event['1P_cam']

        t = np.array([
            946.414, 946.459, 946.503, 946.547, 946.592, 946.636, 946.68, 946.725, 946.769, 946.813, 946.858, 946.902,
            946.946, 946.991, 947.035, 947.08, 947.124, 947.168, 947.213, 947.257, 947.301, 947.346, 947.39, 947.434,
            947.479, 947.523, 947.567, 947.612, 947.656, 947.701, 947.745, 947.789, 947.834, 947.878, 947.922, 947.967,
            948.011, 948.055, 948.1, 948.144, 948.188, 948.233, 948.277, 948.322, 948.366, 948.41, 948.455, 948.499,
            948.543, 948.588, 948.632, 948.676, 948.721, 948.765
        ])

        v = np.array([
            21337.0, 21338.0, 21339.0, 21340.0, 21341.0, 21342.0, 21343.0, 21344.0, 21345.0, 21346.0, 21347.0, 21348.0,
            21349.0, 21350.0, 21351.0, 21352.0, 21353.0, 21354.0, 21355.0, 21356.0, 21357.0, 21358.0, 21359.0, 21360.0,
            21361.0, 21362.0, 21363.0, 21364.0, 21365.0, 21366.0, 21367.0, 21368.0, 21369.0, 21370.0, 21371.0, 21372.0,
            21373.0, 21374.0, 21375.0, 21376.0, 21377.0, 21378.0, 21379.0, 21380.0, 21381.0, 21382.0, 21383.0, 21384.0,
            21385.0, 21386.0, 21387.0, 21388.0, 21389.0, 21390.0
        ])

        assert_array_equal(event.value, v)
        assert_array_equal(event.time, t)



class TestStimlogParser(unittest.TestCase):
    STIM_CONTENT = """
    # VLOG HEADER:code, presentTime, iStim, iTrial, photo, contrast, ori, sf, phase, stim_idx
    # CODES: vstim=10
    # Commit: 6d30281
    # Vstim trigger on: imaging,1.0
    10,1627.423497,27,2,1
    10,1627.440233,27,2,1
    10,1627.45699,27,2,1
    10,1627.473689,27,2,1
    10,1627.490393,27,2,1
    10,1627.5071,27,2,1
    10,1627.523794,27,2,1
    10,1627.540494,27,2,1
    10,1627.557217,27,2,1
    # Started state machine v1.2 - timing sync to rig
    # CODES: stateMachine=20
    # STATE HEADER: code,elapsed,cycle,newState,oldState,stateElapsed,trialType
    20,1628143,0,2,1,2012,0
    10,1627.573911,27,2,0,1,90,0.16,0.016667,1
    10,1627.590636,27,2,0,1,90,0.16,0.033333,2
    10,1627.607361,27,2,0,1,90,0.16,0.05,3
    10,1627.624094,27,2,0,1,90,0.16,0.066667,4
    10,1627.640796,27,2,0,1,90,0.16,0.083333,5
    10,1627.657516,27,2,0,1,90,0.16,0.1,6
    10,1627.67422,27,2,0,1,90,0.16,0.116667,7
    10,1627.69092,27,2,0,1,90,0.16,0.133333,8
    10,1627.70765,27,2,0,1,90,0.16,0.15,9
    10,1627.724397,27,2,0,1,90,0.16,0.166667,10
    10,1627.741089,27,2,0,1,90,0.16,0.183333,11
    10,1627.757789,27,2,0,1,90,0.16,0.2,12
    10,1627.774516,27,2,0,1,90,0.16,0.216667,13
    10,1627.791217,27,2,0,1,90,0.16,0.233333,14
    10,1627.807921,27,2,0,1,90,0.16,0.25,15
    10,1627.824654,27,2,0,1,90,0.16,0.266667,16
    10,1627.841366,27,2,0,1,90,0.16,0.283333,17
    10,1627.858064,27,2,0,1,90,0.16,0.3,18
    10,1627.874785,27,2,0,1,90,0.16,0.316667,19
    10,1627.891524,27,2,0,1,90,0.16,0.333333,20
    10,1627.908224,27,2,0,1,90,0.16,0.35,21
    10,1627.924948,27,2,0,1,90,0.16,0.366667,22
    10,1627.941713,27,2,0,1,90,0.16,0.383333,23
    10,1627.958449,27,2,0,1,90,0.16,0.4,24
    10,1627.975146,27,2,0,1,90,0.16,0.416667,25
    10,1627.991856,27,2,0,1,90,0.16,0.433333,26
    10,1628.008554,27,2,0,1,90,0.16,0.45,27
    10,1628.025257,27,2,0,1,90,0.16,0.466667,28
    10,1628.041969,27,2,0,1,90,0.16,0.483333,29
    10,1628.0587,27,2,0,1,90,0.16,0.5,30
    10,1628.075401,27,2,0,1,90,0.16,0.516667,31
    10,1628.0921,27,2,0,1,90,0.16,0.533333,32
    10,1628.108826,27,2,0,1,90,0.16,0.55,33
    10,1628.125593,27,2,0,1,90,0.16,0.566667,34
    10,1628.14229,27,2,0,1,90,0.16,0.583333,35
    10,1628.159016,27,2,0,1,90,0.16,0.6,36
    10,1628.175755,27,2,0,1,90,0.16,0.616667,37
    10,1628.192507,27,2,0,1,90,0.16,0.633333,38
    10,1628.209217,27,2,0,1,90,0.16,0.65,39
    10,1628.225916,27,2,0,1,90,0.16,0.666667,40
    10,1628.242787,27,2,0,1,90,0.16,0.683333,41
    10,1628.25941,27,2,0,1,90,0.16,0.7,42
    10,1628.276121,27,2,0,1,90,0.16,0.716667,43
    10,1628.29282,27,2,0,1,90,0.16,0.733333,44
    10,1628.309551,27,2,0,1,90,0.16,0.75,45
    10,1628.326262,27,2,0,1,90,0.16,0.766667,46
    10,1628.342981,27,2,0,1,90,0.16,0.783333,47
    10,1628.359684,27,2,0,1,90,0.16,0.8,48
    10,1628.376407,27,2,0,1,90,0.16,0.816667,49
    10,1628.393102,27,2,0,1,90,0.16,0.833333,50
    10,1628.409801,27,2,0,1,90,0.16,0.85,51
    10,1628.426548,27,2,0,1,90,0.16,0.866667,52
    10,1628.443316,27,2,0,1,90,0.16,0.883333,53
    10,1628.46004,27,2,0,1,90,0.16,0.9,54
    10,1628.476812,27,2,0,1,90,0.16,0.916667,55
    10,1628.493492,27,2,0,1,90,0.16,0.933333,56
    10,1628.510189,27,2,0,1,90,0.16,0.95,57
    10,1628.526927,27,2,0,1,90,0.16,0.966667,58
    10,1628.543703,27,2,0,1,90,0.16,0.983333,59
    10,1628.560317,27,2,0,1,90,0.16,1.0,60
    10,1628.577091,27,2,0,1,90,0.16,1.016667,61
    10,1628.593834,27,2,0,1,90,0.16,1.033333,62
    10,1628.610522,27,2,0,1,90,0.16,1.05,63
    10,1628.627228,27,2,0,1,90,0.16,1.066667,64
    10,1628.643909,27,2,0,1,90,0.16,1.083333,65
    10,1628.660671,27,2,0,1,90,0.16,1.1,66
    10,1628.677371,27,2,0,1,90,0.16,1.116667,67
    10,1628.694062,27,2,0,1,90,0.16,1.133333,68
    10,1628.71079,27,2,0,1,90,0.16,1.15,69
    10,1628.727406,27,2,0,1,90,0.16,1.166667,70
    10,1628.744132,27,2,0,1,90,0.16,1.183333,71
    10,1628.760873,27,2,0,1,90,0.16,1.2,72
    10,1628.777577,27,2,0,1,90,0.16,1.216667,73
    10,1628.794421,27,2,0,1,90,0.16,1.233333,74
    10,1628.811107,27,2,0,1,90,0.16,1.25,75
    10,1628.827779,27,2,0,1,90,0.16,1.266667,76
    10,1628.844543,27,2,0,1,90,0.16,1.283333,77
    10,1628.861133,27,2,0,1,90,0.16,1.3,78
    10,1628.87785,27,2,0,1,90,0.16,1.316667,79
    10,1628.894553,27,2,0,1,90,0.16,1.333333,80
    10,1628.911286,27,2,0,1,90,0.16,1.35,81
    10,1628.928033,27,2,0,1,90,0.16,1.366667,82
    10,1628.944734,27,2,0,1,90,0.16,1.383333,83
    10,1628.961414,27,2,0,1,90,0.16,1.4,84
    10,1628.978169,27,2,0,1,90,0.16,1.416667,85
    10,1628.994931,27,2,0,1,90,0.16,1.433333,86
    10,1629.011626,27,2,0,1,90,0.16,1.45,87
    10,1629.028337,27,2,0,1,90,0.16,1.466667,88
    10,1629.045035,27,2,0,1,90,0.16,1.483333,89
    10,1629.061876,27,2,0,1,90,0.16,1.5,90
    10,1629.078499,27,2,0,1,90,0.16,1.516667,91
    10,1629.095196,27,2,0,1,90,0.16,1.533333,92
    10,1629.112105,27,2,0,1,90,0.16,1.55,93
    10,1629.128714,27,2,0,1,90,0.16,1.566667,94
    10,1629.145437,27,2,0,1,90,0.16,1.583333,95
    10,1629.162135,27,2,0,1,90,0.16,1.6,96
    10,1629.178837,27,2,0,1,90,0.16,1.616667,97
    10,1629.195535,27,2,0,1,90,0.16,1.633333,98
    10,1629.21224,27,2,0,1,90,0.16,1.65,99
    10,1629.228939,27,2,0,1,90,0.16,1.666667,100
    10,1629.245651,27,2,0,1,90,0.16,1.683333,101
    10,1629.262414,27,2,0,1,90,0.16,1.7,102
    10,1629.279177,27,2,0,1,90,0.16,1.716667,103
    10,1629.295852,27,2,0,1,90,0.16,1.733333,104
    10,1629.312558,27,2,0,1,90,0.16,1.75,105
    10,1629.329317,27,2,0,1,90,0.16,1.766667,106
    10,1629.345952,27,2,0,1,90,0.16,1.783333,107
    10,1629.362841,27,2,0,1,90,0.16,1.8,108
    10,1629.379414,27,2,0,1,90,0.16,1.816667,109
    10,1629.396119,27,2,0,1,90,0.16,1.833333,110
    10,1629.412838,27,2,0,1,90,0.16,1.85,111
    10,1629.429543,27,2,0,1,90,0.16,1.866667,112
    10,1629.446245,27,2,0,1,90,0.16,1.883333,113
    10,1629.462962,27,2,0,1,90,0.16,1.9,114
    10,1629.479661,27,2,0,1,90,0.16,1.916667,115
    10,1629.496361,27,2,0,1,90,0.16,1.933333,116
    10,1629.513111,27,2,0,1,90,0.16,1.95,117
    10,1629.529843,27,2,0,1,90,0.16,1.966667,118
    10,1629.546549,27,2,0,1,90,0.16,1.983333,119
    10,1629.563252,27,2,0,1,90,0.16,2.0,120
    10,1629.57999,27,2,0,1,90,0.16,2.016667,121
    10,1629.596716,27,2,0,1,90,0.16,2.033333,122
    10,1629.613418,27,2,0,1,90,0.16,2.05,123
    10,1629.630117,27,2,0,1,90,0.16,2.066667,124
    10,1629.646853,27,2,0,1,90,0.16,2.083333,125
    10,1629.663554,27,2,0,1,90,0.16,2.1,126
    10,1629.680281,27,2,0,1,90,0.16,2.116667,127
    10,1629.697041,27,2,0,1,90,0.16,2.133333,128
    10,1629.713735,27,2,0,1,90,0.16,2.15,129
    10,1629.730445,27,2,0,1,90,0.16,2.166667,130
    10,1629.74716,27,2,0,1,90,0.16,2.183333,131
    10,1629.763886,27,2,0,1,90,0.16,2.2,132
    10,1629.780585,27,2,0,1,90,0.16,2.216667,133
    10,1629.7973,27,2,0,1,90,0.16,2.233333,134
    10,1629.81404,27,2,0,1,90,0.16,2.25,135
    10,1629.830777,27,2,0,1,90,0.16,2.266667,136
    10,1629.847489,27,2,0,1,90,0.16,2.283333,137
    10,1629.864191,27,2,0,1,90,0.16,2.3,138
    10,1629.880892,27,2,0,1,90,0.16,2.316667,139
    10,1629.897583,27,2,0,1,90,0.16,2.333333,140
    10,1629.9143,27,2,0,1,90,0.16,2.35,141
    10,1629.931013,27,2,0,1,90,0.16,2.366667,142
    10,1629.947716,27,2,0,1,90,0.16,2.383333,143
    10,1629.964415,27,2,0,1,90,0.16,2.4,144
    10,1629.98114,27,2,0,1,90,0.16,2.416667,145
    10,1629.997856,27,2,0,1,90,0.16,2.433333,146
    10,1630.014568,27,2,0,1,90,0.16,2.45,147
    10,1630.03131,27,2,0,1,90,0.16,2.466667,148
    10,1630.048011,27,2,0,1,90,0.16,2.483333,149
    10,1630.064712,27,2,0,1,90,0.16,2.5,150
    10,1630.081438,27,2,0,1,90,0.16,2.516667,151
    10,1630.098138,27,2,0,1,90,0.16,2.533333,152
    10,1630.114843,27,2,0,1,90,0.16,2.55,153
    10,1630.131599,27,2,0,1,90,0.16,2.566667,154
    10,1630.148297,27,2,0,1,90,0.16,2.583333,155
    10,1630.164994,27,2,0,1,90,0.16,2.6,156
    10,1630.181706,27,2,0,1,90,0.16,2.616667,157
    10,1630.198425,27,2,0,1,90,0.16,2.633333,158
    10,1630.215267,27,2,0,1,90,0.16,2.65,159
    10,1630.231978,27,2,0,1,90,0.16,2.666667,160
    10,1630.248677,27,2,0,1,90,0.16,2.683333,161
    10,1630.265354,27,2,0,1,90,0.16,2.7,162
    10,1630.282141,27,2,0,1,90,0.16,2.716667,163
    10,1630.298714,27,2,0,1,90,0.16,2.733333,164
    10,1630.315417,27,2,0,1,90,0.16,2.75,165
    10,1630.332136,27,2,0,1,90,0.16,2.766667,166
    10,1630.348984,27,2,0,1,90,0.16,2.783333,167
    10,1630.365653,27,2,0,1,90,0.16,2.8,168
    10,1630.382336,27,2,0,1,90,0.16,2.816667,169
    10,1630.399116,27,2,0,1,90,0.16,2.833333,170
    10,1630.41586,27,2,0,1,90,0.16,2.85,171
    10,1630.432528,27,2,0,1,90,0.16,2.866667,172
    10,1630.449297,27,2,0,1,90,0.16,2.883333,173
    10,1630.46603,27,2,0,1,90,0.16,2.9,174
    10,1630.482735,27,2,0,1,90,0.16,2.916667,175
    10,1630.499422,27,2,0,1,90,0.16,2.933333,176
    10,1630.516149,27,2,0,1,90,0.16,2.95,177
    10,1630.532845,27,2,0,1,90,0.16,2.966667,178
    10,1630.549527,27,2,0,1,90,0.16,2.983333,179
    20,1631154,0,0,2,3010,0
    10,1630.566199,27,2,0,1,90,0.16,2.983333,179
    20,1631154,0,1,0,0,0
    10,1630.5829,37,2,1
    10,1630.599641,37,2,1
    10,1630.616357,37,2,1
    10,1630.633046,37,2,1
    10,1630.649848,37,2,1
    10,1630.666606,37,2,1
    10,1630.683355,37,2,1
    10,1630.699972,37,2,1
    10,1630.716764,37,2,1
    10,1630.733468,37,2,1
    10,1630.750187,37,2,1
    10,1630.766871,37,2,1
    10,1630.783614,37,2,1
    10,1630.800333,37,2,1
    10,1630.81704,37,2,1
    # END OF VSTIM
    # Missed 0 frames
    """
    FILEPATH: ClassVar[Path] = Path('.test.stimlog')
    STIM: Stimlog

    @classmethod
    def setUpClass(cls):
        with open(cls.FILEPATH, 'w') as f:
            f.write(cls.STIM_CONTENT)

        cls.STIM = Stimlog(None, cls.FILEPATH)  # for non-riglog testing

    @classmethod
    def tearDownClass(cls):
        cls.FILEPATH.unlink()

    def test_code_version(self):
        self.assertEqual(self.STIM.code_version, '6d30281')

    def test_log_header(self):
        self.assertDictEqual(self.STIM.log_info, {10: 'vstim', 20: 'stateMachine'})

    def test_log_info(self):
        self.assertDictEqual(self.STIM.log_header, {
            10: ['code', 'presentTime', 'iStim', 'iTrial', 'photo', 'contrast', 'ori', 'sf', 'phase', 'stim_idx'],
            20: ['code', 'elapsed', 'cycle', 'newState', 'oldState', 'stateElapsed', 'trialType']
        })

    def test_visual_present_time(self):
        res = np.array([
            1627.423497, 1627.440233, 1627.45699, 1627.473689, 1627.490393, 1627.5071, 1627.523794, 1627.540494,
            1627.557217, 1627.573911, 1627.590636, 1627.607361, 1627.624094, 1627.640796, 1627.657516, 1627.67422,
            1627.69092, 1627.70765, 1627.724397, 1627.741089, 1627.757789, 1627.774516, 1627.791217, 1627.807921,
            1627.824654, 1627.841366, 1627.858064, 1627.874785, 1627.891524, 1627.908224, 1627.924948, 1627.941713,
            1627.958449, 1627.975146, 1627.991856, 1628.008554, 1628.025257, 1628.041969, 1628.0587, 1628.075401,
            1628.0921, 1628.108826, 1628.125593, 1628.14229, 1628.159016, 1628.175755, 1628.192507, 1628.209217,
            1628.225916, 1628.242787, 1628.25941, 1628.276121, 1628.29282, 1628.309551, 1628.326262, 1628.342981,
            1628.359684, 1628.376407, 1628.393102, 1628.409801, 1628.426548, 1628.443316, 1628.46004, 1628.476812,
            1628.493492, 1628.510189, 1628.526927, 1628.543703, 1628.560317, 1628.577091, 1628.593834, 1628.610522,
            1628.627228, 1628.643909, 1628.660671, 1628.677371, 1628.694062, 1628.71079, 1628.727406, 1628.744132,
            1628.760873, 1628.777577, 1628.794421, 1628.811107, 1628.827779, 1628.844543, 1628.861133, 1628.87785,
            1628.894553, 1628.911286, 1628.928033, 1628.944734, 1628.961414, 1628.978169, 1628.994931, 1629.011626,
            1629.028337, 1629.045035, 1629.061876, 1629.078499, 1629.095196, 1629.112105, 1629.128714, 1629.145437,
            1629.162135, 1629.178837, 1629.195535, 1629.21224, 1629.228939, 1629.245651, 1629.262414, 1629.279177,
            1629.295852, 1629.312558, 1629.329317, 1629.345952, 1629.362841, 1629.379414, 1629.396119, 1629.412838,
            1629.429543, 1629.446245, 1629.462962, 1629.479661, 1629.496361, 1629.513111, 1629.529843, 1629.546549,
            1629.563252, 1629.57999, 1629.596716, 1629.613418, 1629.630117, 1629.646853, 1629.663554, 1629.680281,
            1629.697041, 1629.713735, 1629.730445, 1629.74716, 1629.763886, 1629.780585, 1629.7973, 1629.81404,
            1629.830777, 1629.847489, 1629.864191, 1629.880892, 1629.897583, 1629.9143, 1629.931013, 1629.947716,
            1629.964415, 1629.98114, 1629.997856, 1630.014568, 1630.03131, 1630.048011, 1630.064712, 1630.081438,
            1630.098138, 1630.114843, 1630.131599, 1630.148297, 1630.164994, 1630.181706, 1630.198425, 1630.215267,
            1630.231978, 1630.248677, 1630.265354, 1630.282141, 1630.298714, 1630.315417, 1630.332136, 1630.348984,
            1630.365653, 1630.382336, 1630.399116, 1630.41586, 1630.432528, 1630.449297, 1630.46603, 1630.482735,
            1630.499422, 1630.516149, 1630.532845, 1630.549527, 1630.566199, 1630.5829, 1630.599641, 1630.616357,
            1630.633046, 1630.649848, 1630.666606, 1630.683355, 1630.699972, 1630.716764, 1630.733468, 1630.750187,
            1630.766871, 1630.783614, 1630.800333, 1630.81704
        ])
        assert_array_equal(self.STIM.v_present_time, res)

    def test_visual_stim_idx(self):
        res = np.array([
            27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27,
            27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27,
            27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27,
            27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27,
            27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27,
            27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27,
            27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27, 27,
            37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37
        ])
        assert_array_equal(self.STIM.v_stim, res)

    def test_visual_trial_index(self):
        self.assertTrue(np.all(self.STIM.v_trial == 2))

    def test_visual_photo(self):
        res = np.array([
            1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1
        ])

        assert_array_equal(self.STIM.v_photo, res)

    def test_visual_contrast(self):
        res = np.array([
            -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
            1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
            1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
            1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
            1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
            1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
            1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
            1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
            1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
            1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0
        ])

        assert_array_equal(self.STIM.v_contrast, res)

    def test_visual_ori(self):
        res = np.array([
            -1, -1, -1, -1, -1, -1, -1, -1, -1, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90,
            90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90,
            90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90,
            90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90,
            90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90,
            90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90,
            90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90,
            -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1
        ])
        assert_array_equal(self.STIM.v_ori, res)

    def test_visual_sf(self):
        res = np.array([
            -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16,
            0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16,
            0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16,
            0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16,
            0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16,
            0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16,
            0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16,
            0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16,
            0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16,
            0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16,
            0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, 0.16, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
            -1.0, -1.0, -1.0, -1.0, -1.0, -1.0
        ])

        assert_array_equal(self.STIM.v_sf, res)

    def test_visual_phase(self):
        res = np.array([
            -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, 0.016667, 0.033333, 0.05, 0.066667, 0.083333, 0.1,
            0.116667, 0.133333, 0.15, 0.166667, 0.183333, 0.2, 0.216667, 0.233333, 0.25, 0.266667, 0.283333, 0.3,
            0.316667, 0.333333, 0.35, 0.366667, 0.383333, 0.4, 0.416667, 0.433333, 0.45, 0.466667, 0.483333, 0.5,
            0.516667, 0.533333, 0.55, 0.566667, 0.583333, 0.6, 0.616667, 0.633333, 0.65, 0.666667, 0.683333, 0.7,
            0.716667, 0.733333, 0.75, 0.766667, 0.783333, 0.8, 0.816667, 0.833333, 0.85, 0.866667, 0.883333, 0.9,
            0.916667, 0.933333, 0.95, 0.966667, 0.983333, 1.0, 1.016667, 1.033333, 1.05, 1.066667, 1.083333, 1.1,
            1.116667, 1.133333, 1.15, 1.166667, 1.183333, 1.2, 1.216667, 1.233333, 1.25, 1.266667, 1.283333, 1.3,
            1.316667, 1.333333, 1.35, 1.366667, 1.383333, 1.4, 1.416667, 1.433333, 1.45, 1.466667, 1.483333, 1.5,
            1.516667, 1.533333, 1.55, 1.566667, 1.583333, 1.6, 1.616667, 1.633333, 1.65, 1.666667, 1.683333, 1.7,
            1.716667, 1.733333, 1.75, 1.766667, 1.783333, 1.8, 1.816667, 1.833333, 1.85, 1.866667, 1.883333, 1.9,
            1.916667, 1.933333, 1.95, 1.966667, 1.983333, 2.0, 2.016667, 2.033333, 2.05, 2.066667, 2.083333, 2.1,
            2.116667, 2.133333, 2.15, 2.166667, 2.183333, 2.2, 2.216667, 2.233333, 2.25, 2.266667, 2.283333, 2.3,
            2.316667, 2.333333, 2.35, 2.366667, 2.383333, 2.4, 2.416667, 2.433333, 2.45, 2.466667, 2.483333, 2.5,
            2.516667, 2.533333, 2.55, 2.566667, 2.583333, 2.6, 2.616667, 2.633333, 2.65, 2.666667, 2.683333, 2.7,
            2.716667, 2.733333, 2.75, 2.766667, 2.783333, 2.8, 2.816667, 2.833333, 2.85, 2.866667, 2.883333, 2.9,
            2.916667, 2.933333, 2.95, 2.966667, 2.983333, 2.983333, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0,
            -1.0, -1.0, -1.0, -1.0, -1.0, -1.0, -1.0
        ])
        assert_array_equal(self.STIM.v_phase, res)

    def test_visual_frame_idx(self):
        res = np.array([
            -1, -1, -1, -1, -1, -1, -1, -1, -1, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
            21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
            48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74,
            75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100,
            101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121,
            122, 123, 124, 125, 126, 127, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142,
            143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 160, 161, 162, 163,
            164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 179, -1, -1, -1, -1, -1, -1,
            -1, -1, -1, -1, -1, -1, -1, -1, -1
        ])

        assert_array_equal(self.STIM.v_frame_idx, res)

    def test_state_on_v(self):
        assert_array_equal(self.STIM.s_on_v, np.array([1627.557217, 1630.549527, 1630.566199]))

    def test_state_present_time(self):
        assert_array_equal(self.STIM.s_present_time, np.array([1628.143, 1631.154, 1631.154]))

    def test_state_cycle(self):
        assert_array_equal(self.STIM.s_cycle, np.array([0, 0, 0]))

    def test_new_state(self):
        assert_array_equal(self.STIM.s_new_state, np.array([2, 0, 1]))

    def test_old_state(self):
        assert_array_equal(self.STIM.s_old_state, np.array([1, 2, 0]))

    def test_state_elapsed(self):
        assert_array_equal(self.STIM.s_state_elapsed, np.array([2.012, 3.01, 0.]))

    def test_state_trial_type(self):
        assert_array_equal(self.STIM.s_trial_type, np.array([0, 0, 0]))


if __name__ == '__main__':
    unittest.main()
