import unittest
from pathlib import Path
from typing import ClassVar

import numpy as np

from neuralib.stimpy import RiglogData


class TestRiglogParser(unittest.TestCase):
    RIGLOG_CONTENT = """# RIG GIT COMMIT HASH: 88c4705
# CODES: SCREEN=0,IMAGING=1,ENCODER=2,LICKS=3,BUTTON=4,REWARD=5,LAPS=6,CAM1=7,CAM2=8,CAM3=9,ACT0=10,ACT1=11,OPTO=12
# RIG CSV: code,time received,duino time,value
# STARTED EXPERIMENT
[2, 946422, 946403.0, 3735]
[2, 946422, 946410.0, 3739]
[8, 946422, 946414.0, 21337]
[2, 946438, 946417.0, 3743]
[6, 946438, 946424.0, 28395]
[2, 946438, 946424.0, 3747]
[2, 946453, 946431.0, 3751]
[7, 946453, 946431.0, 28394]
[1, 946453, 946434.0, 28361]
[2, 946453, 946438.0, 3755]
[2, 946469, 946446.0, 3759]
[2, 946469, 946453.0, 3763]
[6, 946469, 946457.0, 28396]
[8, 946469, 946459.0, 21338]
[2, 946484, 946460.0, 3767]
[7, 946484, 946465.0, 28395]
[2, 946484, 946467.0, 3771]
[1, 946484, 946467.0, 28362]
[2, 946484, 946474.0, 3775]
[2, 946500, 946481.0, 3779]
[2, 946500, 946488.0, 3783]
[6, 946500, 946490.0, 28397]
[2, 946516, 946496.0, 3787]
[7, 946516, 946498.0, 28396]
[1, 946516, 946500.0, 28363]
[8, 946516, 946503.0, 21339]
[2, 946516, 946503.0, 3791]
[2, 946531, 946511.0, 3795]
[2, 946531, 946518.0, 3799]
[6, 946547, 946524.0, 28398]
[2, 946547, 946526.0, 3803]
[7, 946547, 946531.0, 28397]
[2, 946547, 946534.0, 3807]
[1, 946547, 946534.0, 28364]
[2, 946562, 946541.0, 3811]
[8, 946562, 946547.0, 21340]
[2, 946562, 946549.0, 3815]
[2, 946578, 946556.0, 3819]
[6, 946578, 946557.0, 28399]
[2, 946578, 946564.0, 3823]
[7, 946578, 946565.0, 28398]
[1, 946578, 946567.0, 28365]
[2, 946594, 946571.0, 3827]
[2, 946594, 946579.0, 3831]
[2, 946594, 946586.0, 3835]
[6, 946609, 946590.0, 28400]
[8, 946609, 946592.0, 21341]
[2, 946609, 946593.0, 3839]
[7, 946609, 946598.0, 28399]
[1, 946625, 946600.0, 28366]
[2, 946625, 946601.0, 3843]
[2, 946625, 946609.0, 3847]
[2, 946640, 946616.0, 3851]
[2, 946640, 946624.0, 3855]
[6, 946640, 946624.0, 28401]
[2, 946640, 946631.0, 3859]
[7, 946656, 946631.0, 28400]
[1, 946656, 946634.0, 28367]
[8, 946656, 946636.0, 21342]
[2, 946656, 946638.0, 3863]
[2, 946656, 946644.0, 3867]
[2, 946672, 946651.0, 3871]
[6, 946672, 946657.0, 28402]
[2, 946672, 946657.0, 3875]
[2, 946672, 946663.0, 3879]
[7, 946687, 946665.0, 28401]
[1, 946687, 946667.0, 28368]
[2, 946687, 946670.0, 3883]
[2, 946687, 946676.0, 3887]
[8, 946703, 946680.0, 21343]
[2, 946703, 946682.0, 3891]
[2, 946703, 946688.0, 3895]
[6, 946703, 946690.0, 28403]
[2, 946718, 946694.0, 3899]
[7, 946718, 946698.0, 28402]
[2, 946718, 946700.0, 3903]
[1, 946718, 946701.0, 28369]
[2, 946718, 946706.0, 3907]
[2, 946734, 946713.0, 3911]
[2, 946734, 946719.0, 3915]
[6, 946734, 946724.0, 28404]
[8, 946734, 946725.0, 21344]
[2, 946750, 946726.0, 3919]
[7, 946750, 946731.0, 28403]
[2, 946750, 946732.0, 3923]
[1, 946750, 946734.0, 28370]
[2, 946750, 946739.0, 3927]
[2, 946765, 946747.0, 3931]
[2, 946765, 946754.0, 3935]
[6, 946781, 946757.0, 28405]
[2, 946781, 946762.0, 3939]
[7, 946781, 946765.0, 28404]
[1, 946781, 946767.0, 28371]
[8, 946781, 946769.0, 21345]
[2, 946781, 946770.0, 3943]
[2, 946796, 946778.0, 3947]
[2, 946796, 946786.0, 3951]
[6, 946812, 946790.0, 28406]
[2, 946812, 946794.0, 3955]
[7, 946812, 946798.0, 28405]
[1, 946812, 946801.0, 28372]
[2, 946812, 946801.0, 3959]
[2, 946828, 946808.0, 3963]
[8, 946828, 946813.0, 21346]
[2, 946828, 946815.0, 3967]
[2, 946843, 946822.0, 3971]
[6, 946843, 946824.0, 28407]
[2, 946843, 946830.0, 3975]
[7, 946843, 946831.0, 28406]
[1, 946843, 946834.0, 28373]
[2, 946859, 946837.0, 3979]
[2, 946859, 946845.0, 3983]
[2, 946874, 946853.0, 3987]
[6, 946874, 946857.0, 28408]
[8, 946874, 946858.0, 21347]
[2, 946874, 946862.0, 3991]
[7, 946874, 946865.0, 28407]
[1, 946890, 946867.0, 28374]
[2, 946890, 946870.0, 3995]
[2, 946890, 946880.0, 3999]
[2, 946906, 946889.0, 4003]
[6, 946906, 946890.0, 28409]
[7, 946906, 946898.0, 28408]
[2, 946921, 946899.0, 4007]
[1, 946921, 946901.0, 28375]
[8, 946921, 946902.0, 21348]
[2, 946921, 946908.0, 4011]
[2, 946937, 946918.0, 4015]
[6, 946937, 946924.0, 28410]
[2, 946952, 946928.0, 4019]
[7, 946952, 946931.0, 28409]
[1, 946952, 946934.0, 28376]
[2, 946952, 946937.0, 4023]
[8, 946968, 946946.0, 21349]
[2, 946968, 946947.0, 4027]
[2, 946968, 946956.0, 4031]
[6, 946968, 946957.0, 28411]
[2, 946984, 946964.0, 4035]
[7, 946984, 946965.0, 28410]
[1, 946984, 946967.0, 28377]
[2, 946984, 946973.0, 4039]
[2, 946999, 946981.0, 4043]
[2, 946999, 946990.0, 4047]
[6, 947015, 946990.0, 28412]
[8, 947015, 946991.0, 21350]
[7, 947015, 946998.0, 28411]
[2, 947015, 946999.0, 4051]
[1, 947015, 947001.0, 28378]
[2, 947030, 947007.0, 4055]
[2, 947030, 947016.0, 4059]
[6, 947046, 947024.0, 28413]
[2, 947046, 947024.0, 4063]
[7, 947046, 947031.0, 28412]
[2, 947046, 947033.0, 4067]
[1, 947046, 947034.0, 28379]
[8, 947046, 947035.0, 21351]
[2, 947062, 947041.0, 4071]
[2, 947062, 947049.0, 4075]
[6, 947077, 947057.0, 28414]
[2, 947077, 947057.0, 4079]
[7, 947077, 947064.0, 28413]
[2, 947077, 947065.0, 4083]
[1, 947077, 947067.0, 28380]
[2, 947093, 947073.0, 4087]
[8, 947093, 947080.0, 21352]
[2, 947093, 947081.0, 4091]
[2, 947108, 947090.0, 4095]
[6, 947108, 947090.0, 28415]
[7, 947108, 947098.0, 28414]
[2, 947124, 947098.0, 4099]
[1, 947124, 947101.0, 28381]
[2, 947124, 947107.0, 4103]
[2, 947140, 947116.0, 4107]
[6, 947140, 947124.0, 28416]
[8, 947140, 947124.0, 21353]
[2, 947140, 947125.0, 4111]
[7, 947140, 947131.0, 28415]
[1, 947155, 947134.0, 28382]
[2, 947155, 947134.0, 4115]
[2, 947155, 947144.0, 4119]
[2, 947171, 947152.0, 4123]
[6, 947171, 947157.0, 28417]
[2, 947186, 947161.0, 4127]
[7, 947186, 947165.0, 28416]
[1, 947186, 947167.0, 28383]
[8, 947186, 947168.0, 21354]
[2, 947186, 947170.0, 4131]
[2, 947202, 947178.0, 4135]
[2, 947202, 947187.0, 4139]
[6, 947202, 947190.0, 28418]
[2, 947218, 947195.0, 4143]
[7, 947218, 947198.0, 28417]
[1, 947218, 947201.0, 28384]
[2, 947218, 947203.0, 4147]
[2, 947233, 947212.0, 4151]
[8, 947233, 947213.0, 21355]
[2, 947233, 947220.0, 4155]
[6, 947233, 947224.0, 28419]
[2, 947249, 947229.0, 4159]
[7, 947249, 947231.0, 28418]
[1, 947249, 947234.0, 28385]
[2, 947249, 947239.0, 4163]
[2, 947264, 947248.0, 4167]
[6, 947280, 947257.0, 28420]
[8, 947280, 947257.0, 21356]
[2, 947280, 947258.0, 4171]
[2, 947280, 947264.0, 4173]
[7, 947280, 947265.0, 28419]
[1, 947280, 947268.0, 28386]
[2, 947296, 947274.0, 4177]
[2, 947296, 947284.0, 4181]
[2, 947311, 947290.0, 4183]
[6, 947311, 947290.0, 28421]
[7, 947311, 947298.0, 28420]
[2, 947311, 947301.0, 4187]
[1, 947311, 947301.0, 28387]
[8, 947327, 947301.0, 21357]
[2, 947327, 947311.0, 4191]
[2, 947327, 947317.0, 4193]
[2, 947342, 947323.0, 4195]
[6, 947342, 947324.0, 28422]
[2, 947342, 947329.0, 4197]
[7, 947342, 947331.0, 28421]
[1, 947358, 947334.0, 28388]
[2, 947358, 947335.0, 4199]
[8, 947358, 947346.0, 21358]
[2, 947358, 947346.0, 4203]
[2, 947374, 947352.0, 4205]
[6, 947374, 947357.0, 28423]
[2, 947374, 947363.0, 4209]
[7, 947374, 947364.0, 28422]
[1, 947389, 947368.0, 28389]
[2, 947389, 947373.0, 4213]
[2, 947389, 947379.0, 4215]
[2, 947405, 947388.0, 4219]
[8, 947405, 947390.0, 21359]
[6, 947405, 947390.0, 28424]
[7, 947420, 947398.0, 28423]
[2, 947420, 947398.0, 4223]
[1, 947420, 947401.0, 28390]
[2, 947420, 947407.0, 4227]
[2, 947436, 947416.0, 4231]
[6, 947436, 947424.0, 28425]
[2, 947436, 947425.0, 4235]
[7, 947452, 947431.0, 28424]
[2, 947452, 947434.0, 4239]
[1, 947452, 947434.0, 28391]
[8, 947452, 947434.0, 21360]
[2, 947467, 947443.0, 4243]
[2, 947467, 947452.0, 4247]
[6, 947467, 947457.0, 28426]
[2, 947483, 947460.0, 4251]
[7, 947483, 947464.0, 28425]
[1, 947483, 947468.0, 28392]
[2, 947483, 947468.0, 4255]
[2, 947498, 947477.0, 4259]
[8, 947498, 947479.0, 21361]
[2, 947498, 947485.0, 4263]
[6, 947514, 947490.0, 28427]
[2, 947514, 947493.0, 4267]
[7, 947514, 947498.0, 28426]
[1, 947514, 947501.0, 28393]
[2, 947514, 947502.0, 4271]
[2, 947530, 947511.0, 4275]
[2, 947530, 947520.0, 4279]
[8, 947545, 947523.0, 21362]
[6, 947545, 947524.0, 28428]
[2, 947545, 947529.0, 4283]
[7, 947545, 947531.0, 28427]
[1, 947545, 947534.0, 28394]
[2, 947561, 947539.0, 4287]
[2, 947561, 947548.0, 4291]
[6, 947576, 947557.0, 28429]
[2, 947576, 947558.0, 4295]
[7, 947576, 947564.0, 28428]
[2, 947592, 947567.0, 4299]
[8, 947592, 947567.0, 21363]
[1, 947592, 947568.0, 28395]
[2, 947592, 947576.0, 4303]
[2, 947608, 947586.0, 4307]
[6, 947608, 947590.0, 28430]
[2, 947608, 947595.0, 4311]
[7, 947608, 947598.0, 28429]
[1, 947623, 947601.0, 28396]
[2, 947623, 947604.0, 4315]
[8, 947623, 947612.0, 21364]
[2, 947623, 947613.0, 4319]
[2, 947639, 947621.0, 4323]
[6, 947639, 947624.0, 28431]
[2, 947639, 947630.0, 4327]
[7, 947654, 947631.0, 28430]
[1, 947654, 947634.0, 28397]
[2, 947654, 947639.0, 4331]
[2, 947654, 947647.0, 4335]
[8, 947670, 947656.0, 21365]
[2, 947670, 947656.0, 4339]
[6, 947670, 947657.0, 28432]
[7, 947686, 947665.0, 28431]
[2, 947686, 947665.0, 4343]
[1, 947686, 947668.0, 28398]
[2, 947686, 947675.0, 4347]
[2, 947701, 947685.0, 4351]
[6, 947701, 947690.0, 28433]
[2, 947717, 947695.0, 4355]
[7, 947717, 947698.0, 28432]
[8, 947717, 947701.0, 21366]
[1, 947717, 947701.0, 28399]
[2, 947717, 947705.0, 4359]
[2, 947732, 947711.0, 4361]
[2, 947732, 947721.0, 4365]
[6, 947732, 947724.0, 28434]
[2, 947748, 947727.0, 4367]
[7, 947748, 947731.0, 28433]
[1, 947748, 947734.0, 28400]
[2, 947748, 947737.0, 4371]
[8, 947764, 947745.0, 21367]
[2, 947764, 947747.0, 4375]
[2, 947779, 947757.0, 4379]
[6, 947779, 947757.0, 28435]
[7, 947779, 947765.0, 28434]
[2, 947779, 947766.0, 4383]
[1, 947779, 947768.0, 28401]
[2, 947795, 947775.0, 4387]
[2, 947795, 947784.0, 4391]
[8, 947810, 947789.0, 21368]
[6, 947810, 947790.0, 28436]
[2, 947810, 947793.0, 4395]
[7, 947810, 947798.0, 28435]
[1, 947810, 947801.0, 28402]
[2, 947810, 947803.0, 4399]
[2, 947826, 947812.0, 4403]
[2, 947842, 947821.0, 4407]
[6, 947842, 947824.0, 28437]
[2, 947842, 947831.0, 4411]
[7, 947842, 947831.0, 28436]
[8, 947857, 947834.0, 21369]
[1, 947857, 947835.0, 28403]
[2, 947857, 947840.0, 4415]
[2, 947873, 947849.0, 4419]
[6, 947873, 947857.0, 28438]
[2, 947873, 947857.0, 4423]
[0, 947873, 947863.0, 9]
[7, 947873, 947864.0, 28437]
[2, 947888, 947866.0, 4427]
[1, 947888, 947868.0, 28404]
[2, 947888, 947875.0, 4431]
[8, 947888, 947878.0, 21370]
[2, 947904, 947883.0, 4435]
[6, 947904, 947890.0, 28439]
[2, 947904, 947892.0, 4439]
[7, 947920, 947898.0, 28438]
[2, 947920, 947901.0, 4443]
[1, 947920, 947901.0, 28405]
[2, 947920, 947909.0, 4447]
[2, 947935, 947918.0, 4451]
[8, 947935, 947922.0, 21371]
[6, 947935, 947923.0, 28440]
[2, 947935, 947926.0, 4455]
[7, 947951, 947931.0, 28439]
[2, 947951, 947934.0, 4459]
[1, 947951, 947935.0, 28406]
[2, 947951, 947942.0, 4463]
[2, 947966, 947950.0, 4467]
[6, 947966, 947957.0, 28441]
[2, 947966, 947958.0, 4471]
[7, 947982, 947964.0, 28440]
[2, 947982, 947966.0, 4475]
[8, 947982, 947967.0, 21372]
[1, 947982, 947968.0, 28407]
[2, 947982, 947974.0, 4479]
[2, 947998, 947983.0, 4483]
[6, 948013, 947990.0, 28442]
[2, 948013, 947991.0, 4487]
[7, 948013, 947998.0, 28441]
[2, 948013, 948000.0, 4491]
[1, 948013, 948001.0, 28408]
[2, 948029, 948009.0, 4495]
[8, 948029, 948011.0, 21373]
[2, 948029, 948018.0, 4499]
[6, 948044, 948023.0, 28443]
[2, 948044, 948027.0, 4503]
[7, 948044, 948031.0, 28442]
[1, 948044, 948035.0, 28409]
[2, 948060, 948038.0, 4507]
[2, 948060, 948044.0, 4509]
[2, 948060, 948050.0, 4511]
[8, 948076, 948055.0, 21374]
[6, 948076, 948057.0, 28444]
[2, 948076, 948057.0, 4513]
[2, 948076, 948064.0, 4515]
[7, 948091, 948064.0, 28443]
[1, 948091, 948068.0, 28410]
[2, 948091, 948072.0, 4517]
[2, 948091, 948081.0, 4519]
[6, 948107, 948090.0, 28445]
[2, 948107, 948090.0, 4521]
[7, 948107, 948098.0, 28444]
[2, 948122, 948100.0, 4523]
[8, 948122, 948100.0, 21375]
[1, 948122, 948101.0, 28411]
[2, 948122, 948109.0, 4525]
[2, 948138, 948120.0, 4527]
[6, 948138, 948123.0, 28446]
[7, 948154, 948131.0, 28445]
[2, 948154, 948132.0, 4529]
[1, 948154, 948135.0, 28412]
[8, 948169, 948144.0, 21376]
[2, 948169, 948148.0, 4531]
[6, 948169, 948157.0, 28447]
[7, 948185, 948164.0, 28446]
[2, 948185, 948167.0, 4533]
[1, 948185, 948168.0, 28413]
[2, 948200, 948188.0, 4535]
[8, 948200, 948188.0, 21377]
[6, 948200, 948190.0, 28448]
[7, 948216, 948198.0, 28447]
[1, 948216, 948201.0, 28414]
[2, 948216, 948208.0, 4537]
[6, 948247, 948223.0, 28449]
[7, 948247, 948231.0, 28448]
[8, 948247, 948233.0, 21378]
[1, 948247, 948235.0, 28415]
[2, 948263, 948242.0, 4539]
[6, 948278, 948257.0, 28450]
[7, 948278, 948264.0, 28449]
[1, 948278, 948268.0, 28416]
[8, 948294, 948277.0, 21379]
[6, 948310, 948290.0, 28451]
[7, 948310, 948298.0, 28450]
[1, 948325, 948301.0, 28417]
[8, 948341, 948322.0, 21380]
[6, 948341, 948323.0, 28452]
[7, 948341, 948331.0, 28451]
[1, 948356, 948335.0, 28418]
[2, 948356, 948339.0, 4541]
[6, 948372, 948357.0, 28453]
[7, 948388, 948364.0, 28452]
[8, 948388, 948366.0, 21381]
[1, 948388, 948368.0, 28419]
[2, 948388, 948369.0, 4543]
[6, 948403, 948390.0, 28454]
[2, 948419, 948393.0, 4545]
[7, 948419, 948398.0, 28453]
[1, 948419, 948402.0, 28420]
[8, 948434, 948410.0, 21382]
[2, 948434, 948416.0, 4547]
[6, 948434, 948423.0, 28455]
[7, 948450, 948431.0, 28454]
[1, 948450, 948435.0, 28421]
[2, 948450, 948438.0, 4549]
[8, 948466, 948455.0, 21383]
[6, 948466, 948457.0, 28456]
[2, 948481, 948459.0, 4551]
[7, 948481, 948464.0, 28455]
[1, 948481, 948468.0, 28422]
[2, 948497, 948481.0, 4553]
[6, 948512, 948490.0, 28457]
[7, 948512, 948498.0, 28456]
[8, 948512, 948499.0, 21384]
[1, 948512, 948502.0, 28423]
[2, 948528, 948505.0, 4555]
[6, 948544, 948523.0, 28458]
[2, 948544, 948527.0, 4557]
[7, 948544, 948531.0, 28457]
[1, 948544, 948535.0, 28424]
[8, 948559, 948543.0, 21385]
[2, 948575, 948549.0, 4559]
[5, 948575, 948554.0, 74]
[4, 948575, 948554.0, 29.0]
[6, 948575, 948557.0, 28459]
[7, 948575, 948564.0, 28458]
[1, 948590, 948568.0, 28425]
[2, 948590, 948575.0, 2]
[8, 948606, 948588.0, 21386]
[6, 948606, 948590.0, 28460]
[7, 948606, 948598.0, 28459]
[2, 948622, 948599.0, 4]
[1, 948622, 948602.0, 28426]
[2, 948637, 948618.0, 6]
[6, 948637, 948623.0, 28461]
[7, 948653, 948631.0, 28460]
[8, 948653, 948632.0, 21387]
[2, 948653, 948634.0, 8]
[1, 948653, 948635.0, 28427]
[2, 948668, 948649.0, 10]
[6, 948668, 948657.0, 28462]
[2, 948684, 948664.0, 12]
[7, 948684, 948664.0, 28461]
[3, 948684, 948665.0, 550]
[1, 948684, 948668.0, 28428]
[8, 948700, 948676.0, 21388]
[2, 948700, 948679.0, 14]
[6, 948700, 948690.0, 28463]
[2, 948700, 948692.0, 16]
[7, 948715, 948698.0, 28462]
[1, 948715, 948702.0, 28429]
[2, 948715, 948705.0, 18]
[2, 948731, 948717.0, 20]
[8, 948746, 948721.0, 21389]
[6, 948746, 948723.0, 28464]
[2, 948746, 948728.0, 22]
[7, 948746, 948731.0, 28463]
[1, 948746, 948735.0, 28430]
[2, 948762, 948738.0, 24]
[2, 948762, 948749.0, 26]
[6, 948778, 948757.0, 28465]
[2, 948778, 948760.0, 28]
[7, 948778, 948764.0, 28464]
[8, 948778, 948765.0, 21390]
[1, 948778, 948768.0, 28431]
[2, 948793, 948771.0, 30]
[2, 948793, 948782.0, 32]
[6, 948809, 948790.0, 28466]
[2, 948809, 948793.0, 34]
# STOPPED EXPERIMENT"""

    FILEPATH: ClassVar[Path] = Path('.test.riglog')
    MISSING = np.array([])
    RIG: RiglogData

    @classmethod
    def setUpClass(cls):
        with open(cls.FILEPATH, 'w') as f:
            f.write(cls.RIGLOG_CONTENT)

        cls.RIG = RiglogData(cls.FILEPATH)

    @classmethod
    def tearDownClass(cls):
        cls.FILEPATH.unlink()
        cls.FILEPATH.with_stem('.test_riglog').with_suffix('.npy').unlink()

    def test_source_version(self):
        self.assertEqual(self.RIG.log_config['source_version'], 'stimpy-git')

    def test_version(self):
        self.assertNotIn('version', self.RIG.log_config)

    def test_commit_hash(self):
        self.assertEqual(self.RIG.log_config['commit_hash'], '88c4705')

    def test_codes(self):
        res = {
            'screen': 0,
            'imaging': 1,
            'encoder': 2,
            'licks': 3,
            'button': 4,
            'reward': 5,
            'laps': 6,
            'cam1': 7,
            'cam2': 8,
            'cam3': 9,
            'act0': 10,
            'act1': 11,
            'opto': 12
        }
        self.assertDictEqual(self.RIG.log_config['codes'], res)

    def test_csv_fields(self):
        res = ('code', 'time received', 'duino time', 'value')
        self.assertTupleEqual(self.RIG.log_config['fields'], res)


if __name__ == '__main__':
    unittest.main()
