from typing import Optional

import enum
import numpy as np
import pandas as pd
import scipy.stats as ss
from abc import abstractmethod


class Type(enum.Enum):
    STRAIGHT_LINE = "StraightLine"  # A constant-change (linearly growing (or decaying)) projection form, originating from an initial value.
    COMPOUNDING = "Compounding"  # An exponentially growing (compounding) or decaying projection at a specified rate per period.
    RECURRING = (
        "Recurring"  # A projection form that repeats a specified sequence of values.
    )
    DYNAMIC = "Dynamic"  # A projection form generated by a stochastic process.


class Form:
    type: Type
    """
    A specified methodology, logic, or algorithm for extrapolating a value to a sequence.
    """

    @abstractmethod
    def terms(self, sequence: pd.RangeIndex) -> [float]:
        """
        Returns the set of terms that define the projection's form at each value in the sequence.
        """
        pass


class StraightLine(Form):
    slope: float
    """
    A constant-change (linearly growing (or decaying)) projection form, originating from an initial value.
    To calculate the factor, the projection is modelled as a linear function of (slope * period) + 1
    """

    def __init__(self, slope: float):
        self.type = Type.STRAIGHT_LINE
        self.slope = slope

    def terms(self, sequence: pd.RangeIndex) -> [float]:
        """
        Returns the additive terms to the projection at each value in the sequence.
        """
        return [(self.slope * index) for index in sequence]


class Recurring(StraightLine):
    def __init__(self):
        super().__init__(slope=0)
        self.type = Type.RECURRING


class Compounding(Form):
    rate: float
    """
    An exponentially growing (compounding) or decaying projection at a specified rate per period.
    To calculate the factor, the projection is modelled as an exponential function of (1 + rate) ** period
    """

    def __init__(self, rate: float):
        self.type = Type.COMPOUNDING
        self.rate = rate

    def terms(self, sequence: pd.RangeIndex) -> [float]:
        """
        Returns the multiplicative factors of the projection's form at each value in the sequence.
        """
        return [np.power((1 + self.rate), index) for index in sequence]


class Dynamic(Form):
    series: pd.Series

    def __init__(self, series: pd.Series):
        self.type = Type.DYNAMIC
        self.series = series

    def terms(self, sequence: pd.RangeIndex) -> [float]:
        """
        Returns the multiplicative factors of the projection's form at each value in the sequence.
        """
        if self.series.index.size != sequence.size:
            raise ValueError("Error: series and sequence must match in size")
        return list(self.series.values)
