import os
import re
from lxml import etree
from pathlib import Path
from dataclasses import dataclass, field
from xml.sax.saxutils import escape, unescape
from typing import List, Dict, Any, Tuple, Optional, Callable

from ...lean import Pos
from .node import NODES_WITH_EXPRS
from ....utils import camel_case, is_optional_type, remove_optional_type, parse_int_list


@dataclass
class Expr:
    """
    Base class for Lean expressions.
    """

    tags: str
    _id: int = field(init=False, repr=False, compare=False)

    @classmethod
    def from_data(cls, data: Dict[str, Any], idx: int, tags: str) -> "Expr":
        """
        Build an expr from traced data generated by `lean --ast`.
        """
        assert len(data["expr"][idx]) == 1
        key = next(iter(data["expr"][idx]))
        cls_name = camel_case(key) + "Expr"
        subcls = globals()[cls_name]
        expr: Expr = subcls.from_data(data, data["expr"][idx][key], tags)
        expr._id = idx
        return expr

    @property
    def children(self) -> List[Tuple[str, "Expr"]]:
        """
        Return a list of children exprs.
        """
        all_children = []
        for field in self.__dataclass_fields__.values():
            if isinstance(field.type, type) and issubclass(field.type, Expr):
                value = getattr(self, field.name)
                all_children.append((field.name, value))
            elif field.type == List[Expr]:
                value = getattr(self, field.name)
                for i, v in enumerate(value):
                    all_children.append((f"{field.name}[{i}]", v))
        return all_children

    def traverse_preorder(self, callback: Callable[["Expr"], None]) -> None:
        """
        Traverse the expr tree in preorder.
        """
        callback(self)
        for _, child in self.children:
            child.traverse_preorder(callback)

    def to_xml(self, parent: etree.Element) -> None:
        """
        Serialize an expr to XML.
        """
        tree = etree.SubElement(parent, self.__class__.__name__)

        for field in self.__dataclass_fields__.values():
            if field.name == "_id":
                continue
            v = getattr(self, field.name)
            # Non-expr fields are saved as XML attributes.
            if v is not None and field.type not in (Expr, List[Expr]):
                v = escape(str(v), entities={'"': "&quot;"})
                tree.set(field.name, v)

        # Expr fields are saved as XML children.
        for _, child in self.children:
            child.to_xml(tree)

    @classmethod
    def from_xml(cls, tree: etree.Element) -> "Expr":
        """
        Deserialize an expr from XML.
        """
        subcls = globals()[tree.tag]
        tags = tree.attrib["tags"]
        kwargs: Dict[str, Any] = {}
        i = 0

        for field in subcls.__dataclass_fields__.values():
            if field.name == "tags" or field.name == "_id":
                continue
            if field.type is Expr:
                kwargs[field.name] = Expr.from_xml(tree[i])
                i += 1
            elif field.type is List[Expr]:
                child_exprs = []
                n = len(tree)
                while i < n and re.search(
                    f"\\.{field.name}\\[\\d+\\]$", tree[i].attrib["tags"]
                ):
                    child_exprs.append(Expr.from_xml(tree[i]))
                    i += 1
                kwargs[field.name] = child_exprs
            elif is_optional_type(field.type):
                tp = remove_optional_type(field.type)
                v = tree.attrib.get(field.name, None)
                if tp is Pos and v is not None:
                    kwargs[field.name] = Pos.from_str(v)
                else:
                    kwargs[field.name] = v
            elif field.type is List[int]:
                kwargs[field.name] = parse_int_list(tree.attrib[field.name])
            else:
                kwargs[field.name] = field.type(tree.attrib[field.name])  # type: ignore
            if isinstance(kwargs[field.name], str):
                kwargs[field.name] = unescape(
                    kwargs[field.name], entities={"&quot;": '"'}
                )
        return subcls(tags, **kwargs)  # type: ignore


@dataclass
class VarExpr(Expr):
    """
    A bound variable with a de-Bruijn index.
    """

    index: int

    @classmethod
    def from_data(cls, data: Dict[str, Any], i: int, tags: str) -> "VarExpr":
        return cls(tags, i)


@dataclass
class SorryExpr(Expr):
    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "SorryExpr":
        return cls(tags)


@dataclass
class AcAppExpr(Expr):
    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "AcAppExpr":
        return cls(tags)


@dataclass
class PermAcExpr(Expr):
    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "PermAcExpr":
        return cls(tags)


@dataclass
class SortExpr(Expr):
    """
    A type universe: `Sort u`
    """

    level: int

    @classmethod
    def from_data(cls, data: Dict[str, Any], l: int, tags: str) -> "SortExpr":
        return cls(tags, l)


@dataclass
class ConstExpr(Expr):
    """
    A global constant. These include definitions, constants and inductive type stuff present
    in the environment as well as hard-coded definitions.
    """

    full_name: str
    levels: List[int]
    def_path: Optional[str] = None
    def_pos: Optional[Pos] = None

    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "ConstExpr":
        if isinstance(v[0], str):
            return cls(tags, v[0], v[1])
        else:
            assert isinstance(v[0], list)
            return cls(tags, ".".join([str(_) for _ in v[0]]), v[1])


@dataclass
class MvarExpr(Expr):
    "An `mvar` is a 'hole' yet to be filled in by the elaborator or tactic state."
    name: str
    type: Expr

    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "MvarExpr":
        n = ".".join(str(_) for _ in v["pp"])
        t = Expr.from_data(data, v["type"], tags + ".type")
        return cls(tags, n, t)


@dataclass
class LocalExpr(Expr):
    """
    A local constant. For example, if our tactic state was `h : P ⊢ Q`, `h` would be a local constant.
    """

    name: str
    type: Expr

    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "LocalExpr":
        t = Expr.from_data(data, v["type"], tags + ".type")
        return cls(tags, v["pp"], t)


@dataclass
class AppExpr(Expr):
    """
    Function application.
    """

    func: Expr
    arg: Expr

    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "AppExpr":
        assert len(v) == 2
        f = Expr.from_data(data, v[0], tags + ".func")
        x = Expr.from_data(data, v[1], tags + ".arg")
        return cls(tags, f, x)


@dataclass
class LamExpr(Expr):
    """
    Lambda abstraction. eg ```(λ a : α, x)``
    """

    var_name: str
    var_type: Expr
    body: Expr

    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "LamExpr":
        return cls(
            tags,
            v["name"],
            Expr.from_data(data, v["dom"], tags + ".var_type"),
            Expr.from_data(data, v["body"], tags + ".body"),
        )


@dataclass
class PiExpr(Expr):
    """
    Pi type constructor. eg ```(Π a : α, x)`` and ```(α → β)``
    """

    var_name: str
    var_type: Expr
    body: Expr

    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "PiExpr":
        return cls(
            tags,
            v["name"],
            Expr.from_data(data, v["dom"], tags + ".var_type"),
            Expr.from_data(data, v["body"], tags + ".body"),
        )


@dataclass
class LetExpr(Expr):
    """
    An explicit let binding.
    """

    var_name: str
    var_type: Expr
    var_value: Expr
    body: Expr

    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "LetExpr":
        return cls(
            tags,
            v["name"],
            Expr.from_data(data, v["type"], tags + ".var_type"),
            Expr.from_data(data, v["value"], tags + ".var_value"),
            Expr.from_data(data, v["body"], tags + ".body"),
        )


@dataclass
class FieldNotationExpr(Expr):
    field: str
    args: List[Expr]
    idx: int

    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "FieldNotationExpr":
        args = [
            Expr.from_data(data, a, tags + f".args[{i}]")
            for i, a in enumerate(v["args"])
        ]
        return cls(tags, v["field"], args, v["idx"])


@dataclass
class TypedExprExpr(Expr):
    value: Expr
    type: Expr

    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "TypedExprExpr":
        assert len(v) == 1 and len(v["args"]) == 2
        return cls(
            tags,
            Expr.from_data(data, v["args"][0], tags + ".value"),
            Expr.from_data(data, v["args"][1], tags + ".type"),
        )


@dataclass
class ChoiceExpr(Expr):
    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "ChoiceExpr":
        return cls(tags)


@dataclass
class AnnotationExpr(Expr):
    name: str

    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "AnnotationExpr":
        return cls(tags, v["name"])


@dataclass
class PrenumExpr(Expr):
    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "PrenumExpr":
        return cls(tags)


@dataclass
class ExprQuoteMacroExpr(Expr):
    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "ExprQuoteMacroExpr":
        return cls(tags)


@dataclass
class DelayedAbstractionExpr(Expr):
    @classmethod
    def from_data(
        cls, data: Dict[str, Any], v: Any, tags: str
    ) -> "DelayedAbstractionExpr":
        return cls(tags)


@dataclass
class StringMacroExpr(Expr):
    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "StringMacroExpr":
        return cls(tags)


@dataclass
class StructureInstanceExpr(Expr):
    @classmethod
    def from_data(
        cls, data: Dict[str, Any], v: Any, tags: str
    ) -> "StructureInstanceExpr":
        return cls(tags)


@dataclass
class EquationExpr(Expr):
    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "EquationExpr":
        return cls(tags)


@dataclass
class EquationsExpr(Expr):
    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "EquationsExpr":
        return cls(tags)


@dataclass
class NoEquationExpr(Expr):
    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "NoEquationExpr":
        return cls(tags)


@dataclass
class AsPatternExpr(Expr):
    @classmethod
    def from_data(cls, data: Dict[str, Any], v: Any, tags: str) -> "AsPatternExpr":
        return cls(tags)


def parse_exprs_forest(
    data: Dict[str, Any]
) -> Tuple[List[Expr], Dict[int, Optional[str]]]:
    forest: List[Expr] = []
    tags_map: Dict[int, Any] = {0: None}
    constants = {c["expr"]: c for c in data["constants"]}
    assert data["expr"][0] is None

    is_needed = [False] * len(data["expr"])
    # We only keep exprs that are referenced by ASTs.
    for ast in data["ast"]:
        if ast is None or ast["kind"] not in NODES_WITH_EXPRS:
            continue
        if "expr" in ast:
            is_needed[ast["expr"]] = True

    for i in range(len(data["expr"]) - 1, 0, -1):
        if not is_needed[i]:
            continue
        j = len(forest)
        expr = Expr.from_data(data, i, str(j))

        def _callback(e: Expr) -> None:
            assert e._id <= i
            is_needed[e._id] = False
            tags_map[e._id] = e.tags
            if e._id in constants:
                assert isinstance(e, ConstExpr)
                c = constants[e._id]
                if "lean_path" in c:
                    lean_path = Path(c["lean_path"])
                    idx = lean_path.parts.index("workspace")
                    e.def_path = os.path.sep.join(lean_path.parts[idx + 2 :])
                e.def_pos = Pos(c["pos"][0], 1 + c["pos"][1])
            object.__delattr__(e, "_id")

        expr.traverse_preorder(_callback)
        forest.append(expr)

    return forest, tags_map
