from .base import BaseModel
from typing import Optional
from google import genai

class GoogleGeminiModel(BaseModel):
    def __init__(self, model_name: str, api_key: str):
        super().__init__()
        self.model_name = model_name
        self.client = genai.Client(api_key=api_key)

    def get_response(self, input_data: str, system_prompt: Optional[str] = None) -> str:
        """
        Gets a response from the Google Gemini model.
        
        Args:
            input_data (str): The input data for the model.
            system_prompt (Optional[str]): A system prompt to guide the model's response.
            custom_prompt (Optional[str]): A custom prompt for the model, overwriting the system prompt if applicable.
        If a custom prompt is provided, it will be used instead of the system prompt.
        If no custom prompt is provided, the system prompt will be used.

        Returns:
            str: The response generated by the model.
        """
        try:
            response = self.client.models.generate_content(
                model=self.model_name,
                contents=input_data,
                config=genai.types.GenerateContentConfig(
                    system_instruction=system_prompt if system_prompt else None,
                )
            )
            return response.text if response and hasattr(response, "text") else "No response generated."
        except Exception as e:
            return f"Error: {str(e)}"
    