"""
Julia test script to validate that the serialized PowerSystems.jl system
can be successfully loaded from the JSON file generated by r2x-pypsa.

This test assumes that the Python test has already run and generated:
- test_output.json (main system file)
- time_series_storage.h5 (time series data)  
- time_series_storage_metadata.db (metadata)

Usage:
    cd tests/julia
    julia --project=. test_load_system.jl
"""

using PowerSystems
using Test

function test_load_serialized_system()
    """Test that the serialized system can be loaded successfully."""
    
    # Path to the generated files (relative to project root)
    json_file = "../../test_output.json"
    h5_file = "../../time_series_storage.h5"
    db_file = "../../time_series_storage_metadata.db"
    
    println("Testing PowerSystems.jl system loading...")
    println("Looking for files:")
    println("  JSON: $json_file")
    println("  HDF5: $h5_file") 
    println("  DB:   $db_file")
    
    # Check if required files exist
    if !isfile(json_file)
        error("JSON file not found: $json_file. Please run the Python test first.")
    end
    
    if !isfile(h5_file)
        @warn "HDF5 time series file not found: $h5_file"
    end
    
    if !isfile(db_file)
        @warn "Time series metadata DB not found: $db_file"
    end
    
    @testset "Load PowerSystems.jl System" begin
        # Test that we can load the system without errors
        @test begin
            try
                sys = System(json_file)
                println("✓ Successfully loaded system from $json_file")
                true
            catch e
                println("✗ Failed to load system: $e")
                false
            end
        end
        
        # If loading succeeded, get system information
        if isfile(json_file)
            try
                sys = System(json_file)
                
                # Print basic system information
                println("\nSystem Information:")
                println("  Name: $(get_name(sys))")
                println("  Frequency: $(get_frequency(sys)) Hz")
                println("  Base Power: $(get_base_power(sys)) MVA")
                
                # Count components by type
                buses = get_components(ACBus, sys)
                thermal_gens = get_components(ThermalStandard, sys)
                renewable_gens = get_components(RenewableDispatch, sys)
                hydro_gens = get_components(HydroDispatch, sys)
                loads = get_components(PowerLoad, sys)
                lines = get_components(Line, sys)
                storage = get_components(EnergyReservoirStorage, sys)
                
                println("\nComponent Counts:")
                println("  Buses: $(length(buses))")
                println("  Thermal Generators: $(length(thermal_gens))")
                println("  Renewable Generators: $(length(renewable_gens))")
                println("  Hydro Generators: $(length(hydro_gens))")
                println("  Loads: $(length(loads))")
                println("  Lines: $(length(lines))")
                println("  Storage Units: $(length(storage))")
                
                # Test that we have at least some components
                @test length(buses) > 0 "System should have at least one bus"
                @test (length(thermal_gens) + length(renewable_gens) + length(hydro_gens)) > 0 "System should have at least one generator"
                
                println("\n✓ System validation completed successfully!")
                
            catch e
                println("✗ Error during system validation: $e")
                @test false "System validation failed: $e"
            end
        end
    end
end

# Run the test
if abspath(PROGRAM_FILE) == @__FILE__
    test_load_serialized_system()
end
