"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessPoint = void 0;
const core_1 = require("@aws-cdk/core");
const efs_generated_1 = require("./efs.generated");
class AccessPointBase extends core_1.Resource {
}
/**
 * (experimental) Represents the AccessPoint.
 *
 * @experimental
 */
class AccessPoint extends AccessPointBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new efs_generated_1.CfnAccessPoint(this, 'Resource', {
            fileSystemId: props.fileSystem.fileSystemId,
            rootDirectory: {
                creationInfo: props.createAcl ? {
                    ownerGid: props.createAcl.ownerGid,
                    ownerUid: props.createAcl.ownerUid,
                    permissions: props.createAcl.permissions,
                } : undefined,
                path: props.path,
            },
            posixUser: props.posixUser ? {
                uid: props.posixUser.uid,
                gid: props.posixUser.gid,
                secondaryGids: props.posixUser.secondaryGids,
            } : undefined,
        });
        this.accessPointId = resource.ref;
        this.accessPointArn = core_1.Stack.of(scope).formatArn({
            service: 'elasticfilesystem',
            resource: 'access-point',
            resourceName: this.accessPointId,
        });
        this.fileSystem = props.fileSystem;
    }
    /**
     * (experimental) Import an existing Access Point by attributes.
     *
     * @experimental
     */
    static fromAccessPointAttributes(scope, id, attrs) {
        return new ImportedAccessPoint(scope, id, attrs);
    }
    /**
     * (experimental) Import an existing Access Point by id.
     *
     * @experimental
     */
    static fromAccessPointId(scope, id, accessPointId) {
        return new ImportedAccessPoint(scope, id, {
            accessPointId: accessPointId,
        });
    }
}
exports.AccessPoint = AccessPoint;
class ImportedAccessPoint extends AccessPointBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        if (!attrs.accessPointId) {
            if (!attrs.accessPointArn) {
                throw new Error('One of accessPointId or AccessPointArn is required!');
            }
            this.accessPointArn = attrs.accessPointArn;
            let maybeApId = core_1.Stack.of(scope).parseArn(attrs.accessPointArn).resourceName;
            if (!maybeApId) {
                throw new Error('ARN for AccessPoint must provide the resource name.');
            }
            this.accessPointId = maybeApId;
        }
        else {
            if (attrs.accessPointArn) {
                throw new Error('Only one of accessPointId or AccessPointArn can be provided!');
            }
            this.accessPointId = attrs.accessPointId;
            this.accessPointArn = core_1.Stack.of(scope).formatArn({
                service: 'elasticfilesystem',
                resource: 'access-point',
                resourceName: attrs.accessPointId,
            });
        }
        this._fileSystem = attrs.fileSystem;
    }
    get fileSystem() {
        if (!this._fileSystem) {
            throw new Error("fileSystem is not available when 'fromAccessPointId()' is used. Use 'fromAccessPointAttributes()' instead");
        }
        return this._fileSystem;
    }
}
//# sourceMappingURL=data:application/json;base64,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