"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystem = exports.ThroughputMode = exports.PerformanceMode = exports.LifecyclePolicy = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const access_point_1 = require("./access-point");
const efs_generated_1 = require("./efs.generated");
/**
 * (experimental) EFS Lifecycle Policy, if a file is not accessed for given days, it will move to EFS Infrequent Access.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-lifecyclepolicies
 * @experimental
 */
var LifecyclePolicy;
(function (LifecyclePolicy) {
    LifecyclePolicy["AFTER_7_DAYS"] = "AFTER_7_DAYS";
    LifecyclePolicy["AFTER_14_DAYS"] = "AFTER_14_DAYS";
    LifecyclePolicy["AFTER_30_DAYS"] = "AFTER_30_DAYS";
    LifecyclePolicy["AFTER_60_DAYS"] = "AFTER_60_DAYS";
    LifecyclePolicy["AFTER_90_DAYS"] = "AFTER_90_DAYS";
})(LifecyclePolicy = exports.LifecyclePolicy || (exports.LifecyclePolicy = {}));
/**
 * (experimental) EFS Performance mode.
 *
 * @see https://docs.aws.amazon.com/efs/latest/ug/performance.html#performancemodes
 * @experimental
 */
var PerformanceMode;
(function (PerformanceMode) {
    PerformanceMode["GENERAL_PURPOSE"] = "generalPurpose";
    PerformanceMode["MAX_IO"] = "maxIO";
})(PerformanceMode = exports.PerformanceMode || (exports.PerformanceMode = {}));
/**
 * (experimental) EFS Throughput mode.
 *
 * @see https://docs.aws.amazon.com/efs/latest/ug/performance.html#throughput-modes
 * @experimental
 */
var ThroughputMode;
(function (ThroughputMode) {
    ThroughputMode["BURSTING"] = "bursting";
    ThroughputMode["PROVISIONED"] = "provisioned";
})(ThroughputMode = exports.ThroughputMode || (exports.ThroughputMode = {}));
/**
 * (experimental) The Elastic File System implementation of IFileSystem.
 *
 * It creates a new, empty file system in Amazon Elastic File System (Amazon EFS).
 * It also creates mount target (AWS::EFS::MountTarget) implicitly to mount the
 * EFS file system on an Amazon Elastic Compute Cloud (Amazon EC2) instance or another resource.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html
 * @experimental
 * @resource AWS::EFS::FileSystem
 */
class FileSystem extends core_1.Resource {
    /**
     * (experimental) Constructor for creating a new EFS FileSystem.
     *
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this._mountTargetsAvailable = new core_1.ConcreteDependable();
        if (props.throughputMode === ThroughputMode.PROVISIONED && props.provisionedThroughputPerSecond === undefined) {
            throw new Error('Property provisionedThroughputPerSecond is required when throughputMode is PROVISIONED');
        }
        const filesystem = new efs_generated_1.CfnFileSystem(this, 'Resource', {
            encrypted: props.encrypted,
            kmsKeyId: (_a = props.kmsKey) === null || _a === void 0 ? void 0 : _a.keyArn,
            lifecyclePolicies: (props.lifecyclePolicy ? [{ transitionToIa: props.lifecyclePolicy }] : undefined),
            performanceMode: props.performanceMode,
            throughputMode: props.throughputMode,
            provisionedThroughputInMibps: (_b = props.provisionedThroughputPerSecond) === null || _b === void 0 ? void 0 : _b.toMebibytes(),
            backupPolicy: props.enableAutomaticBackups ? { status: 'ENABLED' } : undefined,
        });
        filesystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = filesystem.ref;
        core_1.Tags.of(this).add('Name', props.fileSystemName || this.node.path);
        const securityGroup = (props.securityGroup || new ec2.SecurityGroup(this, 'EfsSecurityGroup', {
            vpc: props.vpc,
        }));
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        const subnets = props.vpc.selectSubnets(props.vpcSubnets);
        // We now have to create the mount target for each of the mentioned subnet
        let mountTargetCount = 0;
        this.mountTargetsAvailable = [];
        subnets.subnetIds.forEach((subnetId) => {
            const mountTarget = new efs_generated_1.CfnMountTarget(this, 'EfsMountTarget' + (++mountTargetCount), {
                fileSystemId: this.fileSystemId,
                securityGroups: Array.of(securityGroup.securityGroupId),
                subnetId,
            });
            this._mountTargetsAvailable.add(mountTarget);
        });
        this.mountTargetsAvailable = this._mountTargetsAvailable;
    }
    /**
     * (experimental) Import an existing File System from the given properties.
     *
     * @experimental
     */
    static fromFileSystemAttributes(scope, id, attrs) {
        return new ImportedFileSystem(scope, id, attrs);
    }
    /**
     * (experimental) create access point from this filesystem.
     *
     * @experimental
     */
    addAccessPoint(id, accessPointOptions = {}) {
        return new access_point_1.AccessPoint(this, id, {
            fileSystem: this,
            ...accessPointOptions,
        });
    }
}
exports.FileSystem = FileSystem;
/**
 * (experimental) The default port File System listens on.
 *
 * @experimental
 */
FileSystem.DEFAULT_PORT = 2049;
class ImportedFileSystem extends core_1.Resource {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.fileSystemId = attrs.fileSystemId;
        this.connections = new ec2.Connections({
            securityGroups: [attrs.securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        this.mountTargetsAvailable = new core_1.ConcreteDependable();
    }
}
//# sourceMappingURL=data:application/json;base64,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