from dataclasses import dataclass, field
from typing import override
import uuid
from dataclasses_json import dataclass_json
from enum import Enum
from decimal import Decimal

import numpy as np
import logging
logger = logging.getLogger(__name__)


KeyframeValue = object


@dataclass_json
@dataclass
class SpaceVector:
    """A 3-dimensional coordinate structure with clear orientation in cm."""

    lr: float = 0
    """Left-to-Right in cm. Right is positive."""

    bt: float = 0
    """Bottom-to-Top in cm. Upwards is positive."""

    bf: float = 0
    """Back-to-Front in cm. Forwards is positive."""

    @property
    def coordinates(self) -> np.ndarray:
        """Get values as numpy array.  [LR, BT, BF]"""
        return np.array([self.lr, self.bt, self.bf])

    def distance(self, other: "SpaceVector") -> np.floating:
        """Euclidian distance between two SpaceVectors."""
        return np.linalg.norm(self.coordinates - other.coordinates)

    # Arithmetic operations
    def __add__(self, other: "SpaceVector") -> "SpaceVector":
        return SpaceVector(self.lr + other.lr, self.bt + other.bt, self.bf + other.bf)

    def __sub__(self, other: "SpaceVector") -> "SpaceVector":
        return SpaceVector(self.lr - other.lr, self.bt - other.bt, self.bf - other.bf)

    def __mul__(self, scalar: float) -> "SpaceVector":
        return SpaceVector(self.lr * scalar, self.bt * scalar, self.bf * scalar)

    def __truediv__(self, scalar: float) -> "SpaceVector":
        return SpaceVector(self.lr / scalar, self.bt / scalar, self.bf / scalar)

    # In-place arithmetic operations
    def __iadd__(self, other: "SpaceVector") -> "SpaceVector":
        self.lr += other.lr
        self.bt += other.bt
        self.bf += other.bf
        return self

    def __isub__(self, other: "SpaceVector") -> "SpaceVector":
        self.lr -= other.lr
        self.bt -= other.bt
        self.bf -= other.bf
        return self

    def __imul__(self, scalar: float) -> "SpaceVector":
        self.lr *= scalar
        self.bt *= scalar
        self.bf *= scalar
        return self

    def __itruediv__(self, scalar: float) -> "SpaceVector":
        self.lr /= scalar
        self.bt /= scalar
        self.bf /= scalar
        return self

    # Unary operations
    def __neg__(self) -> "SpaceVector":
        return SpaceVector(-self.lr, -self.bt, -self.bf)

    def __pos__(self) -> "SpaceVector":
        return SpaceVector(+self.lr, +self.bt, +self.bf)

    # Comparison operations
    @override
    def __eq__(self, other: object) -> bool:
        if not isinstance(other, SpaceVector):
            return False
        return self.lr == other.lr and self.bt == other.bt and self.bf == other.bf

    @override
    def __ne__(self, other: object) -> bool:
        return not (self == other)

    # String representation
    @override
    def __repr__(self) -> str:
        return f"SpaceVector(lr={self.lr}, bt={self.bt}, bf={self.bf})"

    @override
    def __str__(self) -> str:
        return f"({self.lr}, {self.bt}, {self.bf})"


class Capabilities(Enum):
    SWITCH = "switch"
    BRIGHTNESS = "brightness"
    TEMPERATURE = "temperature"
    COLOR = "color"


class Parameter(Enum):
    """Possible keyframe/fixture parameters."""

    ON = "on"
    HUE = "hue"
    BRIGHTNESS = "brightness"
    SATURATION = "saturation"
    TEMPERATURE = "temperature"
    TOPIC = "topic"
    TRANSITION = "transition"


@dataclass_json
@dataclass
class Fixture:
    """Definition of a fixture that generates light, its capabilities and positioning."""

    id: str
    """Unique identifier for fixture."""

    name: str
    """Human readable name."""

    api: str
    """API this fixture has to be used with."""

    api_id: str | int
    """Custom id for specific API."""

    capabilities: Capabilities
    """Which dynamic capabilities does the fixture have."""

    position: SpaceVector
    """Position of the fixture in the space in cm."""

    target: SpaceVector
    """Position of the light generated by the ficture in the space in cm."""

    spread: SpaceVector
    """Spread of the light in the space at the target in cm."""

    exposure: float
    """In range [0,1], how exposed is the light source of the fixture. 0: no exposure. 1: full exposure."""


@dataclass
class EngineFixture:
    """A fixture for engine use during rendering."""

    fixture: Fixture
    """Reference meta data for fixture."""

    on: bool = True
    """On state."""

    brightness: float = 0
    """Brightness state. [0, 1]"""

    hue: float = 0
    """Hue state. [0, 1]"""

    saturation: float = 0
    """Saturation state. [0, 1]"""

    temperature: float = 0
    """Temperature state, in Kelvin."""

    transition: float = 0.1
    """Transition time in seconds."""

    topic: str | None = None
    """Topic fixture currently displays."""

    changed: bool = False
    """True, if fixture was changed after last update. Changes only detected through 'set' method!"""

    def set(self, parameter: Parameter, value: KeyframeValue):
        """Set value of fixture based on parameter. Set 'changed' flag to true, if value was different from previous."""
        match parameter:
            case Parameter.ON:
                if self.on == value:
                    return
                self.on = value  # type: ignore  # pyright: ignore[reportAttributeAccessIssue]
            case Parameter.HUE:
                if self.hue == value:
                    return
                self.hue = value  # type: ignore  # pyright: ignore[reportAttributeAccessIssue]
            case Parameter.BRIGHTNESS:
                if self.brightness == value:
                    return
                self.brightness = value  # type: ignore  # pyright: ignore[reportAttributeAccessIssue]
            case Parameter.SATURATION:
                if self.saturation == value:
                    return
                self.saturation = value  # type: ignore  # pyright: ignore[reportAttributeAccessIssue]
            case Parameter.TEMPERATURE:
                if self.temperature == value:
                    return
                self.temperature = value  # type: ignore  # pyright: ignore[reportAttributeAccessIssue]
            case Parameter.TOPIC:
                if self.topic == value:
                    return
                self.topic = value  # type: ignore  # pyright: ignore[reportAttributeAccessIssue]
            case _:
                logger.warning(
                    f"Unknown parameter [{parameter}] with value [{value}] for fixture id [{self.fixture.id}]."
                )
                return

        self.changed = True

    def __str__(self) -> str:
        """Human-readable string representation."""
        return (
            f"EngineFixture[{self.fixture.name} (id={self.fixture.id})]: "
            f"{'ON' if self.on else 'OFF'}, "
            f"brightness={self.brightness:.2f}, hue={self.hue:.2f}, "
            f"saturation={self.saturation:.2f}, temperature={self.temperature:.1f}K, "
            f"topic={self.topic or 'None'}"
        )

    def __repr__(self) -> str:
        """Unambiguous representation for debugging."""
        return (
            f"EngineFixture("
            f"fixture=Fixture(id={self.fixture.id!r}, name={self.fixture.name!r}), "
            f"on={self.on!r}, brightness={self.brightness!r}, hue={self.hue!r}, "
            f"saturation={self.saturation!r}, temperature={self.temperature!r}, "
            f"transition={self.transition!r}, topic={self.topic!r})"
        )


@dataclass_json
@dataclass
class Space:
    """A space containing fixtures."""

    id: str
    """Unique identifier for space."""

    name: str
    """Human readable name."""

    fixtures: list[Fixture]
    """All fixtures present in the space."""

    version: str = "1"
    """Metatag to track version of datastructure."""

    def save(self, path: str, overwrite: bool = False) -> int:
        """Write space data to file."""
        with open(path, "w+" if overwrite else "x", encoding="UTF-8") as fp:
            return fp.write(self.to_json())  # type: ignore

    @staticmethod
    def load(path: str) -> "Space":
        """Write soace data to file."""
        with open(path, "r", encoding="UTF-8") as fp:
            json = "\n".join(fp.readlines())

        return Space.schema().loads(json)  # type: ignore


class Interpolation(Enum):
    STEP_PRE = "step-pre"
    """Keep value constant after keyframe, and jump to new value before the next keyframe. Required for bool and str value types."""
    STEP_POST = "step-post"
    """Keep value constant after keyframe, and jump to new value after the current keyframe. Required for bool and str value types."""

    LINEAR = "linear"
    EASE_IN_OUT = "ease-in-out"
    EASE_IN = "ease-in"
    EASE_OUT = "ease-out"
    EXPONENTIAL = "exponential"
    LOGARITHMIC = "logarithmic"
    SINE = "sine"
    SMOOTHSTEP = "smoothstep"
    BOUNCE = "bounce"
    ELASTIC = "elastic"

    @staticmethod
    def reverse(value: "Interpolation") -> "Interpolation":
        match value:
            case Interpolation.STEP_POST:
                return Interpolation.STEP_PRE
            case Interpolation.STEP_PRE:
                return Interpolation.STEP_POST
            case _:
                return value


@dataclass_json
@dataclass
class Keyframe:
    """State description for a specified point in time."""

    time: Decimal
    """Point of time of the keyframe. Stored as decimal for high precision."""

    value: KeyframeValue = None
    """Targeted state value reached by reaching the keyframe."""

    interpolation: Interpolation = Interpolation.STEP_PRE
    """Interpolation of value following the keyframe."""


@dataclass
class EngineKeyframe:
    """A keyframe for engine use during rendering."""

    frame: Keyframe
    """Reference keyframe."""

    delta: Decimal
    """Remaining time in seconds relative to previous keyframes."""


@dataclass_json
@dataclass
class Track:
    """Contains a sequence of keyframes for specified fixtures and parameter."""

    id: str
    """Unique identifier for track."""

    name: str
    """Human readable name."""

    parameter: Parameter
    """Parameter targeted by frame value."""

    fixture_ids: list[str]
    """Fixtures by ids targeted by track."""

    keyframes: list[Keyframe] = field(default_factory=list)
    """Sequence of keyframes describing actions"""


@dataclass
class EngineTrack:
    """A track for engine use during rendering."""

    track: Track
    """Reference track."""

    keyframes: list[EngineKeyframe] = field(default_factory=list)
    """Relative engine keyframes of track."""

    current: EngineKeyframe | None = None
    """Keyframe before the next keyframe. Trigger time has been reached."""

    @property
    def next(self) -> EngineKeyframe | None:
        """Next keyframe in track, or None if empty."""
        if len(self.keyframes) == 0:
            return None
        return self.keyframes[0]

    @property
    def is_running(self) -> bool:
        """Indicates wether end of track has been reached."""
        return self.next is not None

    def __post_init__(self):
        # Calculate relative keyframes
        sorted_keyframes = sorted(self.track.keyframes, key=lambda k: k.time)
        # Make keyframe time relative for engine use
        for previous, current in zip([None, *sorted_keyframes[:-1]], sorted_keyframes):
            delta = current.time
            if previous:
                delta -= previous.time
            self.keyframes.append(EngineKeyframe(current, delta))

        # Initialize first frame
        if self.keyframes[0].delta <= 0:
            self.current = self.keyframes.pop(0)

    def step(self, step_size: Decimal):
        """Step keyframes forward by given step."""
        while self.next and step_size > 0:
            self.next.delta -= step_size
            if self.current:
                self.current.delta -= step_size

            if self.next.delta > 0:
                return

            self.current = self.keyframes.pop(0)
            step_size = self.current.delta.copy_abs()


@dataclass_json
@dataclass
class Sequence:
    """A sequence of light events described over fixtures."""

    id: str
    """A unique identifier for this sequence."""

    name: str
    """Human readable name."""

    tracks: list[Track] = field(default_factory=list)
    """List of tracks associated with sequence."""

    version: str = "1"
    """Metatag to track version of datastructure."""

    def ensure_track(
        self, fixture_id: str, parameter: Parameter = Parameter.BRIGHTNESS
    ) -> Track:
        """Ensures a track for a fixture for a given parameter exists, and returns it. If it does not exist, it creates one."""
        t = next(
            filter(
                lambda t: fixture_id in t.fixture_ids and t.parameter == parameter,
                self.tracks,
            ),
            None,
        )
        if t is None:
            # Create new track
            t = Track(
                f"{fixture_id}-{parameter}-{uuid.uuid4()}",
                f"{fixture_id}-{parameter}",
                parameter,
                [fixture_id],
            )
            self.tracks.append(t)
        return t


@dataclass_json
@dataclass
class Flacker:
    """A collection of sequences and spaces."""

    spaces: list[Space]
    """A collection of spaces."""

    sequences: list[Sequence]
    """A collection of sequences."""

    version: str = "1"
    """Metatag to track version of datastructure."""

    def save(self, path: str, overwrite: bool = False) -> int:
        """Write flacker data to file."""
        with open(path, "w+" if overwrite else "x", encoding="UTF-8") as fp:
            return fp.write(self.to_json())  # type: ignore

    @staticmethod
    def load(path: str) -> "Flacker":
        """Write flacker data to file."""
        with open(path, "r", encoding="UTF-8") as fp:
            json = "\n".join(fp.readlines())

        return Flacker.schema().loads(json)  # type: ignore


class FlackerApiInterface:

    def update_fixture(self, fixtures: list[EngineFixture], time: Decimal) -> bool:
        """Update fixture states in API. Returns true if successful."""
        raise NotImplementedError("'update_fixture' not yet implemented.")


if __name__ == "__main__":
    # Small test
    f = Flacker(
        [
            Space(
                "iosf",
                "Living Room",
                [
                    Fixture(
                        "98iwd",
                        "Cool Light",
                        "hue",
                        "9jsif",
                        Capabilities.BRIGHTNESS,
                        SpaceVector(),
                        SpaceVector(),
                        SpaceVector(),
                        0.5,
                    )
                ],
            )
        ],
        [
            Sequence(
                "094rj",
                "Thunder",
                [
                    Track(
                        "89idf",
                        "Ceiling flicker",
                        Parameter.BRIGHTNESS,
                        ["98iwd"],
                        [Keyframe(Decimal.from_float(0), 1)],
                    )
                ],
            )
        ],
    )

    f.save("test.json")
    a = Flacker.load("test.json")

    print(f.to_json())  # type: ignore
