# ruff: noqa: D104, F401, E402
from importlib.metadata import version

from _duckdb import __version__ as duckdb_version

import duckdb.functional as functional
import duckdb.typing as typing

# duckdb.__version__ returns the version of the distribution package, i.e. the pypi version
__version__ = version("duckdb")


# version() is a more human friendly formatted version string of both the distribution package and the bundled duckdb
def version() -> str:
    return f"{__version__} (with duckdb {duckdb_version})"


_exported_symbols = ["__version__", "version"]

_exported_symbols.extend(["typing", "functional"])


class DBAPITypeObject:
    def __init__(self, types: list[typing.DuckDBPyType]) -> None:
        self.types = types

    def __eq__(self, other: object) -> bool:
        if isinstance(other, typing.DuckDBPyType):
            return other in self.types
        return False

    def __repr__(self) -> str:
        return f"<DBAPITypeObject [{','.join(str(x) for x in self.types)}]>"


# Define the standard DBAPI sentinels
STRING = DBAPITypeObject([typing.VARCHAR])
NUMBER = DBAPITypeObject(
    [
        typing.TINYINT,
        typing.UTINYINT,
        typing.SMALLINT,
        typing.USMALLINT,
        typing.INTEGER,
        typing.UINTEGER,
        typing.BIGINT,
        typing.UBIGINT,
        typing.HUGEINT,
        typing.UHUGEINT,
        typing.DuckDBPyType("BIGNUM"),
        typing.DuckDBPyType("DECIMAL"),
        typing.FLOAT,
        typing.DOUBLE,
    ]
)
DATETIME = DBAPITypeObject(
    [
        typing.DATE,
        typing.TIME,
        typing.TIME_TZ,
        typing.TIMESTAMP,
        typing.TIMESTAMP_TZ,
        typing.TIMESTAMP_NS,
        typing.TIMESTAMP_MS,
        typing.TIMESTAMP_S,
    ]
)
BINARY = DBAPITypeObject([typing.BLOB])
ROWID = None

# Classes
from _duckdb import (
    CaseExpression,
    CoalesceOperator,
    ColumnExpression,
    ConstantExpression,
    CSVLineTerminator,
    DefaultExpression,
    DuckDBPyConnection,
    DuckDBPyRelation,
    ExpectedResultType,
    ExplainType,
    Expression,
    FunctionExpression,
    LambdaExpression,
    PythonExceptionHandling,
    RenderMode,
    SQLExpression,
    StarExpression,
    Statement,
    StatementType,
)

_exported_symbols.extend(
    [
        "DuckDBPyRelation",
        "DuckDBPyConnection",
        "ExplainType",
        "PythonExceptionHandling",
        "Expression",
        "ConstantExpression",
        "ColumnExpression",
        "DefaultExpression",
        "CoalesceOperator",
        "LambdaExpression",
        "StarExpression",
        "FunctionExpression",
        "CaseExpression",
        "SQLExpression",
    ]
)

# These are overloaded twice, we define them inside of C++ so pybind can deal with it
_exported_symbols.extend(["df", "arrow"])
# NOTE: this section is generated by tools/pythonpkg/scripts/generate_connection_wrapper_methods.py.
# Do not edit this section manually, your changes will be overwritten!
# START OF CONNECTION WRAPPER
from _duckdb import (
    aggregate,
    alias,
    append,
    array_type,
    arrow,
    begin,
    checkpoint,
    close,
    commit,
    create_function,
    cursor,
    decimal_type,
    description,
    df,
    distinct,
    dtype,
    duplicate,
    enum_type,
    execute,
    executemany,
    extract_statements,
    fetch_arrow_table,
    fetch_df,
    fetch_df_chunk,
    fetch_record_batch,
    fetchall,
    fetchdf,
    fetchmany,
    fetchnumpy,
    fetchone,
    filesystem_is_registered,
    filter,
    from_arrow,
    from_csv_auto,
    from_df,
    from_parquet,
    from_query,
    get_table_names,
    install_extension,
    interrupt,
    limit,
    list_filesystems,
    list_type,
    load_extension,
    map_type,
    order,
    pl,
    project,
    query,
    query_df,
    query_progress,
    read_csv,
    read_json,
    read_parquet,
    register,
    register_filesystem,
    remove_function,
    rollback,
    row_type,
    rowcount,
    sql,
    sqltype,
    string_type,
    struct_type,
    table,
    table_function,
    tf,
    torch,
    type,
    union_type,
    unregister,
    unregister_filesystem,
    values,
    view,
    write_csv,
)

_exported_symbols.extend(
    [
        "cursor",
        "register_filesystem",
        "unregister_filesystem",
        "list_filesystems",
        "filesystem_is_registered",
        "create_function",
        "remove_function",
        "sqltype",
        "dtype",
        "type",
        "array_type",
        "list_type",
        "union_type",
        "string_type",
        "enum_type",
        "decimal_type",
        "struct_type",
        "row_type",
        "map_type",
        "duplicate",
        "execute",
        "executemany",
        "close",
        "interrupt",
        "query_progress",
        "fetchone",
        "fetchmany",
        "fetchall",
        "fetchnumpy",
        "fetchdf",
        "fetch_df",
        "df",
        "fetch_df_chunk",
        "pl",
        "fetch_arrow_table",
        "arrow",
        "fetch_record_batch",
        "torch",
        "tf",
        "begin",
        "commit",
        "rollback",
        "checkpoint",
        "append",
        "register",
        "unregister",
        "table",
        "view",
        "values",
        "table_function",
        "read_json",
        "extract_statements",
        "sql",
        "query",
        "from_query",
        "read_csv",
        "from_csv_auto",
        "from_df",
        "from_arrow",
        "from_parquet",
        "read_parquet",
        "from_parquet",
        "read_parquet",
        "get_table_names",
        "install_extension",
        "load_extension",
        "project",
        "distinct",
        "write_csv",
        "aggregate",
        "alias",
        "filter",
        "limit",
        "order",
        "query_df",
        "description",
        "rowcount",
    ]
)

# END OF CONNECTION WRAPPER

# Enums
from _duckdb import ANALYZE, COLUMNS, DEFAULT, RETURN_NULL, ROWS, STANDARD

_exported_symbols.extend(["ANALYZE", "DEFAULT", "RETURN_NULL", "STANDARD"])


# read-only properties
from _duckdb import (
    __formatted_python_version__,
    __interactive__,
    __jupyter__,
    __standard_vector_size__,
    apilevel,
    comment,
    identifier,
    keyword,
    numeric_const,
    operator,
    paramstyle,
    string_const,
    threadsafety,
    token_type,
    tokenize,
)

_exported_symbols.extend(
    [
        "__standard_vector_size__",
        "__interactive__",
        "__jupyter__",
        "__formatted_python_version__",
        "apilevel",
        "comment",
        "identifier",
        "keyword",
        "numeric_const",
        "operator",
        "paramstyle",
        "string_const",
        "threadsafety",
        "token_type",
        "tokenize",
    ]
)


from _duckdb import (
    connect,
    default_connection,
    set_default_connection,
)

_exported_symbols.extend(
    [
        "connect",
        "default_connection",
        "set_default_connection",
    ]
)

# Exceptions
from _duckdb import (
    BinderException,
    CatalogException,
    ConnectionException,
    ConstraintException,
    ConversionException,
    DataError,
    Error,
    FatalException,
    HTTPException,
    IntegrityError,
    InternalError,
    InternalException,
    InterruptException,
    InvalidInputException,
    InvalidTypeException,
    IOException,
    NotImplementedException,
    NotSupportedError,
    OperationalError,
    OutOfMemoryException,
    OutOfRangeException,
    ParserException,
    PermissionException,
    ProgrammingError,
    SequenceException,
    SerializationException,
    SyntaxException,
    TransactionException,
    TypeMismatchException,
    Warning,
)

_exported_symbols.extend(
    [
        "Error",
        "DataError",
        "ConversionException",
        "OutOfRangeException",
        "TypeMismatchException",
        "FatalException",
        "IntegrityError",
        "ConstraintException",
        "InternalError",
        "InternalException",
        "InterruptException",
        "NotSupportedError",
        "NotImplementedException",
        "OperationalError",
        "ConnectionException",
        "IOException",
        "HTTPException",
        "OutOfMemoryException",
        "SerializationException",
        "TransactionException",
        "PermissionException",
        "ProgrammingError",
        "BinderException",
        "CatalogException",
        "InvalidInputException",
        "InvalidTypeException",
        "ParserException",
        "SyntaxException",
        "SequenceException",
        "Warning",
    ]
)

# Value
from duckdb.value.constant import (
    BinaryValue,
    BitValue,
    BlobValue,
    BooleanValue,
    DateValue,
    DecimalValue,
    DoubleValue,
    FloatValue,
    HugeIntegerValue,
    IntegerValue,
    IntervalValue,
    LongValue,
    NullValue,
    ShortValue,
    StringValue,
    TimestampMilisecondValue,
    TimestampNanosecondValue,
    TimestampSecondValue,
    TimestampTimeZoneValue,
    TimestampValue,
    TimeTimeZoneValue,
    TimeValue,
    UnsignedBinaryValue,
    UnsignedIntegerValue,
    UnsignedLongValue,
    UnsignedShortValue,
    UUIDValue,
    Value,
)

_exported_symbols.extend(
    [
        "Value",
        "NullValue",
        "BooleanValue",
        "UnsignedBinaryValue",
        "UnsignedShortValue",
        "UnsignedIntegerValue",
        "UnsignedLongValue",
        "BinaryValue",
        "ShortValue",
        "IntegerValue",
        "LongValue",
        "HugeIntegerValue",
        "FloatValue",
        "DoubleValue",
        "DecimalValue",
        "StringValue",
        "UUIDValue",
        "BitValue",
        "BlobValue",
        "DateValue",
        "IntervalValue",
        "TimestampValue",
        "TimestampSecondValue",
        "TimestampMilisecondValue",
        "TimestampNanosecondValue",
        "TimestampTimeZoneValue",
        "TimeValue",
        "TimeTimeZoneValue",
    ]
)

__all__ = _exported_symbols
