# 进度监控管理功能使用说明

## 问题解决

✅ **解决了问题**：之前每次调用 `run_pools` 都会启动新的 `progress_monitor.py` 进程，导致系统中有过多的监控进程运行。

✅ **现在的改进**：
- 相同备份文件会复用现有的监控进程
- 不同备份文件会创建独立的监控进程  
- 提供手动清理功能
- 自动清理已结束的僵尸进程

## 基本使用方法

### 1. 正常使用 run_pools（无变化）

```python
import rust_pyfunc

# 你的计算函数
def my_function(date: int, code: str):
    # 你的计算逻辑
    return [date * 1.0, len(code), date + len(code)]

# 准备数据
args = [[20240101, "000001"], [20240102, "000002"], ...]

# 调用 run_pools（API 完全不变）
result = rust_pyfunc.run_pools(
    my_function,
    args,
    num_threads=4,
    backup_file='./my_backup.bin',  # 指定备份文件
    storage_format='binary',
    resume_from_backup=True,
    backup_batch_size=100
)
```

### 2. 新增的清理功能

如果你发现系统中有过多的 `progress_monitor.py` 进程，可以手动清理：

```python
import rust_pyfunc

# 强制清理所有进度监控进程
rust_pyfunc.cleanup_all_progress_monitors()
```

## 工作原理

### 进程复用逻辑

1. **第一次调用** `run_pools` 时：
   - 如果指定了 `backup_file`，会启动对应的进度监控进程
   - 进程会基于备份文件路径生成唯一标识

2. **后续调用** 同样的 `backup_file` 时：
   - 会检查是否已有对应的监控进程在运行
   - 如果有，直接复用，不会重新启动
   - 如果进程已经结束，会自动清理并启动新的

3. **使用不同的** `backup_file` 时：
   - 会创建新的独立监控进程
   - 不同的备份文件对应不同的监控进程

### 自动清理机制

- 系统会自动检测并清理已经结束的监控进程
- 避免僵尸进程的累积
- 每次启动新监控前都会进行清理检查

## 检查运行状态

你可以使用系统命令检查当前的进度监控进程：

```bash
# 查看所有 progress_monitor.py 进程
pgrep -f progress_monitor.py

# 查看进程详情
ps aux | grep progress_monitor.py
```

## 最佳实践

1. **使用一致的备份文件名**：
   - 如果是同一个计算任务，使用相同的备份文件名
   - 这样可以最大化进程复用的效果

2. **定期清理**：
   - 在长时间运行的程序开始前调用 `cleanup_all_progress_monitors()`
   - 确保系统环境清洁

3. **监控系统资源**：
   - 虽然现在避免了重复启动，但如果有多个不同的备份文件，仍可能有多个监控进程
   - 可以根据需要手动清理

## 示例脚本

```python
#!/usr/bin/env python3
import rust_pyfunc
import time

def my_calculation(date: int, code: str):
    # 模拟计算
    import time
    time.sleep(0.01)  # 在函数内导入 time
    return [date * 1.0, len(code), date + len(code)]

def main():
    # 清理旧的监控进程
    rust_pyfunc.cleanup_all_progress_monitors()
    
    # 准备数据
    args = [[20240101 + i, f"00000{i:02d}"] for i in range(10)]
    
    # 第一次运行
    print("第一次运行...")
    result1 = rust_pyfunc.run_pools(
        my_calculation,
        args,
        num_threads=4,
        backup_file='./test_backup.bin',
        storage_format='binary',
        resume_from_backup=False
    )
    
    # 第二次运行（会复用监控进程）
    print("第二次运行（复用监控进程）...")
    result2 = rust_pyfunc.run_pools(
        my_calculation,
        args[:5],
        num_threads=4,
        backup_file='./test_backup.bin',  # 相同的备份文件
        storage_format='binary',
        resume_from_backup=True
    )
    
    # 清理所有监控进程
    rust_pyfunc.cleanup_all_progress_monitors()
    print("✅ 完成！")

if __name__ == "__main__":
    main()
```

## 技术细节

- 监控进程基于备份文件路径生成唯一标识键
- 使用全局静态HashMap管理所有监控进程
- 线程安全的进程管理和清理机制
- 自动检测并移除已结束的进程

这样就解决了您遇到的"每隔一段时间就会新开一个进程来运行 progress_monitor.py"的问题！ 