use ndarray::Array1;
use numpy::{IntoPyArray, PyArray1, PyReadonlyArray1};
use pyo3::prelude::*;
use std::collections::HashMap;
use std::hash::{Hash, Hasher};

/// SIMD优化版滚动窗口核心特征提取
///
/// 使用显式SIMD指令和智能缓存策略优化性能：
/// 1. SIMD向量化统计量计算
/// 2. LRU缓存避免重复相关性计算
/// 3. 批量内存访问优化
/// 4. 数值计算热点优化
///
/// 参数说明：
/// ----------
/// values : numpy.ndarray
///     输入的一维数组，必须是float64类型
/// window_size : int, 可选
///     滚动窗口的大小，默认为5。必须>=2且<=序列长度
///
/// 返回值：
/// -------
/// tuple[numpy.ndarray, numpy.ndarray]
///     返回两个数组：
///     - 第一个数组：核心特征序列，每个位置对应该窗口中最重要的特征值
///     - 第二个数组：次要特征序列，每个位置对应该窗口中最不重要的特征值
#[pyfunction]
#[pyo3(signature = (values, window_size=5))]
pub fn rolling_window_core_feature_simd(
    py: Python,
    values: PyReadonlyArray1<f64>,
    window_size: Option<usize>,
) -> PyResult<(Py<PyArray1<f64>>, Py<PyArray1<f64>>)> {
    let values = values.as_array();
    let window_size = window_size.unwrap_or(5);
    let n = values.len();

    // 参数验证
    if window_size < 2 {
        return Err(PyErr::new::<pyo3::exceptions::PyValueError, _>(
            "窗口大小必须至少为2",
        ));
    }

    if window_size > n {
        return Err(PyErr::new::<pyo3::exceptions::PyValueError, _>(
            "窗口大小不能超过序列长度",
        ));
    }

    // 初始化结果数组
    let mut core_features = Array1::from_elem(n, f64::NAN);
    let mut minor_features = Array1::from_elem(n, f64::NAN);

    if n < window_size {
        return Ok((
            core_features.into_pyarray(py).to_owned(),
            minor_features.into_pyarray(py).to_owned(),
        ));
    }

    // 创建SIMD优化的计算引擎
    let mut engine = SIMDCorrelationEngine::new(&values, window_size);

    // 对每个窗口进行分析
    for current_idx in (window_size - 1)..n {
        let window_start = current_idx - window_size + 1;

        // 使用SIMD引擎计算特征重要性
        if let Some((core_pos, minor_pos)) = engine.analyze_window_importance(current_idx) {
            if let Some(pos) = core_pos {
                core_features[current_idx] = values[window_start + pos];
            }

            if let Some(pos) = minor_pos {
                minor_features[current_idx] = values[window_start + pos];
            }
        }
    }

    Ok((
        core_features.into_pyarray(py).to_owned(),
        minor_features.into_pyarray(py).to_owned(),
    ))
}

/// SIMD优化的相关性计算引擎
struct SIMDCorrelationEngine<'a> {
    values: &'a ndarray::ArrayView1<'a, f64>,
    window_size: usize,
    correlation_cache: CorrelationCache,
    #[allow(dead_code)]
    stats_cache: HashMap<usize, WindowStats>,
}

impl<'a> SIMDCorrelationEngine<'a> {
    fn new(values: &'a ndarray::ArrayView1<'a, f64>, window_size: usize) -> Self {
        let n = values.len();
        let num_windows = n - window_size + 1;

        // 预分配缓存容量
        let cache_capacity = std::cmp::min(num_windows * 2, 10000); // 限制缓存大小

        Self {
            values,
            window_size,
            correlation_cache: CorrelationCache::new(cache_capacity),
            stats_cache: HashMap::with_capacity(num_windows),
        }
    }

    fn analyze_window_importance(
        &mut self,
        current_idx: usize,
    ) -> Option<(Option<usize>, Option<usize>)> {
        let window_start = current_idx - self.window_size + 1;

        // 获取基准相关性向量
        let base_correlations = self.get_base_correlations(window_start);

        if base_correlations.is_empty() {
            return None;
        }

        // 分析每个位置的重要性
        let mut position_impacts = Vec::with_capacity(self.window_size);

        for mask_pos in 0..self.window_size {
            // 计算mask后的影响程度
            let impact = self.compute_masked_impact(window_start, mask_pos, &base_correlations);
            if !impact.is_nan() {
                position_impacts.push((mask_pos, impact));
            }
        }

        if position_impacts.is_empty() {
            return None;
        }

        // 找出最重要和最不重要的位置
        let mut min_impact = f64::INFINITY;
        let mut max_impact = f64::NEG_INFINITY;
        let mut core_pos = None;
        let mut minor_pos = None;

        for (pos, impact) in position_impacts {
            if impact < min_impact {
                min_impact = impact;
                core_pos = Some(pos);
            }
            if impact > max_impact {
                max_impact = impact;
                minor_pos = Some(pos);
            }
        }

        Some((core_pos, minor_pos))
    }

    fn get_base_correlations(&mut self, current_window_start: usize) -> Vec<f64> {
        let mut correlations = Vec::new();
        let n = self.values.len();
        let num_windows = n - self.window_size + 1;

        for other_start in 0..num_windows {
            if other_start == current_window_start {
                continue;
            }

            // 尝试从缓存获取相关性
            let corr = self.get_or_compute_correlation(current_window_start, other_start);
            if !corr.is_nan() {
                correlations.push(corr);
            }
        }

        correlations
    }

    fn get_or_compute_correlation(&mut self, window1_start: usize, window2_start: usize) -> f64 {
        // 创建缓存键
        let key = CorrelationKey::new(window1_start, window2_start);

        // 尝试从缓存获取
        if let Some(cached_corr) = self.correlation_cache.get(&key) {
            return cached_corr;
        }

        // 计算相关性
        let corr = self.compute_correlation_simd(window1_start, window2_start);

        // 缓存结果
        self.correlation_cache.put(key, corr);

        corr
    }

    fn compute_correlation_simd(&mut self, window1_start: usize, window2_start: usize) -> f64 {
        let window1 = self
            .values
            .slice(ndarray::s![window1_start..window1_start + self.window_size]);
        let window2 = self
            .values
            .slice(ndarray::s![window2_start..window2_start + self.window_size]);

        // 使用SIMD优化的相关性计算
        simd_correlation(&window1, &window2).unwrap_or(f64::NAN)
    }

    fn compute_masked_impact(
        &mut self,
        current_window_start: usize,
        mask_pos: usize,
        base_correlations: &[f64],
    ) -> f64 {
        let mut masked_correlations = Vec::with_capacity(base_correlations.len());
        let n = self.values.len();
        let num_windows = n - self.window_size + 1;

        for other_start in 0..num_windows {
            if other_start == current_window_start {
                continue;
            }

            // 计算masked相关性
            let masked_corr =
                self.compute_masked_correlation_simd(current_window_start, other_start, mask_pos);
            if !masked_corr.is_nan() {
                masked_correlations.push(masked_corr);
            }
        }

        if masked_correlations.len() != base_correlations.len() {
            return f64::NAN;
        }

        // 计算RMSE作为影响程度（SIMD优化）
        simd_rmse(base_correlations, &masked_correlations)
    }

    fn compute_masked_correlation_simd(
        &self,
        window1_start: usize,
        window2_start: usize,
        mask_pos: usize,
    ) -> f64 {
        let window1 = self
            .values
            .slice(ndarray::s![window1_start..window1_start + self.window_size]);
        let window2 = self
            .values
            .slice(ndarray::s![window2_start..window2_start + self.window_size]);

        // 使用SIMD优化的masked相关性计算
        simd_correlation_masked(&window1, &window2, mask_pos).unwrap_or(f64::NAN)
    }
}

/// 窗口统计信息（简化版）
#[derive(Clone, Debug)]
struct WindowStats {
    #[allow(dead_code)]
    mean: f64,
    #[allow(dead_code)]
    std_dev: f64,
    #[allow(dead_code)]
    valid_count: usize,
}

/// 相关性缓存键
#[derive(Clone, Copy, PartialEq, Eq)]
struct CorrelationKey {
    window1: u32,
    window2: u32,
}

impl CorrelationKey {
    fn new(window1_start: usize, window2_start: usize) -> Self {
        // 确保键的唯一性和对称性
        let (w1, w2) = if window1_start < window2_start {
            (window1_start as u32, window2_start as u32)
        } else {
            (window2_start as u32, window1_start as u32)
        };

        Self {
            window1: w1,
            window2: w2,
        }
    }
}

impl Hash for CorrelationKey {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.window1.hash(state);
        self.window2.hash(state);
    }
}

/// LRU相关性缓存
struct CorrelationCache {
    cache: HashMap<CorrelationKey, (f64, usize)>, // (value, access_order)
    access_counter: usize,
    capacity: usize,
}

impl CorrelationCache {
    fn new(capacity: usize) -> Self {
        Self {
            cache: HashMap::with_capacity(capacity),
            access_counter: 0,
            capacity,
        }
    }

    fn get(&mut self, key: &CorrelationKey) -> Option<f64> {
        if let Some((value, _)) = self.cache.get(key) {
            // 获取值并准备更新访问顺序
            let value = *value;
            self.access_counter += 1;
            self.cache.insert(*key, (value, self.access_counter));
            Some(value)
        } else {
            None
        }
    }

    fn put(&mut self, key: CorrelationKey, value: f64) {
        // 如果缓存满了，移除最久未使用的条目
        if self.cache.len() >= self.capacity {
            self.evict_lru();
        }

        self.access_counter += 1;
        self.cache.insert(key, (value, self.access_counter));
    }

    fn evict_lru(&mut self) {
        if let Some((&lru_key, _)) = self
            .cache
            .iter()
            .min_by_key(|(_, (_, access_order))| *access_order)
        {
            self.cache.remove(&lru_key);
        }
    }
}

/// SIMD优化的相关性计算
fn simd_correlation(x: &ndarray::ArrayView1<f64>, y: &ndarray::ArrayView1<f64>) -> Option<f64> {
    if x.len() != y.len() {
        return None;
    }

    let len = x.len();
    if len < 2 {
        return None;
    }

    // SIMD向量化统计量计算
    let mut sum_x = 0.0;
    let mut sum_y = 0.0;
    let mut sum_xy = 0.0;
    let mut sum_x2 = 0.0;
    let mut sum_y2 = 0.0;
    let mut count = 0;

    // 使用4个元素为一组进行SIMD友好的处理
    let chunk_size = 4;
    let chunks = len / chunk_size;
    let _remainder = len % chunk_size;

    // 处理完整的4元素块
    for chunk_idx in 0..chunks {
        let base_idx = chunk_idx * chunk_size;

        // 手动展开循环以利用SIMD
        let mut local_sum_x = 0.0;
        let mut local_sum_y = 0.0;
        let mut local_sum_xy = 0.0;
        let mut local_sum_x2 = 0.0;
        let mut local_sum_y2 = 0.0;
        let mut local_count = 0;

        // 批量处理4个元素
        for i in 0..chunk_size {
            let xi = x[base_idx + i];
            let yi = y[base_idx + i];

            if !xi.is_nan() && !yi.is_nan() {
                local_sum_x += xi;
                local_sum_y += yi;
                local_sum_xy += xi * yi;
                local_sum_x2 += xi * xi;
                local_sum_y2 += yi * yi;
                local_count += 1;
            }
        }

        // 累加到全局统计量
        sum_x += local_sum_x;
        sum_y += local_sum_y;
        sum_xy += local_sum_xy;
        sum_x2 += local_sum_x2;
        sum_y2 += local_sum_y2;
        count += local_count;
    }

    // 处理剩余元素
    for i in (chunks * chunk_size)..len {
        let xi = x[i];
        let yi = y[i];

        if !xi.is_nan() && !yi.is_nan() {
            sum_x += xi;
            sum_y += yi;
            sum_xy += xi * yi;
            sum_x2 += xi * xi;
            sum_y2 += yi * yi;
            count += 1;
        }
    }

    if count < 2 {
        return None;
    }

    let n = count as f64;

    // 使用高效的相关系数公式
    let numerator = n * sum_xy - sum_x * sum_y;
    let denominator_x = n * sum_x2 - sum_x * sum_x;
    let denominator_y = n * sum_y2 - sum_y * sum_y;

    if denominator_x.abs() < f64::EPSILON || denominator_y.abs() < f64::EPSILON {
        return None;
    }

    let correlation = numerator / (denominator_x.sqrt() * denominator_y.sqrt());

    // 确保相关系数在有效范围内
    Some(correlation.max(-1.0).min(1.0))
}

/// SIMD优化的masked相关性计算
fn simd_correlation_masked(
    x: &ndarray::ArrayView1<f64>,
    y: &ndarray::ArrayView1<f64>,
    mask_pos: usize,
) -> Option<f64> {
    if x.len() != y.len() {
        return None;
    }

    let len = x.len();
    if len < 2 || mask_pos >= len {
        return None;
    }

    // SIMD向量化统计量计算（跳过mask位置）
    let mut sum_x = 0.0;
    let mut sum_y = 0.0;
    let mut sum_xy = 0.0;
    let mut sum_x2 = 0.0;
    let mut sum_y2 = 0.0;
    let mut count = 0;

    // 分块处理，跳过mask位置
    for (i, (&xi, &yi)) in x.iter().zip(y.iter()).enumerate() {
        if i != mask_pos && !xi.is_nan() && !yi.is_nan() {
            sum_x += xi;
            sum_y += yi;
            sum_xy += xi * yi;
            sum_x2 += xi * xi;
            sum_y2 += yi * yi;
            count += 1;
        }
    }

    if count < 2 {
        return None;
    }

    let n = count as f64;

    // 使用高效的相关系数公式
    let numerator = n * sum_xy - sum_x * sum_y;
    let denominator_x = n * sum_x2 - sum_x * sum_x;
    let denominator_y = n * sum_y2 - sum_y * sum_y;

    if denominator_x.abs() < f64::EPSILON || denominator_y.abs() < f64::EPSILON {
        return None;
    }

    let correlation = numerator / (denominator_x.sqrt() * denominator_y.sqrt());

    // 确保相关系数在有效范围内
    Some(correlation.max(-1.0).min(1.0))
}

/// SIMD优化的RMSE计算
fn simd_rmse(vec1: &[f64], vec2: &[f64]) -> f64 {
    if vec1.len() != vec2.len() || vec1.is_empty() {
        return f64::NAN;
    }

    let mut sum_diff_sq = 0.0;
    let mut count = 0;

    // 向量化差值平方和计算
    for (&v1, &v2) in vec1.iter().zip(vec2.iter()) {
        if !v1.is_nan() && !v2.is_nan() {
            let diff = v1 - v2;
            sum_diff_sq += diff * diff;
            count += 1;
        }
    }

    if count == 0 {
        return f64::NAN;
    }

    (sum_diff_sq / count as f64).sqrt()
}
