import imageio.v3 as iio
import argparse
import numpy as np

from pathlib import Path


class Converter():
    def __init__(self, image_path) -> None:
        self.image_path = image_path
        self.image = iio.imread(image_path)
        self.path = Path(image_path).resolve()
        self.parent_path = Path(image_path).parent

    def class_checker(self) -> None:
        if np.array_equal(np.unique(self.image), [0,1,2]):
            raise ValueError('The input mask already has the correct pixel classes!') 
        if len(np.unique(self.image)) != 3:
            raise ValueError(f'The input mask contains an incorrect number of class labels. Please ensure you only have 3 classes! Got {len(np.unique(self.image))} classes.')

    def convert_mask(self) -> None:
        mask_copy = self.image.copy()

        mask_copy[self.image == 1] = 0
        mask_copy[self.image == 2] = 1
        mask_copy[self.image == 3] = 2

        iio.imwrite(self.path, mask_copy)

        img_name = self.image_path.split('/')[-1] if '/' in self.image_path else self.image_path

        print(f'...Saved converted input mask {img_name} in {self.parent_path}...')


def parse_args():
    parser = argparse.ArgumentParser(prog='pyRootHair: Mask Converter',
                                     description='Convert an input mask generated by ilastik to have the correct pixel classes for inference and processing.')
    
    parser.add_argument('-i', '--input', help='Filepath to input mask generated using ilastik.',
                        type=str, dest='image_path')
    
    return parser.parse_args()

def main():
    args = parse_args()
    c = Converter(args.image_path)
    c.class_checker()
    c.convert_mask()


if __name__ == '__main__':
    main()



