import datetime as dt
import pytest
import pytz
from unittest.mock import MagicMock, patch, Mock
import requests
import json
import pandas as pd

from lumibot.tools.projectx_helpers import ProjectXAuth, ProjectXClient, ProjectX, _to_utc_iso
from lumibot.entities import Asset


@pytest.fixture
def mock_projectx_config():
    """Mock ProjectX configuration for testing"""
    return {
        "firm": "TEST",
        "api_key": "test_api_key",
        "username": "test_user",
        "base_url": "https://test.projectx.com",
        "preferred_account_name": "TestAccount",
        "streaming_base_url": "wss://test.projectx.com/hub"
    }


class TestProjectXAuth:
    """Test ProjectX authentication functionality"""

    def test_get_auth_token_success(self, mock_projectx_config):
        """Test successful authentication token retrieval"""
        
        # Mock successful response
        mock_response = Mock()
        mock_response.json.return_value = {"success": True, "token": "test_auth_token"}
        mock_response.raise_for_status = Mock()

        with patch('requests.post', return_value=mock_response):
            token = ProjectXAuth.get_auth_token(mock_projectx_config)

            assert token == "test_auth_token"

    def test_get_auth_token_failure(self, mock_projectx_config):
        """Test authentication failure handling"""
        
        # Mock failed response
        mock_response = Mock()
        mock_response.json.return_value = {"success": False, "errorCode": 401, "errorMessage": "Invalid credentials"}
        mock_response.raise_for_status = Mock()

        with patch('requests.post', return_value=mock_response):
            token = ProjectXAuth.get_auth_token(mock_projectx_config)
            # Should return None on failure
            assert token is None

    def test_get_auth_token_invalid_json(self, mock_projectx_config):
        """Test handling of invalid JSON response"""
        
        # Mock response with invalid JSON
        mock_response = Mock()
        mock_response.json.side_effect = json.JSONDecodeError("Invalid JSON", "", 0)
        mock_response.raise_for_status = Mock()

        with patch('requests.post', return_value=mock_response):
            token = ProjectXAuth.get_auth_token(mock_projectx_config)
            # Should return None on JSON error
            assert token is None

    def test_get_auth_token_missing_config(self):
        """Test error handling for missing configuration"""
        
        incomplete_config = {
            "firm": "TEST",
            # Missing api_key, username, base_url
        }
        
        with pytest.raises(ValueError, match="Missing required configuration"):
            ProjectXAuth.get_auth_token(incomplete_config)


@pytest.fixture
def mock_client(mock_projectx_config):
    """Create ProjectXClient with mocked dependencies"""
    with patch.object(ProjectXAuth, 'get_auth_token', return_value="test_token"):
        client = ProjectXClient(mock_projectx_config)
        client.api = MagicMock()  # Mock the underlying API client
        return client


class TestProjectXClient:
    """Test ProjectX client functionality"""

    def test_client_initialization(self, mock_projectx_config):
        """Test client initialization"""
        
        with patch.object(ProjectXAuth, 'get_auth_token', return_value="test_token"):
            client = ProjectXClient(mock_projectx_config)
            
            assert client.firm == "TEST"
            assert client.token == "test_token"

    def test_get_accounts(self, mock_client):
        """Test account retrieval"""
        
        # Mock accounts response
        mock_accounts = [
            {
                "AccountId": 12345,
                "AccountName": "TestAccount",
                "Balance": 100000.0,
                "Equity": 99500.0
            }
        ]
        mock_client.api.account_search = MagicMock(return_value={"success": True, "accounts": mock_accounts})
        
        accounts = mock_client.get_accounts()
        
        assert len(accounts) == 1
        assert accounts[0]["AccountId"] == 12345

    def test_get_preferred_account_id(self, mock_client):
        """Test preferred account ID retrieval"""
        
        # Mock accounts response with practice account names
        mock_accounts = [
            {"id": 12345, "name": "PracAccount", "balance": 100000.0},
            {"id": 67890, "name": "OtherAccount", "balance": 50000.0}
        ]
        mock_client.api.account_search = MagicMock(return_value={"success": True, "accounts": mock_accounts})
        
        account_id = mock_client.get_preferred_account_id()
        
        # Should return first practice account's ID
        assert account_id == 12345

    def test_get_account_balance(self, mock_client):
        """Test account balance retrieval"""
        
        # Mock accounts response with matching account ID
        mock_accounts = [
            {"id": 12345, "name": "TestAccount", "balance": 50000.0},
            {"id": 67890, "name": "OtherAccount", "balance": 25000.0}
        ]
        mock_client.api.account_search = MagicMock(return_value={"success": True, "accounts": mock_accounts})
        
        balance = mock_client.get_account_balance(12345)
        
        assert balance["cash"] == 50000.0
        assert balance["equity"] == 50000.0

    def test_get_positions(self, mock_client):
        """Test position retrieval"""
        
        # Mock positions response
        mock_positions = [
            {
                "symbol": "MES",
                "contractId": "CON.F.US.MES.U25",
                "size": 5,
                "avgPrice": 5150.0,
                "unrealizedPnL": 250.0
            }
        ]
        mock_client.api.position_search_open = MagicMock(return_value={"success": True, "positions": mock_positions})
        
        positions = mock_client.get_positions(12345)
        
        assert len(positions) == 1
        assert positions[0]["symbol"] == "MES"

    def test_get_orders(self, mock_client):
        """Test order retrieval"""
        
        # Mock orders response
        mock_orders = [
            {
                "id": 123456,
                "symbol": "MES",
                "size": 1,
                "status": 2,  # Open
                "type": 2,    # Market
                "side": 0     # Buy
            }
        ]
        mock_client.api.order_search = MagicMock(return_value={"success": True, "orders": mock_orders})
        
        orders = mock_client.get_orders(12345, "2024-01-01", "2024-01-31")
        
        assert len(orders) == 1
        assert orders[0]["id"] == 123456

    def test_place_order_success(self, mock_client):
        """Test successful order placement"""
        
        # Mock successful order response
        mock_response = {
            "success": True,
            "orderId": 789012,
            "message": "Order placed successfully"
        }
        mock_client.api.order_place = MagicMock(return_value=mock_response)
        
        response = mock_client.place_order(12345, "CON.F.US.MES.U25", "buy", 1, 2)
        
        assert response["success"] is True
        assert response["orderId"] == 789012

    def test_cancel_order_success(self, mock_client):
        """Test successful order cancellation"""
        
        # Mock successful cancel response
        mock_response = {
            "success": True,
            "message": "Order canceled successfully"
        }
        mock_client.api.order_cancel = MagicMock(return_value=mock_response)
        
        result = mock_client.cancel_order(12345, "123456")
        
        assert result is True

    def test_get_historical_data(self, mock_client):
        """Test historical data retrieval"""
        
        # Mock historical data response
        mock_df = pd.DataFrame({
            'time': pd.date_range('2024-01-15 09:30:00', periods=2, freq='1min'),
            'open': [5150.0, 5152.0],
            'high': [5155.0, 5157.0],
            'low': [5148.0, 5151.0],
            'close': [5152.0, 5156.0],
            'volume': [1000, 1200]
        })
        mock_client.api.history_retrieve_bars = MagicMock(return_value=mock_df)
        
        data = mock_client.get_historical_data("CON.F.US.MES.U25", "2024-01-15T09:30:00", "2024-01-15T10:30:00")
        
        assert isinstance(data, pd.DataFrame)
        assert len(data) == 2

    def test_search_contracts(self, mock_client):
        """Test contract search functionality"""
        
        # Mock contract search response
        mock_contracts = [
            {
                "ContractId": "CON.F.US.MES.U25",
                "Symbol": "MES",
                "Description": "E-mini S&P 500 Future",
                "ExpirationDate": "2025-09-19"
            }
        ]
        mock_client.api.contract_search = MagicMock(return_value={"success": True, "contracts": mock_contracts})
        
        contracts = mock_client.search_contracts("MES")
        
        assert len(contracts) == 1
        assert contracts[0]["Symbol"] == "MES"

    def test_get_contract_details(self, mock_client):
        """Test contract details retrieval"""
        
        # Mock contract details response - get_contract_details returns contracts[0] not contract field
        mock_response = {
            "success": True,
            "contracts": [{
                "ContractId": "CON.F.US.MES.U25",
                "Symbol": "MES",
                "Description": "E-mini S&P 500 Future",
                "ContractSize": 50,
                "TickSize": 0.25
            }]
        }
        mock_client.api.contract_search_id = MagicMock(return_value=mock_response)
        
        details = mock_client.get_contract_details("CON.F.US.MES.U25")
        
        # The method returns contracts[0] directly
        assert details["Symbol"] == "MES"
        assert details["TickSize"] == 0.25

    def test_find_contract_by_symbol(self, mock_client):
        """Test finding contract by symbol.

        To avoid coupling to real-time rollover dates, mock the Asset generator
        so the first candidate is MESU25 and verify we convert to ProjectX ID.
        """

        mock_contracts = [
            {
                "ContractId": "CON.F.US.MES.U25",
                "Symbol": "MES"
            }
        ]
        mock_client.api.contract_search = MagicMock(return_value={"success": True, "contracts": mock_contracts})

        with patch.object(Asset, 'get_potential_futures_contracts', return_value=['MESU25', 'MES.U25', 'MESU2025']):
            contract_id = mock_client.find_contract_by_symbol("MES")
            assert contract_id == "CON.F.US.MES.U25"

    def test_contract_id_conversion_no_hardcoded_mappings(self, mock_client):
        """
        Test that contract ID conversion doesn't use hardcoded mappings.
        This was a major issue - hardcoded 'MES': 'CON.F.US.MES.U25' mappings.
        """
        
        # For this test, we need to mock out the hardcoded fallback to force API search
        original_common_futures = mock_client.find_contract_by_symbol.__func__.__code__.co_consts
        
        # Mock the Asset class method to force fallback behavior
        with patch.object(Asset, 'get_potential_futures_contracts') as mock_contracts:
            mock_contracts.side_effect = Exception("Asset method failed")  # Force fallback
            
            # Mock the search_contracts method
            mock_client.search_contracts = MagicMock(return_value=[
                {"contractId": "CON.F.US.UNKNOWN.U25", "Symbol": "UNKNOWN"}
            ])
            
            # Test with a symbol that's NOT in the hardcoded mapping
            contract_id = mock_client.find_contract_by_symbol("UNKNOWN")
            
            # Should find contract dynamically via API search
            assert contract_id == "CON.F.US.UNKNOWN.U25"
            # Verify the search_contracts method was called
            mock_client.search_contracts.assert_called_with("UNKNOWN")

    def test_get_contract_tick_size(self, mock_client):
        """Test contract tick size retrieval"""
        
        # Mock contract details with tick size
        mock_details = {
            "success": True,
            "contract": {
                "TickSize": 0.25
            }
        }
        mock_client.api.contract_search_id = MagicMock(return_value=mock_details)
        
        tick_size = mock_client.get_contract_tick_size("CON.F.US.MES.U25")
        
        assert tick_size == 0.25

    def test_round_to_tick_size(self, mock_client):
        """Test price rounding to tick size"""
        
        # Test rounding functionality - actual implementation rounds down to nearest tick
        rounded_price = mock_client.round_to_tick_size(5150.123, 0.25)
        
        # Should round to nearest tick (implementation rounds down)
        assert rounded_price == 5150.0  # Implementation rounds down, not to nearest


@pytest.fixture
def mock_api_client(mock_projectx_config):
    """Create ProjectX API client with mocked dependencies"""
    with patch.object(ProjectXAuth, 'get_auth_token', return_value="test_token"):
        client = ProjectX(mock_projectx_config, "test_token")
        return client


class TestProjectXApiClient:
    """Test low-level ProjectX API client"""

    def test_api_client_initialization(self, mock_projectx_config):
        """Test API client initialization"""
        with patch.object(ProjectXAuth, 'get_auth_token', return_value="test_token"):
            client = ProjectX(mock_projectx_config, "test_token")

            assert client.firm == "TEST"
            assert client.base_url == "https://test.projectx.com/"  # Note trailing slash

    def test_api_request_headers(self, mock_api_client):
        """Test that API requests include proper authentication headers"""
        
        # Verify headers are set correctly
        expected_headers = {
            "Authorization": "Bearer test_token",
            "Content-Type": "application/json"
        }
        
        assert mock_api_client.headers == expected_headers

    def test_account_search(self, mock_api_client):
        """Test account search functionality"""
        
        # Mock successful response
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.return_value = {"success": True, "accounts": []}
        
        with patch('requests.post', return_value=mock_response):
            result = mock_api_client.account_search()
            
            assert result["success"] is True

    def test_position_search_open(self, mock_api_client):
        """Test position search functionality"""
        
        # Mock successful response
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.return_value = {"success": True, "positions": []}
        
        with patch('requests.post', return_value=mock_response):
            result = mock_api_client.position_search_open(12345)
            
            assert result["success"] is True

    def test_order_search(self, mock_api_client):
        """Test order search functionality"""
        
        # Mock successful response
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.return_value = {"success": True, "orders": []}
        
        with patch('requests.post', return_value=mock_response):
            result = mock_api_client.order_search(12345, "2024-01-01T00:00:00", "2024-01-31T23:59:59")
            
            assert result["success"] is True

    def test_order_place(self, mock_api_client):
        """Test order placement"""
        
        # Mock successful response
        mock_response = Mock()
        mock_response.json.return_value = {"success": True, "orderId": 789012}
        
        with patch('requests.post', return_value=mock_response):
            result = mock_api_client.order_place(12345, "CON.F.US.MES.U25", 2, 0, 1)
            
            assert result["success"] is True
            assert result["orderId"] == 789012

    def test_order_cancel(self, mock_api_client):
        """Test order cancellation"""
        
        # Mock successful response
        mock_response = Mock()
        mock_response.json.return_value = {"success": True}
        
        with patch('requests.post', return_value=mock_response):
            result = mock_api_client.order_cancel(12345, 789012)
            
            assert result["success"] is True

    def test_contract_search(self, mock_api_client):
        """Test contract search"""
        
        # Mock successful response
        mock_response = Mock()
        mock_response.json.return_value = {
            "success": True,
            "contracts": [{"ContractId": "CON.F.US.MES.U25", "Symbol": "MES"}]
        }
        
        with patch('requests.post', return_value=mock_response):
            result = mock_api_client.contract_search("MES")
            
            assert result["success"] is True
            assert len(result["contracts"]) == 1

    def test_contract_search_id(self, mock_api_client):
        """Test contract search by ID"""
        
        # Mock successful response
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.return_value = {
            "success": True,
            "contract": {"ContractId": "CON.F.US.MES.U25", "Symbol": "MES"}
        }
        
        with patch('requests.post', return_value=mock_response):
            result = mock_api_client.contract_search_id("CON.F.US.MES.U25")
            
            assert result["success"] is True

    def test_history_retrieve_bars(self, mock_api_client):
        """Test historical bars retrieval"""
        
        # Mock successful response - the actual implementation expects a JSON response with bars field
        mock_response = Mock()
        mock_response.json.return_value = {
            "success": True,
            "bars": [
                {
                    "t": "2024-01-15T09:30:00Z",
                    "o": 5150.0,
                    "h": 5155.0,
                    "l": 5148.0,
                    "c": 5152.0,
                    "v": 1000
                }
            ]
        }
        
        with patch('requests.post', return_value=mock_response):
            # Use proper ISO format without timezone conversion issues
            df = mock_api_client.history_retrieve_bars(
                "CON.F.US.MES.U25", 
                "2024-01-15T09:30:00Z",  # Include Z for UTC
                "2024-01-15T10:30:00Z", 
                1, 1, 
                is_est=False  # Disable EST conversion to avoid parsing issues
            )
            
            assert isinstance(df, pd.DataFrame)

    def test_get_streaming_client(self, mock_api_client):
        """Test streaming client creation"""
        
        # Import the SIGNALR_AVAILABLE flag to check if signalrcore is available
        from lumibot.tools.projectx_helpers import SIGNALR_AVAILABLE
        
        if SIGNALR_AVAILABLE:
            # When signalrcore is available, should return streaming client
            streaming_client = mock_api_client.get_streaming_client(12345)
            assert streaming_client is not None
        else:
            # When signalrcore is not available, should raise ImportError
            with pytest.raises(ImportError, match="signalrcore library is required"):
                mock_api_client.get_streaming_client(12345)


class TestProjectXErrorHandling:
    """Test ProjectX error handling scenarios"""

    def test_authentication_retry_logic(self, mock_projectx_config):
        """Test authentication retry logic for transient failures"""
        
        # Mock successful authentication
        with patch.object(ProjectXAuth, 'get_auth_token', return_value="test_token"):
            token = ProjectXAuth.get_auth_token(mock_projectx_config)
            assert token == "test_token"

    def test_api_rate_limiting_handling(self, mock_api_client):
        """Test API rate limiting handling"""
        
        # Mock rate limit response
        mock_response = Mock()
        mock_response.status_code = 429
        mock_response.json.return_value = {"success": False, "error": "Rate limited"}
        
        with patch('requests.post', return_value=mock_response):
            result = mock_api_client.account_search()
            
            # Should handle rate limiting gracefully
            assert result["success"] is False

    def test_network_error_handling(self, mock_api_client):
        """Test network error handling"""
        
        # Mock network error
        with patch('requests.post', side_effect=requests.exceptions.RequestException("Network error")):
            result = mock_api_client.account_search()
            
            # Should handle network errors gracefully
            assert result["success"] is False
            assert "error" in result

    def test_invalid_response_handling(self, mock_api_client):
        """Test handling of invalid API responses"""
        
        # Mock invalid JSON response - need to handle this at the right level
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.side_effect = json.JSONDecodeError("Invalid JSON", "", 0)
        
        with patch('requests.post', return_value=mock_response):
            # The actual implementation doesn't catch JSON errors, so this will raise
            with pytest.raises(json.JSONDecodeError):
                mock_api_client.account_search()

    def test_http_error_codes(self, mock_api_client):
        """Test handling of various HTTP error codes"""
        
        error_codes = [400, 401, 403, 404, 500, 502, 503]
        
        for status_code in error_codes:
            mock_response = Mock()
            mock_response.status_code = status_code
            mock_response.reason = "Error"
            
            with patch('requests.post', return_value=mock_response):
                result = mock_api_client.account_search()
                
                # Should handle all error codes gracefully
                assert result["success"] is False
                assert f"HTTP {status_code}" in result["error"]


class TestProjectXDatetime:
    def test_to_utc_iso_from_z_string(self):
        res = _to_utc_iso("2025-01-02T09:30:00Z", is_est=True)
        assert res == "2025-01-02T14:30:00+00:00"

    def test_to_utc_iso_from_naive_dt(self):
        start_dt = dt.datetime(2025, 1, 2, 9, 30)
        res = _to_utc_iso(start_dt, is_est=True)
        assert res == "2025-01-02T14:30:00+00:00"

    def test_to_utc_iso_from_aware_utc_no_est(self):
        start_dt = pytz.utc.localize(dt.datetime(2025, 1, 2, 14, 30))
        res = _to_utc_iso(start_dt, is_est=False)
        assert res == "2025-01-02T14:30:00+00:00"

    def test_to_utc_iso_string_without_z(self):
        res = _to_utc_iso("2025-01-02T09:30:00", is_est=True)
        assert res == "2025-01-02T14:30:00+00:00"