from http import HTTPStatus
from typing import Any, Optional, Union, cast

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.attachment import Attachment
from ...models.attachment_create_from_url import AttachmentCreateFromUrl
from ...types import Response


def _get_kwargs(
    id: str,
    *,
    body: AttachmentCreateFromUrl,
) -> dict[str, Any]:
    headers: dict[str, Any] = {}

    _kwargs: dict[str, Any] = {
        "method": "post",
        "url": "/tasks/{id}/attachments/from-url".format(
            id=id,
        ),
    }

    _body = body.to_dict()

    _kwargs["json"] = _body
    headers["Content-Type"] = "application/json"

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Any, Attachment]]:
    if response.status_code == 200:
        response_200 = Attachment.from_dict(response.json())

        return response_200
    if response.status_code == 400:
        response_400 = cast(Any, None)
        return response_400
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Any, Attachment]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: AttachmentCreateFromUrl,
) -> Response[Union[Any, Attachment]]:
    """Attach a file from a provided URL to a task

     Attach a file from a provided URL to a task. The file will be downloaded and attached
    asynchronously. This operation may take a few moments to complete.

    Args:
        id (str):
        body (AttachmentCreateFromUrl):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, Attachment]]
    """

    kwargs = _get_kwargs(
        id=id,
        body=body,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: AttachmentCreateFromUrl,
) -> Optional[Union[Any, Attachment]]:
    """Attach a file from a provided URL to a task

     Attach a file from a provided URL to a task. The file will be downloaded and attached
    asynchronously. This operation may take a few moments to complete.

    Args:
        id (str):
        body (AttachmentCreateFromUrl):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, Attachment]
    """

    return sync_detailed(
        id=id,
        client=client,
        body=body,
    ).parsed


async def asyncio_detailed(
    id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: AttachmentCreateFromUrl,
) -> Response[Union[Any, Attachment]]:
    """Attach a file from a provided URL to a task

     Attach a file from a provided URL to a task. The file will be downloaded and attached
    asynchronously. This operation may take a few moments to complete.

    Args:
        id (str):
        body (AttachmentCreateFromUrl):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, Attachment]]
    """

    kwargs = _get_kwargs(
        id=id,
        body=body,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: AttachmentCreateFromUrl,
) -> Optional[Union[Any, Attachment]]:
    """Attach a file from a provided URL to a task

     Attach a file from a provided URL to a task. The file will be downloaded and attached
    asynchronously. This operation may take a few moments to complete.

    Args:
        id (str):
        body (AttachmentCreateFromUrl):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, Attachment]
    """

    return (
        await asyncio_detailed(
            id=id,
            client=client,
            body=body,
        )
    ).parsed
