from pydantic import BaseModel, ConfigDict, field_validator, model_validator
from typing import Optional, Mapping, Any
from importlib import import_module
import yaml


def get_machine(name):
    from .machine import Machine

    *package_path, class_name = name.split(".")
    package_name = ".".join(package_path)
    package = import_module(package_name)
    machine = getattr(package, class_name)
    assert issubclass(machine, Machine)
    return machine


class MachineConfig(BaseModel):
    model_config = ConfigDict(extra="allow")

    config: Optional[Mapping[str, Any]] = {}
    replace: Optional[str] = None

    @field_validator("replace")
    @classmethod
    def get_machines(cls, replacement):
        return get_machine(replacement)

    @model_validator(mode="after")
    def validate_extra(self):
        for key, val in self.__pydantic_extra__.items():
            setattr(self, key, self.__class__(**val))
        return self


class PigeonTransitionsConfig(BaseModel):
    root: str
    machines: Optional[MachineConfig] = MachineConfig()

    @field_validator("root")
    @classmethod
    def get_machine(cls, root: str):
        return get_machine(root)

    @classmethod
    def load(cls, data):
        return cls(**yaml.safe_load(data))

    @classmethod
    def load_file(cls, file):
        with open(file) as f:
            return cls.load(f.read())


class ConfigPropogator:
    def __init__(self, config=None, **kwargs):
        super().__init__(**kwargs)

    @classmethod
    def init_child(cls, name, config, *args, **kwargs):
        """This is a helper routine for propogating configuration to child machines."""
        if isinstance(config, MachineConfig) and hasattr(config, name):
            machine_config = getattr(config, name)
            kwargs["config"] = machine_config
            kwargs.update(machine_config.config)
            if machine_config.replace is not None:
                return machine_config.replace(*args, **kwargs)
        return cls(*args, **kwargs)
