from .base import BaseMachine, root


class MailboxException(Exception):
    pass


class MailboxMachine(BaseMachine):
    sends_to = {}
    sends_to_parent = []
    gets_from = {}
    gets_from_parent = []

    def __init__(self, *args, **kwargs):
        self._mailbox = {}
        super().__init__(*args, **kwargs)

    def send_to(self, child, field, data):
        if child not in self._children:
            raise MailboxException(
                f"Cannot send to '{child}' as it is not a child of '{self.get_state_path()}'."
            )
        if field not in self.sends_to.get(child, []):
            raise MailboxException(
                f"The field '{field}' is not specified in '{self.__class__.__name__}.sends_to'."
            )
        self._send_to(self._children[child], "parent", field, data)

    def send_to_parent(self, field, data):
        if self.is_root:
            raise MailboxException("Root machine has no parent.")
        if field not in self.sends_to_parent:
            raise MailboxException(
                f"The field '{field}' is not specified in '{self.__class__.__name__}.sends_to_parent'."
            )
        self._send_to(self._parent, self.state_name, field, data)

    def _send_to(self, state, source, field, data):
        if source not in state._mailbox:
            state._mailbox[source] = {field: data}
        else:
            state._mailbox[source][field] = data

    def get_from(self, child, field):
        if child not in self._children:
            raise MailboxException(
                f"Cannot get from '{child}' as it is not a child of '{self.get_state_path()}'."
            )
        if field not in self.gets_from.get(child, []):
            raise MailboxException(
                f"The field '{field}' is not specified in '{self.__class__.__name__}.gets_from'."
            )
        if field not in self._mailbox.get(child, {}):
            raise MailboxException(
                f"The field '{field}' has not been sent to '{self.get_state_path()}' by '{child}' yet."
            )
        return self._mailbox[child][field]

    def get_from_parent(self, field):
        if self.is_root:
            raise MailboxException("Root machine has no parent.")
        if field not in self.gets_from_parent:
            raise MailboxException(
                f"The field '{field}' is not specified in '{self.__class__.__name__}.gets_from_parent'."
            )
        if field not in self._mailbox.get("parent", {}):
            raise MailboxException(
                f"The field '{field}' has not been sent to '{self.get_state_path()}' by the parent."
            )
        return self._mailbox["parent"][field]

    def check_mailboxes(self):
        if self.is_root and self.sends_to_parent != []:
            raise MailboxException(
                f"Root machine has no parent. '{self.__class__.__name__}' must have 'sends_to_parent' set to '[]'."
            )
        if self.is_root and self.gets_from_parent != []:
            raise MailboxException(
                f"Root machine has no parent. '{self.__class__.__name__}' must have 'gets_from_parent' set to '[]'."
            )
        for name in self.sends_to:
            if name not in self._children:
                raise MailboxException(
                    f"Cannot send {self.sends_to[name]} to '{name}' as the child does not exist."
                )
        for name in self.gets_from:
            if name not in self._children:
                raise MailboxException(
                    f"Cannot get {self.gets_from[name]} from '{name}' as the child does not exist."
                )
        for name, child in self._children.items():
            if set(self.sends_to.get(name, [])) != set(child.gets_from_parent):
                raise MailboxException(
                    f"Inconsistent specifications, '{self.get_state_path()}' expects to send {self.sends_to.get(name, [])} to '{name}' which expects to recieve {child.gets_from_parent}."
                )
            if set(self.gets_from.get(name, [])) != set(child.sends_to_parent):
                raise MailboxException(
                    f"Inconsistent specifications, '{self.get_state_path()}' expects to recieve {self.gets_from.get(name, [])} from '{name}' which expects to send {child.sends_to_parent}."
                )
            child.check_mailboxes()

    @root
    def _start(self, *args, **kwargs):
        self.check_mailboxes()
        super()._start(*args, **kwargs)
