import contextlib
import json
import logging

from starlette.requests import Request

logger = logging.getLogger('app')


def create_request_log(
        request: Request,
        status_code: int,
        request_body: bytes,
        message: str,
        trace: str | None = None,
        error: Exception | None = None,
        ignore_fields: list[str] | None = None
) -> None:
    query_params = dict(request.query_params)
    request_body = json.loads(request_body) if request_body != b'' else {}  # type: ignore
    if ignore_fields:
        for field in ignore_fields:
            with contextlib.suppress(KeyError):
                del request_body[field]  # type: ignore
                del query_params[field]

    data: dict[str, str | int | dict | None] = {
        "user_id": request.query_params.get('user_id', ""),
        "method": request.method,
        "path": request.url.path,
        "client_ip":  request.client.host if request.client else None,
        "user_agent": request.headers.get("user-agent", ""),
        "query_params": str(query_params),
        "body": str(request_body)
    }
    if error:
        data.update({
            "status_code": status_code,
            "error": str(error),
            "trace": trace[-1000:] if trace else None
        })
        logger.error(message, extra=data)
    else:
        data.update({
            "status_code": status_code
        })
        logger.info(message, extra=data)
