# setup.py
from setuptools import setup, find_packages
import os

with open("README.md", "r", encoding="utf-8") as fh:
    long_description = fh.read()

with open("requirements.txt", "r", encoding="utf-8") as fh:
    requirements = [line.strip() for line in fh if line.strip() and not line.startswith("#")]

# Optional dependencies for different feature sets
extras_require = {
    # Production features (Phase 1)
    'production': [
        'fastapi>=0.104.0',
        'uvicorn[standard]>=0.24.0',
        'sse-starlette>=1.6.5',
    ],
    
    # Security features (Phase 2)
    'security': [
        'pyjwt>=2.8.0',
        'cryptography>=41.0.0',
        'python-jose[cryptography]>=3.3.0',
        'passlib[bcrypt]>=1.7.4',
    ],
    
    # Advanced orchestration (Phase 3)
    'orchestration': [
        'celery>=5.3.4',
        'redis>=5.0.0',
    ],
    
    # Ecosystem features (Phase 4)
    'ecosystem': [
        'pydantic>=2.5.0',
        'pillow>=10.1.0',
        'opencv-python>=4.8.0',
    ],
    
    # RAG capabilities
    'rag': [
        'chromadb>=0.4.18',
        'langchain>=0.1.0',
        'sentence-transformers>=2.2.2',
        'unstructured>=0.11.0',
    ],
    
    # Memory persistence
    'postgres': [
        'psycopg2-binary>=2.9.9',
        'sqlalchemy>=2.0.23',
    ],
    
    # Analytics
    'analytics': [
        'prometheus-client>=0.19.0',
        'plotly>=5.18.0',
    ],
    
    # Development tools
    'dev': [
        'pytest>=7.4.3',
        'pytest-asyncio>=0.21.1',
        'pytest-cov>=4.1.0',
        'black>=23.12.0',
        'flake8>=6.1.0',
        'mypy>=1.7.1',
        'isort>=5.13.0',
    ],
}

# 'all' includes everything
extras_require['all'] = list(set(sum(extras_require.values(), [])))

setup(
    name="fluxgraph",
    version="2.0.0",  # Major version for enterprise features
    author="Ihtesham Jahangir",
    author_email="ceo@alphanetwork.com.pk",
    description="Production-grade enterprise AI agent orchestration framework - The only open-source alternative to Microsoft Agent Framework",
    long_description=long_description,
    long_description_content_type="text/markdown",
    url="https://github.com/ihtesham-jahangir/fluxgraph",
    project_urls={
        "Bug Tracker": "https://github.com/ihtesham-jahangir/fluxgraph/issues",
        "Documentation": "https://fluxgraph.readthedocs.io",
        "Source Code": "https://github.com/ihtesham-jahangir/fluxgraph",
        "Discord": "https://discord.gg/Z9bAqjYvPc",
    },
    packages=find_packages(where="."),
    package_dir={"": "."},
    include_package_data=True,
    classifiers=[
        "Development Status :: 4 - Beta",
        "Intended Audience :: Developers",
        "License :: OSI Approved :: MIT License",
        "Operating System :: OS Independent",
        "Programming Language :: Python :: 3",
        "Programming Language :: Python :: 3.8",
        "Programming Language :: Python :: 3.9",
        "Programming Language :: Python :: 3.10",
        "Programming Language :: Python :: 3.11",
        "Programming Language :: Python :: 3.12",
        "Topic :: Software Development :: Libraries :: Python Modules",
        "Topic :: Scientific/Engineering :: Artificial Intelligence",
        "Framework :: FastAPI",
        "Typing :: Typed",
    ],
    keywords=[
        "ai", "agents", "llm", "gpt", "anthropic", "langchain", "orchestration",
        "fastapi", "async", "multi-agent", "rag", "langraph", "autogen", "crewai",
        "enterprise", "production", "security", "audit", "compliance", "hitl",
        "batch-processing", "mcp", "streaming", "session-management"
    ],
    python_requires='>=3.8',
    install_requires=requirements,
    extras_require=extras_require,
    entry_points={
        'console_scripts': [
            'flux=fluxgraph.core.app:main',
        ],
    },
    zip_safe=False,
)
