import { DatabaseOptions } from '../database-options';
import { ITable, TableAction } from '../table';
import { IUser } from '../user';
import { Construct } from 'constructs';
/**
 * The Redshift table and action that make up a privilege that can be granted to a Redshift user.
 */
export interface TablePrivilege {
    /**
     * The table on which privileges will be granted.
     */
    readonly table: ITable;
    /**
     * The actions that will be granted.
     */
    readonly actions: TableAction[];
}
/**
 * Properties for specifying privileges granted to a Redshift user on Redshift tables.
 */
export interface UserTablePrivilegesProps extends DatabaseOptions {
    /**
     * The user to which privileges will be granted.
     */
    readonly user: IUser;
    /**
     * The privileges to be granted.
     *
     * @default [] - use `addPrivileges` to grant privileges after construction
     */
    readonly privileges?: TablePrivilege[];
}
/**
 * Privileges granted to a Redshift user on Redshift tables.
 *
 * This construct is located in the `private` directory to ensure that it is not exported for direct public use. This
 * means that user privileges must be managed through the `Table.grant` method or the `User.addTablePrivileges`
 * method. Thus, each `User` will have at most one `UserTablePrivileges` construct to manage its privileges. For details
 * on why this is a Good Thing, see the README, under "Granting Privileges".
 */
export declare class UserTablePrivileges extends Construct {
    private privileges;
    constructor(scope: Construct, id: string, props: UserTablePrivilegesProps);
    /**
     * Grant this user additional privileges.
     */
    addPrivileges(table: ITable, ...actions: TableAction[]): void;
}
