"""
Classes for handling comparison files generated by the Fortran version of sfit.
"""
import os.path
import numpy as np


class FortranSFitFile51(object):
    """
    Class to parse the fort.51 comparison file generated by
    the Fortran version of sfit.
    """
    def __init__(self, filename='fort.51', dir_=None):
        if dir_ is not None:
            filename = os.path.join(dir_, filename)

        with open(filename, 'r') as input_file:
            key = None
            for line in input_file.readlines():
                words = line.split()
                if words[0] == '#':
                    key = words[-1]
                    continue
                elif len(words) == 1:
                    value = float(words[0])
                else:
                    value = np.array([float(item) for item in words])

                self.__setattr__(key, value)

        self.source_fluxes = [fs for fs in self.a[9::3]]


class FortranSFitFile61(object):
    """
    Class to parse the fort.61 comparison file generated by
    the Fortran version of sfit.

    'nob' = observatory #
    'k' = data point #
    't' = time
    'dfdrho' = fs * dA_FSPL / drho
    'mag' = FSPL magnification (A * (B0 - gamma * B1)
    'db0' = db0/dz
    'db1' = db1/dz
    """

    def __init__(self, filename='fort.61', dir_=None):
        if dir_ is not None:
            filename = os.path.join(dir_, filename)

        self.data = np.genfromtxt(
            filename, dtype=None, encoding='utf-8',
            names=['nob', 'k', 't', 'dfdrho', 'mag', 'db0', 'db1'])
        for name in self.data.dtype.names:
            self.__setattr__(name, self.data[name])

        self.sfit_nob_indices = self._set_nob_indices()

    def _set_nob_indices(self):
        nobs = np.unique(self.data['nob'])
        nob_indices = []
        for nob in np.sort(nobs):
            sfit_index = (self.data['nob'] == nob)
            nob_indices.append(sfit_index)

        return nob_indices


class FortranSFitFile62(FortranSFitFile61):
    """
    Class to parse the fort.62 comparison file generated by
    the Fortran version of sfit.

    'nob' = observatory #
    'k' = data point #
    't' = time
    'dfdt0' = fs * dA_FSPL / dt0
    'dfdu0' = fs * dA_FSPL / du0
    'dfdtE' = fs * dA_FSPL / dtE
    'dfdrho' = fs * dA_FSPL / drho
    'dAdu' = dA_PSPL/du
    'df' = fs*(damp*(b0-gamma*b1)+amp*(db0-gamma*db1)/rho)
    'res' = residuals (flux)
    'sig2' = errors^2
    """

    def __init__(self, filename='fort.62', dir_=None):
        if dir_ is not None:
            filename = os.path.join(dir_, filename)

        self.data = np.genfromtxt(
            filename, dtype=None, encoding='utf-8',
            names=['nob', 'k', 't', 'dfdt0', 'dfdu0', 'dfdtE', 'dfdrho',
                   'dAdu', 'df', 'res', 'sig2'])
        for name in self.data.dtype.names:
            self.__setattr__(name, self.data[name])

        self.sfit_nob_indices = self._set_nob_indices()


class FortranSFitFile63(FortranSFitFile61):
    """
    Class to parse the fort.63 comparison file generated by
    the Fortran version of sfit.

    'nob' = observatory #
    'k' = data point #
    't' = time
    'x' = u(t)
    'x2' = u(t)^2
    'amp' = PSPL magnification
    'b0' = b0
    'b1' = b1
    """

    def __init__(self, filename='fort.63', dir_=None):
        if dir_ is not None:
            filename = os.path.join(dir_, filename)

        self.data = np.genfromtxt(
            filename, dtype=None, encoding='utf-8',
            names=['nob', 'k', 't', 'x', 'x2', 'amp', 'b0', 'b1'])
        for name in self.data.dtype.names:
            self.__setattr__(name, self.data[name])

        self.sfit_nob_indices = self._set_nob_indices()


def read_sfit_files(dir_):
    try:
        fort51 = FortranSFitFile51(dir_=dir_)
    except OSError:
        fort51 = None

    try:
        fort61 = FortranSFitFile61(dir_=dir_)
    except OSError:
        fort61 = None

    try:
        fort62 = FortranSFitFile62(dir_=dir_)
    except OSError:
        fort62 = None

    try:
        fort63 = FortranSFitFile63(dir_=dir_)
    except OSError:
        fort63 = None

    return {'51': fort51, '61': fort61, '62': fort62, '63': fort63}
