import json

import requests

from . import check_input_attribute, standard_error_message
from ._vulnerability_parser.vulnerability_parser import VulnerabilityLookupParser

misperrors = {"error": "Error"}
mispattributes = {"input": ["vulnerability"], "format": "misp_standard"}
moduleinfo = {
    "version": "2",
    "author": "Christian Studer",
    "description": "An expansion module to query Vulnerability Lookup",
    "module-type": ["expansion", "hover"],
    "name": "Vulnerability Lookup",
    "logo": "vulnerability_lookup.png",
    "requirements": [],
    "features": (
        "The module takes a vulnerability attribute as input and queries Vulnerability Lookup to gather additional"
        " information based on the Vulnerability ID. The result of the query is then parsed and converted into MISP"
        " content which can be added to the original event to enrich the input attribute."
    ),
    "references": ["https://vulnerability.circl.lu"],
    "input": "Vulnerability Attribute",
    "output": "Additional information on the vulnerability, gathered from the Vulnerability Lookup API.",
}
api_url = "https://vulnerability.circl.lu"


def handler(q=False):
    if q is False:
        return q
    request = json.loads(q)
    if not check_input_attribute(request.get("attribute", {})):
        return {"error": f"{standard_error_message}, which should contain at least a type, a value and an UUID."}
    attribute = request["attribute"]
    if attribute.get("type") != "vulnerability":
        misperrors["error"] = "Vulnerability ID missing"
        return misperrors
    lookup = requests.get(f"{api_url}/api/vulnerability/{attribute['value']}")
    if lookup.status_code == 200:
        vulnerability = lookup.json()
        if not vulnerability:
            misperrors["error"] = "Non existing Vulnerability ID."
            return misperrors
    else:
        misperrors["error"] = "Vulnerability Lookup API not accessible."
        return misperrors
    parser = VulnerabilityLookupParser(attribute, api_url)
    parser.parse_lookup_result(vulnerability)
    return parser.get_results()


def introspection():
    return mispattributes


def version():
    return moduleinfo
