"""
Exploit Testing Tool

Context Engineering:
- Safe exploitation testing for educational/authorized use
- Returns structured exploit results
- Built-in safety checks and consent verification
- Educational mode by default

Educational exploitation testing - authorized use only.
"""

from typing import Dict, Any, List, Literal
from pydantic import BaseModel, Field
from loguru import logger
from pathlib import Path
import re

from alprina_cli.tools.base import AlprinaToolBase, ToolOk, ToolError


class ExploitParams(BaseModel):
    """
    Parameters for exploit testing.

    Context: Focused schema for exploitation testing.
    """
    target: str = Field(
        description="Target to test (file, directory, or URL)"
    )
    exploit_type: Literal["safe", "poc", "educational"] = Field(
        default="safe",
        description="Exploit mode: safe (non-destructive), poc (proof-of-concept), educational (learning)"
    )
    vulnerability: str = Field(
        default="",
        description="Specific vulnerability to test (e.g., 'sql_injection', 'xss', 'path_traversal')"
    )
    max_attempts: int = Field(
        default=5,
        description="Maximum exploit attempts"
    )


class ExploitTool(AlprinaToolBase[ExploitParams]):
    """
    Exploit testing tool for authorized security testing.

    Context Engineering Benefits:
    - Safe mode by default (non-destructive)
    - Educational focus
    - Structured exploit results
    - Built-in safety checks

    IMPORTANT: This tool is for:
    - Educational purposes
    - Authorized penetration testing
    - CTF competitions
    - Security research with proper authorization

    Exploit Types:
    - safe: Non-destructive testing only
    - poc: Proof-of-concept demonstrations
    - educational: Learning and training

    Supported Vulnerabilities:
    - sql_injection: SQL injection testing
    - xss: Cross-site scripting testing
    - path_traversal: Path traversal testing
    - command_injection: Command injection testing

    Usage:
    ```python
    tool = ExploitTool()
    result = await tool.execute(ExploitParams(
        target="./test_app",
        exploit_type="educational",
        vulnerability="sql_injection"
    ))
    ```
    """

    name: str = "Exploit"
    description: str = """Exploit testing for authorized security testing.

AUTHORIZED USE ONLY - For education, pentesting, CTF, research.

Capabilities:
- SQL injection testing
- XSS testing
- Path traversal testing
- Command injection testing

Returns: Structured exploit results with safety information"""
    params: type[ExploitParams] = ExploitParams

    async def execute(self, params: ExploitParams) -> ToolOk | ToolError:
        """
        Execute exploit testing.

        Context: Returns structured, educational results.
        """
        logger.info(f"Exploit: {params.target} (type={params.exploit_type}, vuln={params.vulnerability})")

        try:
            # Safety check: Ensure safe mode
            if params.exploit_type not in ["safe", "poc", "educational"]:
                return ToolError(
                    message="Only safe, poc, and educational modes are supported",
                    brief="Invalid exploit mode"
                )

            # Determine target type
            target_path = Path(params.target).expanduser()
            is_local = target_path.exists()

            if is_local:
                results = await self._exploit_local(params, target_path)
            else:
                results = await self._exploit_remote(params)

            # Limit attempts
            if len(results) > params.max_attempts:
                results = results[:params.max_attempts]
                truncated = True
            else:
                truncated = False

            # Calculate success rate
            successful = sum(1 for r in results if r.get("successful", False))
            success_rate = (successful / len(results) * 100) if results else 0

            return ToolOk(
                content={
                    "target": params.target,
                    "exploit_type": params.exploit_type,
                    "vulnerability": params.vulnerability or "general",
                    "results": results,
                    "summary": {
                        "total_attempts": len(results),
                        "successful": successful,
                        "success_rate": success_rate,
                        "truncated": truncated,
                        "target_type": "local" if is_local else "remote"
                    },
                    "safety_notice": "This tool is for authorized testing only. Always obtain proper authorization."
                }
            )

        except Exception as e:
            logger.error(f"Exploit testing failed: {e}")
            return ToolError(
                message=f"Exploit testing failed: {str(e)}",
                brief="Exploit failed"
            )

    async def _exploit_local(
        self,
        params: ExploitParams,
        target_path: Path
    ) -> List[Dict[str, Any]]:
        """
        Test exploits on local target.

        Context: Safe, educational exploitation testing.
        """
        results = []

        if target_path.is_file():
            results.extend(self._test_file_exploits(target_path, params))
        else:
            # Test directory
            files = list(target_path.rglob("*"))
            for file_path in files:
                if file_path.is_file():
                    results.extend(self._test_file_exploits(file_path, params))

                    # Limit attempts
                    if len(results) >= params.max_attempts:
                        break

        return results

    def _test_file_exploits(
        self,
        file_path: Path,
        params: ExploitParams
    ) -> List[Dict[str, Any]]:
        """Test exploits on individual file"""
        results = []

        try:
            # Skip binary files
            if self._is_binary(file_path):
                return results

            content = file_path.read_text(errors="ignore")
            lines = content.splitlines()

            # Test specific vulnerability or all
            if not params.vulnerability or params.vulnerability == "sql_injection":
                results.extend(self._test_sql_injection(file_path, content, lines))

            if not params.vulnerability or params.vulnerability == "xss":
                results.extend(self._test_xss(file_path, content, lines))

            if not params.vulnerability or params.vulnerability == "path_traversal":
                results.extend(self._test_path_traversal(file_path, content, lines))

            if not params.vulnerability or params.vulnerability == "command_injection":
                results.extend(self._test_command_injection(file_path, content, lines))

        except Exception as e:
            logger.warning(f"Could not test {file_path}: {e}")

        return results

    def _test_sql_injection(
        self,
        file_path: Path,
        content: str,
        lines: List[str]
    ) -> List[Dict[str, Any]]:
        """Test SQL injection vulnerabilities (educational)"""
        results = []

        # Look for vulnerable SQL patterns
        vulnerable_patterns = [
            (r"execute\([^)]*\+[^)]*\)", "String concatenation in SQL"),
            (r'SELECT.*"\s*\+\s*', "SQL query with concatenation"),
            (r"\.format\(.*SELECT", "format() with SQL query"),
        ]

        for pattern, description in vulnerable_patterns:
            matches = re.finditer(pattern, content, re.IGNORECASE)
            for match in matches:
                line_num = content[:match.start()].count('\n') + 1

                # Educational exploit payloads
                test_payloads = [
                    "' OR '1'='1",
                    "1' UNION SELECT NULL--",
                    "' OR 1=1--"
                ]

                results.append({
                    "vulnerability": "sql_injection",
                    "file": str(file_path),
                    "line_number": line_num,
                    "description": description,
                    "matched_code": lines[line_num - 1].strip()[:80],
                    "successful": False,  # Safe mode - don't actually exploit
                    "payloads_tested": test_payloads,
                    "educational_note": "Vulnerable pattern detected. In real testing, try payloads to confirm.",
                    "severity": "HIGH"
                })

        return results

    def _test_xss(
        self,
        file_path: Path,
        content: str,
        lines: List[str]
    ) -> List[Dict[str, Any]]:
        """Test XSS vulnerabilities (educational)"""
        results = []

        # Look for unescaped user input
        xss_patterns = [
            (r"\.innerHTML\s*=", "Direct innerHTML assignment"),
            (r"document\.write\(", "document.write usage"),
            (r"\.html\([^)]*\+", "jQuery html() with concatenation"),
        ]

        for pattern, description in xss_patterns:
            matches = re.finditer(pattern, content, re.IGNORECASE)
            for match in matches:
                line_num = content[:match.start()].count('\n') + 1

                test_payloads = [
                    "<script>alert('XSS')</script>",
                    "<img src=x onerror=alert('XSS')>",
                    "<svg onload=alert('XSS')>"
                ]

                results.append({
                    "vulnerability": "xss",
                    "file": str(file_path),
                    "line_number": line_num,
                    "description": description,
                    "matched_code": lines[line_num - 1].strip()[:80],
                    "successful": False,
                    "payloads_tested": test_payloads,
                    "educational_note": "Potential XSS vector. Test with payloads in controlled environment.",
                    "severity": "MEDIUM"
                })

        return results

    def _test_path_traversal(
        self,
        file_path: Path,
        content: str,
        lines: List[str]
    ) -> List[Dict[str, Any]]:
        """Test path traversal vulnerabilities (educational)"""
        results = []

        # Look for file operations with user input
        traversal_patterns = [
            (r"open\([^)]*\+[^)]*\)", "File open with concatenation"),
            (r"Path\([^)]*\+[^)]*\)", "Path construction with concatenation"),
            (r"readFile\([^)]*\+", "readFile with concatenation"),
            (r"=\s*[^+]*\+\s*user[_a-zA-Z]*.*\n.*open\(", "File path with user input"),
        ]

        for pattern, description in traversal_patterns:
            matches = re.finditer(pattern, content, re.IGNORECASE)
            for match in matches:
                line_num = content[:match.start()].count('\n') + 1

                test_payloads = [
                    "../../../etc/passwd",
                    "..\\..\\..\\windows\\system32\\config\\sam",
                    "....//....//....//etc/passwd"
                ]

                results.append({
                    "vulnerability": "path_traversal",
                    "file": str(file_path),
                    "line_number": line_num,
                    "description": description,
                    "matched_code": lines[line_num - 1].strip()[:80],
                    "successful": False,
                    "payloads_tested": test_payloads,
                    "educational_note": "Potential path traversal. Validate and sanitize file paths.",
                    "severity": "HIGH"
                })

        return results

    def _test_command_injection(
        self,
        file_path: Path,
        content: str,
        lines: List[str]
    ) -> List[Dict[str, Any]]:
        """Test command injection vulnerabilities (educational)"""
        results = []

        # Look for dangerous functions with user input
        cmd_patterns = [
            (r"os\.system\([^)]*\+[^)]*\)", "os.system with concatenation"),
            (r"subprocess\.call\([^)]*\+[^)]*\)", "subprocess with concatenation"),
            (r"eval\(", "eval() usage"),
            (r"exec\(", "exec() usage"),
        ]

        for pattern, description in cmd_patterns:
            matches = re.finditer(pattern, content, re.IGNORECASE)
            for match in matches:
                line_num = content[:match.start()].count('\n') + 1

                test_payloads = [
                    "; ls -la",
                    "| whoami",
                    "&& cat /etc/passwd"
                ]

                results.append({
                    "vulnerability": "command_injection",
                    "file": str(file_path),
                    "line_number": line_num,
                    "description": description,
                    "matched_code": lines[line_num - 1].strip()[:80],
                    "successful": False,
                    "payloads_tested": test_payloads,
                    "educational_note": "Command injection risk. Never execute user input directly.",
                    "severity": "CRITICAL"
                })

        return results

    async def _exploit_remote(self, params: ExploitParams) -> List[Dict[str, Any]]:
        """
        Test exploits on remote target.

        Context: Educational remote testing.
        """
        results = []

        # Educational notice for remote testing
        results.append({
            "vulnerability": "remote_testing",
            "description": "Remote exploit testing",
            "successful": False,
            "educational_note": (
                "Remote exploitation requires proper authorization. "
                "This tool provides educational information only. "
                "For real testing, use tools like Burp Suite, OWASP ZAP, or Metasploit."
            ),
            "severity": "INFO"
        })

        # Basic checks
        if params.target.startswith("http://"):
            results.append({
                "vulnerability": "insecure_protocol",
                "description": "Target uses HTTP (insecure)",
                "successful": True,  # This is a confirmed finding
                "recommendation": "Use HTTPS instead of HTTP",
                "severity": "MEDIUM"
            })

        return results

    def _is_binary(self, file_path: Path) -> bool:
        """Check if file is binary"""
        try:
            with open(file_path, 'rb') as f:
                chunk = f.read(8192)
            return b'\x00' in chunk
        except Exception:
            return False
