// SPDX-License-Identifier: GPL-3.0-or-later
pragma solidity ^0.8.0;

/**
 * Simplified recreation of the Balancer V2 vulnerability
 * The _upscaleArray function has a rounding error that allows
 * precision loss attacks when balances reach 8-9 wei range
 */

contract BalancerV2VulnerablePool {
    uint256 private constant _SCALING_FACTOR = 1e18;

    mapping(address => uint256) public balances;
    uint256 public totalSupply;

    /**
     * VULNERABILITY: The _upscaleArray function rounds down during scaling
     * When balances are in the 8-9 wei range, integer division causes
     * rounding discrepancies that can reach 10% per operation
     */
    function _upscaleArray(uint256[] memory amounts, uint256[] memory scalingFactors)
        internal
        pure
        returns (uint256[] memory)
    {
        uint256[] memory upscaled = new uint256[](amounts.length);

        for (uint256 i = 0; i < amounts.length; i++) {
            // VULNERABILITY: mulDown causes precision loss
            // When amounts[i] is very small (8-9 wei), this creates
            // significant relative precision errors
            upscaled[i] = _mulDown(amounts[i], scalingFactors[i]);
        }

        return upscaled;
    }

    /**
     * VULNERABILITY: Rounds down, causing invariant underestimation
     */
    function _mulDown(uint256 a, uint256 b) internal pure returns (uint256) {
        uint256 product = a * b;
        // Integer division rounds down - this is the vulnerability
        return product / _SCALING_FACTOR;
    }

    /**
     * VULNERABILITY: batchSwap allows chaining multiple swaps
     * Attacker can compound rounding losses across 65+ micro-swaps
     */
    function batchSwap(
        uint256[] memory amountsIn,
        uint256[] memory scalingFactors
    ) external returns (uint256[] memory) {
        // Scale input amounts
        uint256[] memory scaledAmounts = _upscaleArray(amountsIn, scalingFactors);

        // Process swaps (simplified)
        for (uint256 i = 0; i < scaledAmounts.length; i++) {
            // Each swap compounds the precision loss
            _processSwap(scaledAmounts[i]);
        }

        return scaledAmounts;
    }

    /**
     * VULNERABILITY: Invariant calculation using underestimated values
     * BPT price drops artificially, allowing extraction of value
     */
    function _calculateInvariant(uint256[] memory balances)
        internal
        pure
        returns (uint256)
    {
        uint256 sum = 0;
        for (uint256 i = 0; i < balances.length; i++) {
            // Invariant D is underestimated due to rounding errors
            sum += balances[i];
        }
        return sum;
    }

    function _processSwap(uint256 amount) internal {
        // Simplified swap logic
        totalSupply += amount;
    }

    /**
     * ADDITIONAL VULNERABILITY: No minimum balance checks
     * Allows balances to be driven to 8-9 wei range
     */
    function swap(
        address tokenIn,
        address tokenOut,
        uint256 amountIn
    ) external {
        // MISSING: Minimum balance validation
        // Should prevent balances from reaching precision-loss range

        balances[tokenIn] += amountIn;
        // No check if balances[tokenOut] drops to dangerous levels
        balances[tokenOut] -= amountIn;
    }
}

/**
 * ATTACK PATTERN:
 * 1. Deploy malicious contract with constructor that executes 65+ swaps
 * 2. Each swap uses tiny amounts (8-9 wei) to trigger precision loss
 * 3. _upscaleArray rounds down, underestimating invariant D
 * 4. BPT price drops artificially
 * 5. Redeem BPT at favorable rates, extracting value
 * 6. Repeat across multiple pools and chains
 *
 * IMPACT: $128M drained in 30 minutes across 6 chains
 *
 * FIX:
 * - Use SafeMath or require minimum balance thresholds
 * - Round up instead of down in critical calculations
 * - Add invariant validation after each operation
 * - Implement circuit breakers for unusual activity
 */
