"""
Tests for ExploitTool

Context: Validates safe, educational exploitation testing.
"""

import pytest
import tempfile
from pathlib import Path
from alprina_cli.tools.security.exploit import ExploitTool, ExploitParams
from alprina_cli.tools.base import ToolOk, ToolError


@pytest.mark.asyncio
async def test_exploit_tool_creation():
    """Test creating exploit tool"""
    tool = ExploitTool()
    assert tool.name == "Exploit"
    assert tool.params == ExploitParams


@pytest.mark.asyncio
async def test_exploit_sql_injection_detection():
    """Test SQL injection exploit detection"""
    tool = ExploitTool()

    with tempfile.NamedTemporaryFile(mode='w', suffix=".py", delete=False) as f:
        f.write('query = "SELECT * FROM users WHERE id = " + user_input\n')
        f.write('cursor.execute(query)\n')
        temp_file = Path(f.name)

    try:
        params = ExploitParams(
            target=str(temp_file),
            exploit_type="educational",
            vulnerability="sql_injection"
        )
        result = await tool.execute(params)

        assert isinstance(result, ToolOk)
        assert "safety_notice" in result.content
        assert result.content["vulnerability"] == "sql_injection"

        # Should find SQL injection vulnerability
        sql_results = [
            r for r in result.content["results"]
            if r.get("vulnerability") == "sql_injection"
        ]
        assert len(sql_results) > 0

        # Should have educational payloads
        assert "payloads_tested" in sql_results[0]
        assert len(sql_results[0]["payloads_tested"]) > 0

        # Safe mode - should not be successful
        assert sql_results[0]["successful"] is False

    finally:
        temp_file.unlink()


@pytest.mark.asyncio
async def test_exploit_xss_detection():
    """Test XSS exploit detection"""
    tool = ExploitTool()

    with tempfile.NamedTemporaryFile(mode='w', suffix=".js", delete=False) as f:
        f.write('element.innerHTML = userInput;\n')
        f.write('document.write(data);\n')
        temp_file = Path(f.name)

    try:
        params = ExploitParams(
            target=str(temp_file),
            exploit_type="educational",
            vulnerability="xss"
        )
        result = await tool.execute(params)

        assert isinstance(result, ToolOk)

        # Should find XSS vulnerabilities
        xss_results = [
            r for r in result.content["results"]
            if r.get("vulnerability") == "xss"
        ]
        assert len(xss_results) > 0

        # Should have XSS payloads
        assert "payloads_tested" in xss_results[0]
        assert any("<script>" in p for p in xss_results[0]["payloads_tested"])

    finally:
        temp_file.unlink()


@pytest.mark.asyncio
async def test_exploit_path_traversal_detection():
    """Test path traversal exploit detection"""
    tool = ExploitTool()

    with tempfile.NamedTemporaryFile(mode='w', suffix=".py", delete=False) as f:
        f.write('file_path = base_dir + user_path\n')
        f.write('with open(file_path) as f:\n')
        f.write('    data = f.read()\n')
        temp_file = Path(f.name)

    try:
        params = ExploitParams(
            target=str(temp_file),
            exploit_type="educational",
            vulnerability="path_traversal"
        )
        result = await tool.execute(params)

        assert isinstance(result, ToolOk)

        # Should find path traversal vulnerabilities
        traversal_results = [
            r for r in result.content["results"]
            if r.get("vulnerability") == "path_traversal"
        ]
        assert len(traversal_results) > 0

        # Should have traversal payloads
        assert "payloads_tested" in traversal_results[0]
        assert any("../" in p for p in traversal_results[0]["payloads_tested"])

    finally:
        temp_file.unlink()


@pytest.mark.asyncio
async def test_exploit_command_injection_detection():
    """Test command injection exploit detection"""
    tool = ExploitTool()

    with tempfile.NamedTemporaryFile(mode='w', suffix=".py", delete=False) as f:
        f.write('import os\n')
        f.write('os.system("ping " + user_input)\n')
        f.write('eval(code)\n')
        temp_file = Path(f.name)

    try:
        params = ExploitParams(
            target=str(temp_file),
            exploit_type="educational",
            vulnerability="command_injection"
        )
        result = await tool.execute(params)

        assert isinstance(result, ToolOk)

        # Should find command injection vulnerabilities
        cmd_results = [
            r for r in result.content["results"]
            if r.get("vulnerability") == "command_injection"
        ]
        assert len(cmd_results) > 0

        # Should have command injection payloads
        assert "payloads_tested" in cmd_results[0]

    finally:
        temp_file.unlink()


@pytest.mark.asyncio
async def test_exploit_safe_mode():
    """Test safe mode never reports successful exploits"""
    tool = ExploitTool()

    with tempfile.NamedTemporaryFile(mode='w', suffix=".py", delete=False) as f:
        f.write('query = "SELECT * FROM users WHERE id = " + user_input\n')
        f.write('os.system("ls " + user_input)\n')
        temp_file = Path(f.name)

    try:
        params = ExploitParams(
            target=str(temp_file),
            exploit_type="safe"  # Safe mode
        )
        result = await tool.execute(params)

        assert isinstance(result, ToolOk)

        # In safe mode, no exploits should be successful
        for r in result.content["results"]:
            if r.get("vulnerability") != "insecure_protocol":  # Except protocol check
                assert r.get("successful") is False

    finally:
        temp_file.unlink()


@pytest.mark.asyncio
async def test_exploit_educational_mode():
    """Test educational mode includes learning notes"""
    tool = ExploitTool()

    with tempfile.NamedTemporaryFile(mode='w', suffix=".py", delete=False) as f:
        f.write('password = "test"\n')
        f.write('query = "SELECT * FROM users WHERE id = " + user_input\n')
        temp_file = Path(f.name)

    try:
        params = ExploitParams(
            target=str(temp_file),
            exploit_type="educational"
        )
        result = await tool.execute(params)

        assert isinstance(result, ToolOk)

        # Educational mode should include notes
        if len(result.content["results"]) > 0:
            assert any(
                "educational_note" in r
                for r in result.content["results"]
            )

    finally:
        temp_file.unlink()


@pytest.mark.asyncio
async def test_exploit_all_vulnerabilities():
    """Test scanning for all vulnerability types"""
    tool = ExploitTool()

    with tempfile.NamedTemporaryFile(mode='w', suffix=".py", delete=False) as f:
        # Multiple vulnerability types
        f.write('query = "SELECT * FROM users WHERE id = " + user_input\n')
        f.write('os.system("ls " + user_input)\n')
        f.write('file_path = base + user_path\n')
        f.write('open(file_path)\n')
        temp_file = Path(f.name)

    try:
        params = ExploitParams(
            target=str(temp_file),
            exploit_type="educational",
            vulnerability=""  # Empty means all types
        )
        result = await tool.execute(params)

        assert isinstance(result, ToolOk)

        # Should find multiple vulnerability types
        vuln_types = set(r.get("vulnerability") for r in result.content["results"])
        assert len(vuln_types) > 1

    finally:
        temp_file.unlink()


@pytest.mark.asyncio
async def test_exploit_max_attempts():
    """Test max_attempts parameter limits results"""
    tool = ExploitTool()

    with tempfile.TemporaryDirectory() as temp_dir:
        # Create multiple vulnerable files
        for i in range(10):
            (Path(temp_dir) / f"file{i}.py").write_text(
                'query = "SELECT * FROM users WHERE id = " + user_input\n'
            )

        params = ExploitParams(
            target=temp_dir,
            exploit_type="safe",
            max_attempts=3
        )
        result = await tool.execute(params)

        assert isinstance(result, ToolOk)
        assert len(result.content["results"]) <= 3
        if result.content["summary"]["total_attempts"] > 3:
            assert result.content["summary"]["truncated"] is True


@pytest.mark.asyncio
async def test_exploit_directory():
    """Test exploiting entire directory"""
    tool = ExploitTool()

    with tempfile.TemporaryDirectory() as temp_dir:
        # Create multiple files
        (Path(temp_dir) / "app.py").write_text(
            'query = "SELECT * FROM users WHERE id = " + user_input\n'
        )
        (Path(temp_dir) / "utils.py").write_text(
            'os.system("ping " + host)\n'
        )

        params = ExploitParams(
            target=temp_dir,
            exploit_type="educational"
        )
        result = await tool.execute(params)

        assert isinstance(result, ToolOk)
        assert result.content["summary"]["target_type"] == "local"
        assert len(result.content["results"]) > 0


@pytest.mark.asyncio
async def test_exploit_remote_http():
    """Test remote exploit with HTTP target"""
    tool = ExploitTool()

    params = ExploitParams(
        target="http://example.com",
        exploit_type="educational"
    )
    result = await tool.execute(params)

    assert isinstance(result, ToolOk)
    assert result.content["summary"]["target_type"] == "remote"

    # Should detect insecure HTTP
    insecure_results = [
        r for r in result.content["results"]
        if r.get("vulnerability") == "insecure_protocol"
    ]
    assert len(insecure_results) > 0
    assert insecure_results[0]["successful"] is True


@pytest.mark.asyncio
async def test_exploit_summary_statistics():
    """Test summary statistics calculation"""
    tool = ExploitTool()

    with tempfile.NamedTemporaryFile(mode='w', suffix=".py", delete=False) as f:
        f.write('query = "SELECT * FROM users WHERE id = " + user_input\n')
        temp_file = Path(f.name)

    try:
        params = ExploitParams(
            target=str(temp_file),
            exploit_type="educational"
        )
        result = await tool.execute(params)

        assert isinstance(result, ToolOk)

        summary = result.content["summary"]
        assert "total_attempts" in summary
        assert "successful" in summary
        assert "success_rate" in summary
        assert 0 <= summary["success_rate"] <= 100

    finally:
        temp_file.unlink()


@pytest.mark.asyncio
async def test_exploit_safety_notice():
    """Test that safety notice is always included"""
    tool = ExploitTool()

    with tempfile.NamedTemporaryFile(mode='w', suffix=".py", delete=False) as f:
        f.write('password = "test"\n')
        temp_file = Path(f.name)

    try:
        params = ExploitParams(
            target=str(temp_file),
            exploit_type="safe"
        )
        result = await tool.execute(params)

        assert isinstance(result, ToolOk)
        assert "safety_notice" in result.content
        assert "authorized" in result.content["safety_notice"].lower()

    finally:
        temp_file.unlink()


def test_exploit_to_dict():
    """Test tool serialization"""
    tool = ExploitTool()
    tool_dict = tool.to_dict()

    assert tool_dict["name"] == "Exploit"
    assert "target" in tool_dict["parameters"]["properties"]
    assert "exploit_type" in tool_dict["parameters"]["properties"]
