from .abstract_stopping_criterion import AbstractStoppingCriterion
from ..util.data import Data

from ..discrete_distribution import IIDStdUniform
from ..discrete_distribution.abstract_discrete_distribution import AbstractIIDDiscreteDistribution
from ..true_measure import Gaussian,Uniform
from ..integrand import Keister,BoxIntegral,CustomFun,Genz,SensitivityIndices
from ..util import MaxSamplesWarning, ParameterWarning, ParameterError
import numpy as np
from time import time
from scipy.stats import norm
import warnings


class CubMCCLTVec(AbstractStoppingCriterion):
    r"""
    IID Monte Carlo stopping criterion stopping criterion based on the Central Limit Theorem with doubling sample sizes.
    
    Examples:
        >>> k = Keister(IIDStdUniform(seed=7))
        >>> sc = CubMCCLTVec(k,abs_tol=5e-2,rel_tol=0)
        >>> solution,data = sc.integrate()
        >>> solution
        array(1.38366791)
        >>> data
        Data (Data)
            solution        1.384
            comb_bound_low  1.343
            comb_bound_high 1.424
            comb_bound_diff 0.080
            comb_flags      1
            n_total         1024
            n               2^(10)
            time_integrate  ...
        CubMCCLTVec (AbstractStoppingCriterion)
            inflate         1
            alpha           0.010
            abs_tol         0.050
            rel_tol         0
            n_init          2^(8)
            n_limit         2^(30)
        Keister (AbstractIntegrand)
        Gaussian (AbstractTrueMeasure)
            mean            0
            covariance      2^(-1)
            decomp_type     PCA
        IIDStdUniform (AbstractIIDDiscreteDistribution)
            d               1
            replications    1
            entropy         7

        Vector outputs
        
        >>> f = BoxIntegral(IIDStdUniform(3,seed=7),s=[-1,1])
        >>> abs_tol = 2.5e-2
        >>> sc = CubMCCLTVec(f,abs_tol=abs_tol,rel_tol=0)
        >>> solution,data = sc.integrate()
        >>> solution
        array([1.18448043, 0.95435347])
        >>> data
        Data (Data)
            solution        [1.184 0.954]
            comb_bound_low  [1.165 0.932]
            comb_bound_high [1.203 0.977]
            comb_bound_diff [0.038 0.045]
            comb_flags      [ True  True]
            n_total         8192
            n               [8192 1024]
            time_integrate  ...
        CubMCCLTVec (AbstractStoppingCriterion)
            inflate         1
            alpha           0.010
            abs_tol         0.025
            rel_tol         0
            n_init          2^(8)
            n_limit         2^(30)
        BoxIntegral (AbstractIntegrand)
            s               [-1  1]
        Uniform (AbstractTrueMeasure)
            lower_bound     0
            upper_bound     1
        IIDStdUniform (AbstractIIDDiscreteDistribution)
            d               3
            replications    1
            entropy         7
        >>> sol3neg1 = -np.pi/4-1/2*np.log(2)+np.log(5+3*np.sqrt(3))
        >>> sol31 = np.sqrt(3)/4+1/2*np.log(2+np.sqrt(3))-np.pi/24
        >>> true_value = np.array([sol3neg1,sol31])
        >>> assert (abs(true_value-solution)<abs_tol).all()
        
        Sensitivity indices 

        >>> function = Genz(IIDStdUniform(3,seed=7))
        >>> integrand = SensitivityIndices(function)
        >>> sc = CubMCCLTVec(integrand,abs_tol=2.5e-2,rel_tol=0)
        >>> solution,data = sc.integrate()
        >>> data
        Data (Data)
            solution        [[0.024 0.203 0.662]
                             [0.044 0.308 0.78 ]]
            comb_bound_low  [[0.006 0.186 0.644]
                             [0.02  0.286 0.761]]
            comb_bound_high [[0.042 0.221 0.681]
                             [0.067 0.329 0.798]]
            comb_bound_diff [[0.036 0.035 0.037]
                             [0.047 0.043 0.037]]
            comb_flags      [[ True  True  True]
                             [ True  True  True]]
            n_total         262144
            n               [[[  4096  65536 262144]
                              [  4096  65536 262144]
                              [  4096  65536 262144]]
        <BLANKLINE>
                             [[   512  32768 262144]
                              [   512  32768 262144]
                              [   512  32768 262144]]]
            time_integrate  ...
        CubMCCLTVec (AbstractStoppingCriterion)
            inflate         1
            alpha           0.010
            abs_tol         0.025
            rel_tol         0
            n_init          2^(8)
            n_limit         2^(30)
        SensitivityIndices (AbstractIntegrand)
            indices         [[ True False False]
                             [False  True False]
                             [False False  True]]
        Uniform (AbstractTrueMeasure)
            lower_bound     0
            upper_bound     1
        IIDStdUniform (AbstractIIDDiscreteDistribution)
            d               3
            replications    1
            entropy         7
    """

    def __init__(self,
                 integrand, 
                 abs_tol = 1e-2,
                 rel_tol = 0.,
                 n_init = 256.,
                 n_limit = 2**30,
                 error_fun = "EITHER",
                 inflate = 1,
                 alpha = 0.01,
                 ):
        r"""
        Args:
            integrand (AbstractIntegrand): The integrand.
            abs_tol (np.ndarray): Absolute error tolerance.
            rel_tol (np.ndarray): Relative error tolerance.
            n_init (int): Initial number of samples. 
            n_limit (int): Maximum number of samples.
            error_fun (Union[str,callable]): Function mapping the approximate solution, absolute error tolerance, and relative error tolerance to the current error bound.

                - `'EITHER'`, the default, requires the approximation error must be below either the absolue *or* relative tolerance.
                    Equivalent to setting
                    ```python
                    error_fun = lambda sv,abs_tol,rel_tol: np.maximum(abs_tol,abs(sv)*rel_tol)
                    ```
                - `'BOTH'` requires the approximation error to be below both the absolue *and* relative tolerance. 
                    Equivalent to setting
                    ```python
                    error_fun = lambda sv,abs_tol,rel_tol: np.minimum(abs_tol,abs(sv)*rel_tol)
                    ```
            inflate (float): Inflation factor $\geq 1$ to multiply by the variance estimate to make it more conservative.
            alpha (np.ndarray): Uncertainty level in $(0,1)$. 
        """
        self.parameters = ['inflate','alpha','abs_tol','rel_tol','n_init','n_limit']
        # Input Checks
        if np.log2(n_init)%1!=0:
            warnings.warn('n_init must be a power of two. Using n_init = 2**5',ParameterWarning)
            n_init = 2**5
        if np.log2(n_limit)%1!=0:
            warnings.warn('n_init must be a power of two. Using n_limit = 2**30',ParameterWarning)
            n_limit = 2**30
        # Set Attributes
        self.n_init = int(n_init)
        self.n_limit = int(n_limit)
        assert isinstance(error_fun,str) or callable(error_fun)
        if isinstance(error_fun,str):
            if error_fun.upper()=="EITHER":
                error_fun = lambda sv,abs_tol,rel_tol: np.maximum(abs_tol,abs(sv)*rel_tol)
            elif error_fun.upper()=="BOTH":
                error_fun = lambda sv,abs_tol,rel_tol: np.minimum(abs_tol,abs(sv)*rel_tol)
            else:
                raise ParameterError("str error_fun must be 'EITHER' or 'BOTH'")
        self.error_fun = error_fun
        self.alpha = alpha
        self.inflate = float(inflate)
        assert self.inflate>=1
        # QMCPy Objs
        self.integrand = integrand
        self.true_measure = self.integrand.true_measure
        self.discrete_distrib = self.integrand.discrete_distrib
        super(CubMCCLTVec,self).__init__(allowed_distribs=[AbstractIIDDiscreteDistribution],allow_vectorized_integrals=True)
        assert self.integrand.discrete_distrib.no_replications==True, "Require the discrete distribution has replications=None"
        self.alphas_indv,identity_dependency = self._compute_indv_alphas(np.full(self.integrand.d_comb,self.alpha))
        self.set_tolerance(abs_tol,rel_tol)
        self.z_star = -norm.ppf(self.alphas_indv/2)
    
    def integrate(self):
        t_start = time()
        data = Data(
            parameters = [
                'solution',
                'comb_bound_low',
                'comb_bound_high',
                'comb_bound_diff',
                'comb_flags',
                'n_total',
                'n',
                'time_integrate'])
        data.flags_indv = np.tile(False,self.integrand.d_indv)
        data.compute_flags = np.tile(True,self.integrand.d_indv)
        data.n = np.tile(self.n_init,self.integrand.d_indv)
        data.n_min = 0
        data.n_max = self.n_init
        data.solution_indv = np.tile(np.nan,self.integrand.d_indv)
        data.xfull = np.empty((0,self.integrand.d))
        data.yfull = np.empty(self.integrand.d_indv+(0,))
        while True:
            xnext = self.discrete_distrib(n=data.n_max-data.n_min)
            data.xfull = np.concatenate([data.xfull,xnext],0)
            ynext = self.integrand.f(xnext,compute_flags=data.compute_flags)
            ynext[~data.compute_flags] = np.nan
            data.yfull = np.concatenate([data.yfull,ynext],-1)
            data.n[data.compute_flags] = data.n_max
            yfullfinite = np.isfinite(data.yfull)
            data.solution_indv = data.yfull.mean(-1,where=yfullfinite)
            data.sigmahat = data.yfull.std(-1,ddof=1,where=yfullfinite)
            data.ci_half_width = self.z_star*self.inflate*data.sigmahat/np.sqrt(data.n)
            data.indv_bound_low = data.solution_indv-data.ci_half_width
            data.indv_bound_high = data.solution_indv+data.ci_half_width
            data.n_total = data.n.max() 
            data.comb_bound_low,data.comb_bound_high = self.integrand.bound_fun(data.indv_bound_low,data.indv_bound_high)
            data.comb_bound_diff = data.comb_bound_high-data.comb_bound_low
            fidxs = np.isfinite(data.comb_bound_low)&np.isfinite(data.comb_bound_high)
            slow,shigh,abs_tols,rel_tols = data.comb_bound_low[fidxs],data.comb_bound_high[fidxs],self.abs_tols[fidxs],self.rel_tols[fidxs]
            data.solution = np.tile(np.nan,data.comb_bound_low.shape)
            data.solution[fidxs] = 1/2*(slow+shigh+self.error_fun(slow,abs_tols,rel_tols)-self.error_fun(shigh,abs_tols,rel_tols))
            data.comb_flags = np.tile(False,data.comb_bound_low.shape)
            data.comb_flags[fidxs] = (shigh-slow) <= (self.error_fun(slow,abs_tols,rel_tols)+self.error_fun(shigh,abs_tols,rel_tols))
            data.flags_indv = self.integrand.dependency(data.comb_flags)
            data.compute_flags = ~data.flags_indv
            if np.sum(data.compute_flags)==0:
                break # sufficiently estimated
            elif 2*data.n_total>self.n_limit:
                warning_s = """
                Already generated %d samples.
                Trying to generate %d new samples would exceeds n_limit = %d.
                No more samples will be generated.
                Note that error tolerances may not be satisfied. """ \
                % (int(data.n_total),int(data.n_total),int(self.n_limit))
                warnings.warn(warning_s, MaxSamplesWarning)
                break
            data.n_min = data.n_max
            data.n_max = 2*data.n_min
        data.stopping_crit = self
        data.integrand = self.integrand
        data.true_measure = self.integrand.true_measure
        data.discrete_distrib = self.true_measure.discrete_distrib
        data.time_integrate = time()-t_start
        return data.solution,data
    
    def set_tolerance(self, abs_tol=None, rel_tol=None, rmse_tol=None):
        assert rmse_tol is None, "rmse_tol not supported by this stopping criterion."
        if abs_tol is not None:
            self.abs_tol = abs_tol
            self.abs_tols = np.full(self.integrand.d_comb,self.abs_tol)
        if rel_tol is not None:
            self.rel_tol = rel_tol
            self.rel_tols = np.full(self.integrand.d_comb,self.rel_tol)    
