from __future__ import annotations

import os
import shutil
import signal
import subprocess
import threading
import uuid
import webbrowser
from typing import Optional, Sequence

import click
import importlib_metadata
from packaging.version import Version
from rich import print
from rich.progress import Progress, SpinnerColumn
from rich.prompt import Confirm
from urllib3.util import parse_url

import coiled
from coiled.cli.cluster.ssh import add_key_to_agent, check_ssh
from coiled.compatibility import DISTRIBUTED_VERSION
from coiled.errors import DoesNotExist

from ..utils import CONTEXT_SETTINGS

# Path on VM to sync to.
# We use `/scratch` for now because it's already bind-mounted into docker.
SYNC_TARGET = "/scratch/synced"
MIN_DISTRIBUTED_VERSION = Version("2022.8.1")
MUTAGEN_NAME_FORMAT = "coiled-{cluster_id}"


def check_distributed_version() -> bool:
    if DISTRIBUTED_VERSION < MIN_DISTRIBUTED_VERSION:
        print(
            "[bold red]"
            f"distributed>{MIN_DISTRIBUTED_VERSION} is required to launch notebooks. "
            f"You have {DISTRIBUTED_VERSION}."
            "[/]"
        )
        return False
    return True


def check_jupyter() -> bool:
    try:
        importlib_metadata.distribution("jupyter_server")
    except ModuleNotFoundError:
        print("[bold red]Jupyter must be installed locally to launch notebooks.[/]")
        return False

    try:
        importlib_metadata.distribution("jupyter_server_proxy")
    except ModuleNotFoundError:
        print(
            "[bold red]jupyter-server-proxy is not installed, "
            "without this you won't be able to access Dask dashboard for local clusters created on notebook server.[/]"
        )

    return True


def check_mutagen() -> bool:
    if not shutil.which("mutagen"):
        print(
            "[bold red]"
            "mutagen must be installed to synchronize files with notebooks.[/]\n"
            "Install via homebrew (on macOS, Linux, or Windows) with:\n\n"
            "brew install mutagen-io/mutagen/mutagen@0.16\n\n"
            "Or, visit https://github.com/mutagen-io/mutagen/releases/latest to download "
            "a static, pre-compiled binary for your system, and place it anywhere on your $PATH."
        )
        return False
    return True


def check_ssh_keygen() -> bool:
    if not shutil.which("ssh-keygen"):
        print("[bold red]Unable to find `ssh-keygen`, you may need to install OpenSSH or add it to your paths.[/]")
        return False
    return True


def mutagen_session_exists(cluster_id: int) -> bool:
    sessions = (
        subprocess.run(
            [
                "mutagen",
                "sync",
                "list",
                "--label-selector",
                f"managed-by=coiled,cluster-id={cluster_id}",
                "--template",
                "{{range .}}{{.Name}}{{end}}",
            ],
            check=True,
            text=True,
            capture_output=True,
        )
        .stdout.strip()
        .splitlines()
    )

    if not sessions:
        return False
    if sessions == [MUTAGEN_NAME_FORMAT.format(cluster_id=cluster_id)]:
        return True

    if len(sessions) == 1:
        raise RuntimeError(
            f"Unexpected mutagen session name {sessions[0]!r}. "
            f"Expected {MUTAGEN_NAME_FORMAT.format(cluster_id=cluster_id)!r}."
        )

    raise RuntimeError(f"Multiple mutagen sessions found for cluster {cluster_id}: {sessions}")


@click.command(context_settings=CONTEXT_SETTINGS)
@click.option(
    "--name",
    default=None,
    help="Cluster name. If not given, defaults to a hash based on current working directory.",
)
@click.option(
    "--account",
    default=None,
    help="Coiled account (uses default account if not specified)",
)
@click.option(
    "--sync",
    default=False,
    is_flag=True,
    help="Sync the working directory with the filesystem on the notebook. Requires mutagen.",
)
@click.option(
    "--software",
    default=None,
    help=(
        "Software environment name to use. If neither software nor container is specified, "
        "all the currently-installed Python packages are replicated on the VM using package sync."
    ),
)
@click.option(
    "--container",
    default=None,
    help=(
        "Container image to use. If neither software nor container is specified, "
        "all the currently-installed Python packages are replicated on the VM using package sync."
    ),
)
@click.option(
    "--vm-type",
    default=[],
    multiple=True,
    help="VM type to use. Specify multiple times to provide multiple options.",
)
@click.option(
    "--gpu",
    default=False,
    is_flag=True,
    help="Use GPU notebook server.",
)
@click.option(
    "--region",
    default=None,
    help="The cloud provider region in which to run the notebook.",
)
@click.option(
    "--open",
    default=True,
    is_flag=True,
    help="Whether to open the notebook in the default browser once it's launched",
)
@click.option(
    "--block/--no-block",
    default=True,
    is_flag=True,
    help="Whether to block while the notebook is running.",
)
def start_notebook(
    name: Optional[str],
    account: Optional[str],
    sync: bool,
    software: Optional[str],
    container: Optional[str],
    vm_type: Sequence[str],
    gpu: bool,
    region: Optional[str],
    open: bool,
    block: bool,
):
    """
    Launch or re-open a notebook session, with optional file syncing.

    .. warning::

        ``coiled notebook`` is an experimental feature and is subject to breaking changes.

    If a notebook session with the same ``name`` already exists, it's not re-created.
    If file sync was initially not enabled, running ``coiled notebook start --sync``
    will begin file sync without re-launching the notebook.
    """

    # when using package sync, check that local env has jupyter and recent distributed
    if not software:
        if not (check_distributed_version() and check_jupyter()):
            return

    if sync and not (check_mutagen() and check_ssh() and check_ssh_keygen()):
        return

    env = None
    if container and "rapidsai" in container:
        env = {"DISABLE_JUPYTER": "true"}  # needed for "stable" RAPIDS image

    name = name or f"notebook-{uuid.uuid4().hex[:8]}"
    with coiled.Cloud(account=account) as cloud:
        print(f"Starting notebook [bold]{name}[/bold]...")
        # TODO how can we get the widget to show up during CLI commands?
        cluster = coiled.Cluster(
            name=name,
            cloud=cloud,
            n_workers=0,
            software=software,
            container=container,
            jupyter=True,
            scheduler_options={"idle_timeout": "24 hours"},
            scheduler_vm_types=list(vm_type) if vm_type else None,
            worker_vm_types=list(vm_type) if vm_type else None,
            allow_ssh=True,
            environ=env,
            scheduler_gpu=gpu,
            region=region,
            tags={"coiled-cluster-type": "notebook"},
        )

        url = cluster.jupyter_link
        cluster_id = cluster.cluster_id
        assert cluster_id is not None

        # by default, jupyter on the scheduler gives us client to that very scheduler
        # clear ENV var so default `Client()` on notebook gives us a new local cluster
        with cluster.get_client() as client:

            def _():
                import os

                del os.environ["DASK_SCHEDULER_ADDRESS"]

            client.run_on_scheduler(_)

        if sync:
            url = parse_url(url)._replace(path="/jupyter/lab/tree/synced").url

            if mutagen_session_exists(cluster_id):
                print("[bold]File sync session already active; reusing it.[/]")
            else:
                print("[bold]Launching file synchronization...[/]")
                ssh_info = cloud.get_ssh_key(cluster_id)

                scheduler_address = ssh_info["scheduler_public_address"]
                target = f"ubuntu@{scheduler_address}"

                add_key_to_agent(scheduler_address, key=ssh_info["private_key"])

                # Update known_hosts. We can't specify SSH options to mutagen so we can't pass
                # `-o StrictHostKeyChecking=no`. Could alternatively add an entry in `~/.ssh/config`,
                # but that feels more intrusive.
                # TODO get public key from Coiled
                subprocess.run(
                    f"ssh-keyscan {scheduler_address} >> ~/.ssh/known_hosts",
                    shell=True,
                    check=True,
                    capture_output=True,
                )

                # Start mutagen
                subprocess.run(
                    [
                        "mutagen",
                        "sync",
                        "create",
                        "--name",
                        MUTAGEN_NAME_FORMAT.format(cluster_id=cluster_id),
                        "--label",
                        "managed-by=coiled",
                        "--label",
                        f"cluster-id={cluster_id}",
                        "--ignore-vcs",
                        "--max-staging-file-size=1 GiB",
                        ".",
                        f"{target}:{SYNC_TARGET}",
                    ],
                    check=True,
                )

                # Within the docker container, symlink the sync directory (`/scratch/sync`)
                # into the working directory for Jupyter, so you can actually see the synced
                # files in the Jupyter browser. We use a symlink since the container doesn't
                # have capabilities to make a bind mount.
                # TODO if we don't like the symlink, Coiled could see what the workdir is for
                # the image before running, and bind-mount `/sync` on the host to `$workdir/sync`
                # in the container? Custom docker images make this tricky; we can't assume anything
                # about the directory layout or what the working directory will be.
                subprocess.run(
                    [
                        "ssh",
                        target,
                        f"docker exec tmp-dask-1 bash -c 'mkdir -p {SYNC_TARGET} && ln -s {SYNC_TARGET} .'",
                    ],
                    check=True,
                )

    print(f"[bold]Jupyter available at {url}[/]")
    print()

    if open:
        webbrowser.open(url, new=2)

    if block:
        print("[green]Use Control-C to stop this notebook server [/]")
        with Progress(SpinnerColumn()) as progress:
            # Wait for shutdown signal from user

            def signal_handler_noop(_, frame):
                # Ignore the input signal
                return

            def signal_handler(_, frame):
                progress.stop()
                # Restore original handler so KeyboardInterrupt show up as exceptions
                signal.signal(signal.SIGINT, original_handler)
                try:
                    exit = Confirm.ask("Are you sure you want to stop this notebook server?", default=True)
                except KeyboardInterrupt:
                    # Register noop handler since we're shutting down and
                    # want to make sure the notebook is shutdown even when
                    # hammering ctrl-C
                    signal.signal(signal.SIGINT, signal_handler_noop)
                    exit = True
                if exit:
                    interrupt.set()
                else:
                    signal.signal(signal.SIGINT, signal_handler)
                    print("[green]Continuing with this notebook server... [/]")
                    progress.start()

            original_handler = signal.getsignal(signal.SIGINT)
            signal.signal(signal.SIGINT, signal_handler)
            interrupt = threading.Event()
            progress.add_task("spinning...", total=None)  # starts an infinite spinner
            interrupt.wait()

        # `.callback` gets the underlying function from the click command.
        # It can sometimes be `None`. Not sure why, but we handle that case here
        # to make type checking happy.
        stop_func = stop_notebook.callback
        if stop_func is not None:
            stop_func(name, account)
    else:
        stop_command = "coiled notebook stop"
        if account:
            stop_command = f"{stop_command} --account {account}"
        stop_command = f"{stop_command} {name}"

        print(f"To stop this notebook server: [green]{stop_command}[/]")


@click.command(context_settings=CONTEXT_SETTINGS)
@click.argument("name")
@click.option(
    "--account",
    default=None,
    help="Coiled account (uses default account if not specified)",
)
def stop_notebook(name: str, account: Optional[str]):
    """
    Shut down a notebook session

    .. warning::

        ``coiled notebook`` is an experimental feature and is subject to breaking changes.
    """
    with coiled.Cloud(account=account) as cloud:
        try:
            cluster_id = cloud.get_cluster_by_name(name)
        except DoesNotExist:
            print(f"[bold red]Notebook {name!r} does not exist[/]")
            return  # TODO exit 1

        if shutil.which("mutagen") and mutagen_session_exists(cluster_id):
            # NOTE: we can't tell if the user asked for `--sync` or not at creation.
            # Best we can do is check if mutagen is installed and the session exists.
            if not (check_ssh() and check_ssh_keygen()):
                return

            # Stop mutagen
            print(f"Stopping sync with notebook {name!r} ({cluster_id})")
            subprocess.run(
                ["mutagen", "sync", "terminate", MUTAGEN_NAME_FORMAT.format(cluster_id=cluster_id)],
                check=True,
            )

            ssh_info = cloud.get_ssh_key(cluster_id)
            scheduler_address = ssh_info["scheduler_public_address"]
            add_key_to_agent(scheduler_address, key=ssh_info["private_key"], delete=True)

            # Remove `known_hosts` entries.
            # TODO don't like touching the user's `known_hosts` file like this.
            subprocess.run(
                [
                    "ssh-keygen",
                    "-f",
                    os.path.expanduser("~/.ssh/known_hosts"),
                    "-R",
                    scheduler_address,
                ]
            )

        print(f"Stopping notebook {name!r} ({cluster_id})...")
        cloud.delete_cluster(cluster_id)


@click.command(context_settings=CONTEXT_SETTINGS)
@click.argument("name")
def monitor_sync(name: str):
    "Monitor file sync status for a notebook session."
    if not check_mutagen():
        return

    with coiled.Cloud() as cloud:
        try:
            cluster_id = cloud.get_cluster_by_name(name)
        except DoesNotExist:
            print(f"[bold red]Cluster {name!r} does not exist[/]")
            return  # TODO exit 1

    if not mutagen_session_exists(cluster_id):
        print(f"[bold red]No file synchronization session for cluster {name!r} ({cluster_id})[/]")
        return  # TODO exit 1

    subprocess.run(["mutagen", "sync", "monitor", MUTAGEN_NAME_FORMAT.format(cluster_id=cluster_id)])
