import logging
import time

from dask.utils import parse_timedelta
from distributed import SchedulerPlugin

try:
    from distributed.compatibility import PeriodicCallback  # pyright: reportPrivateImportUsage=false
except ImportError:
    # For `distributed<2022.10.1` (xref https://github.com/dask/distributed/pull/7165)
    from tornado.ioloop import PeriodicCallback

logger = logging.getLogger("distributed.scheduler")


class NoClientShutdown(SchedulerPlugin):
    """Shut down a scheduler if all of the clients have gone"""

    name = "NoClientShutdown"

    def __init__(self, keepalive=None, interval="1 second"):
        self.started = False
        self.keepalive = parse_timedelta(keepalive) or 0
        self.interval = parse_timedelta(interval)
        self.countdown = False
        self.countdown_from_time = 0

    async def start(self, scheduler):
        self.scheduler = scheduler
        if self.scheduler.tasks or set(self.scheduler.clients) - {"fire-and-forget"}:
            self.started = True

        pc = PeriodicCallback(self.check, self.interval * 1000)
        pc.start()
        scheduler.periodic_callbacks["no-client-shutdown"] = pc

    def add_client(self, scheduler, client):
        """On adding a client, reset any planned death"""
        self.started = True
        self.countdown = False
        self.countdown_from_time = 0

    def check(self):
        """Check if we should kill the scheduler"""
        if not self.started:
            return
        if not self.scheduler.tasks and set(self.scheduler.clients) == {"fire-and-forget"}:
            if not self.countdown:
                self.countdown = True
                self.countdown_from_time = time.monotonic()
        else:
            self.countdown = False
            self.countdown_from_time = 0

        if self.countdown and time.monotonic() > self.countdown_from_time + self.keepalive:
            logger.info(
                f"Shutting down scheduler, last active client {time.monotonic()-self.countdown_from_time:.2f}s ago "
                f"with keepalive={self.keepalive}s"
            )
            # scheduler._ongoing_background_tasks requires distributed>=2022.6.1
            self.scheduler._ongoing_background_tasks.call_soon(self.scheduler.close)
            self.stop()

    def stop(self):
        self.scheduler.periodic_callbacks["no-client-shutdown"].stop()
