# Infosourceful Kernel

Shared models, JSON Schemas, and a CLI validator for the Infosourceful ecosystem.

## Overview

Infosourceful Kernel serves as the single source of truth for domain models, JSON schemas, and validation utilities used across the Infosourceful platform. It provides:

- **Domain Models**: Pydantic models for artifacts, decisions, relations, and actors
- **JSON Schemas**: Versioned meta schemas for artifact validation
- **CLI Validator**: Command-line tool for validating artifacts
- **Validation Utilities**: Reusable validation functions for applications

## Installation

### From PyPI (Recommended)

```bash
pip install infosourceful-kernel
```

### From GitHub Packages

If the package is published to GitHub Packages, you can install it using:

```bash
# Configure pip to use GitHub Packages
pip install --index-url https://pypi.org/simple/ --extra-index-url https://pypi.github.com/infosourceful/infosourceful-kernel/simple/ infosourceful-kernel
```

### Install Specific Version

```bash
# Install latest stable version
pip install infosourceful-kernel==1.0.0

# Install from source (development)
pip install git+https://github.com/Infosourceful/infosourceful-kernel.git@v1.0.0
```

### Development Installation

For development or to contribute:

```bash
git clone https://github.com/Infosourceful/infosourceful-kernel.git
cd infosourceful-kernel
pip install -e .
```

## Usage

### CLI Validation

The `ik` command provides artifact validation:

```bash
# Validate a JSON file
ik artifact.json

# Validate from stdin
echo '{"schema_version": "1.0.0", ...}' | ik

# Validate with specific schema version
ik artifact.json --schema-version 1.0

# Validate YAML files
ik artifact.yaml --format yaml

# Verbose output
ik artifact.json --verbose
```

### Python API

```python
from infosourceful_kernel.models import Artifact, ArtifactType
from infosourceful_kernel.validation import validate_artifact_json

# Create an artifact
artifact = Artifact(
    schema_version="1.0.0",
    artifact_id="my_artifact_123",
    artifact_type=ArtifactType.DECK,
    project_key="my_project",
    source_systems=["system1", "system2"],
    hash="abcdef1234567890abcdef1234567890"
)

# Validate JSON content
result = validate_artifact_json(json_content)
if result.is_valid:
    print("✅ Validation passed")
else:
    print("❌ Validation failed:", result.errors)
```

## Schema Versioning

Infosourceful Kernel follows semantic versioning for schemas:

- **MAJOR**: Breaking schema/API changes
- **MINOR**: Additive (non-breaking) changes
- **PATCH**: Bug fixes only

Schema files are stored as `meta-v{N}.0.json` in the `schemas/` directory, with major versions kept side-by-side for backward compatibility.

## Domain Models

### Artifact

The core model representing any artifact in the Infosourceful system:

```python
from infosourceful_kernel.models import Artifact, ArtifactType

artifact = Artifact(
    schema_version="1.0.0",
    artifact_id="unique_id_123",
    artifact_type=ArtifactType.DECK,
    project_key="project_name",
    source_systems=["system1"],
    hash="content_hash_here"
)
```

### Actor

Represents people or systems involved in the process:

```python
from infosourceful_kernel.models import Actor, ActorRole

actor = Actor(
    name="John Doe",
    role=ActorRole.OWNER,
    email="john@example.com"
)
```

### Decision

Represents decisions made during the process:

```python
from infosourceful_kernel.models import Decision, DecisionStatus
from datetime import date

decision = Decision(
    id="dec_001",
    title="Approval Decision",
    date=date(2024, 1, 15),
    rationale="Based on requirements analysis",
    status=DecisionStatus.APPROVED
)
```

## Development

### Setup

```bash
git clone https://github.com/Infosourceful/infosourceful-kernel.git
cd infosourceful-kernel
pip install -e .
```

### Testing

```bash
pytest tests/ -v
```

### Linting

```bash
flake8 infosourceful_kernel tests
black --check infosourceful_kernel tests
isort --check-only infosourceful_kernel tests
```

## License

Proprietary - Infosourceful Works

## Contributing

This repository is maintained by the Infosourceful team. For questions or issues, please contact support@infosourceful.com.
