import argparse, json, yaml, sys
from importlib.resources import files
from jsonschema import Draft202012Validator, ValidationError


def load_meta(path: str):
    raw = open(path, "rb").read()
    return (
        yaml.safe_load(raw)
        if path.lower().endswith((".yml", ".yaml"))
        else json.loads(raw)
    )


def load_schema_for_major(major: str):
    schema_path = files("infosourceful_kernel.schemas").joinpath(
        f"meta-v{major}.0.json"
    )
    return json.loads(schema_path.read_text(encoding="utf-8"))


def main():
    p = argparse.ArgumentParser(description="Validate an Infosourceful .meta file.")
    p.add_argument("meta_path")
    args = p.parse_args()

    try:
        meta = load_meta(args.meta_path)
        sv = str(meta.get("schema_version", "1.0.0"))
        major = sv.split(".")[0]
        schema = load_schema_for_major(major)
        Draft202012Validator(schema).validate(meta)
        print("OK")
    except FileNotFoundError:
        print("ERROR: file not found.", file=sys.stderr)
        sys.exit(2)
    except ValidationError as e:
        print(f"INVALID: {e.message}", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"ERROR: {e}", file=sys.stderr)
        sys.exit(3)
