"""
Artifact model for Infosourceful Kernel.
"""

from typing import List, Optional
from datetime import date
from pydantic import BaseModel, Field
from .enums import ArtifactType
from .actor import Actor
from .decision import Decision


class Period(BaseModel):
    """Represents a time period."""

    from_date: date = Field(alias="from")
    to_date: date = Field(alias="to")

    class Config:
        """Pydantic configuration."""

        populate_by_name = True


class Risk(BaseModel):
    """Represents a risk associated with an artifact."""

    id: str
    title: str
    probability: Optional[float] = Field(None, ge=0.0, le=1.0, alias="prob")
    impact: Optional[float] = Field(None, ge=0.0, le=1.0)
    owner: Optional[str] = None

    class Config:
        """Pydantic configuration."""

        populate_by_name = True


class WorkItem(BaseModel):
    """Represents a work item associated with an artifact."""

    id: str
    title: str
    status: Optional[str] = None
    url: Optional[str] = None


class Artifact(BaseModel):
    """Represents an artifact in the Infosourceful ecosystem."""

    schema_version: str = Field(..., pattern=r"^1\.\d+\.\d+$")
    artifact_id: str = Field(..., min_length=6)
    artifact_type: ArtifactType
    project_key: str
    source_systems: List[str]
    hash: str = Field(..., min_length=16)

    # Optional fields
    period: Optional[Period] = None
    actors: Optional[List[Actor]] = None
    decisions: Optional[List[Decision]] = None
    risks: Optional[List[Risk]] = None
    work_items: Optional[List[WorkItem]] = None

    class Config:
        """Pydantic configuration."""

        use_enum_values = True
        populate_by_name = True
