"""
Test schema roundtrip validation for Infosourceful Kernel.

This module tests that artifacts can be validated against the JSON schema
and that the Pydantic models work correctly with the schema.
"""

import json
import tempfile
from pathlib import Path
from datetime import date

import pytest
from pydantic import ValidationError

from infosourceful_kernel.models.artifact import Artifact, Period, Risk, WorkItem
from infosourceful_kernel.models.actor import Actor
from infosourceful_kernel.models.decision import Decision
from infosourceful_kernel.models.enums import ArtifactType, ActorRole, DecisionStatus
from infosourceful_kernel.validation import (
    validate_artifact_json,
    validate_artifact_model,
)


class TestSchemaRoundtrip:
    """Test schema roundtrip validation."""

    def test_minimal_artifact_validation(self):
        """Test validation of minimal artifact."""
        minimal_artifact = {
            "schema_version": "1.0.0",
            "artifact_id": "test123",
            "artifact_type": "deck",
            "project_key": "test_project",
            "source_systems": ["system1", "system2"],
            "hash": "abcdef1234567890abcdef1234567890",
        }

        result = validate_artifact_json(json.dumps(minimal_artifact))
        assert result.is_valid
        assert result.artifact_id == "test123"
        assert result.artifact_type == "deck"

    def test_full_artifact_validation(self):
        """Test validation of full artifact with all optional fields."""
        full_artifact = {
            "schema_version": "1.0.0",
            "artifact_id": "test456",
            "artifact_type": "brief",
            "project_key": "test_project",
            "source_systems": ["system1"],
            "hash": "fedcba0987654321fedcba0987654321",
            "period": {"from": "2024-01-01", "to": "2024-12-31"},
            "actors": [
                {"name": "John Doe", "role": "owner", "email": "john@example.com"}
            ],
            "decisions": [
                {
                    "id": "dec1",
                    "title": "Approval decision",
                    "date": "2024-01-15",
                    "rationale": "Approved based on requirements",
                }
            ],
            "risks": [
                {
                    "id": "risk1",
                    "title": "Technical risk",
                    "prob": 0.3,
                    "impact": 0.7,
                    "owner": "Tech Lead",
                }
            ],
            "work_items": [
                {
                    "id": "work1",
                    "title": "Implementation task",
                    "status": "in_progress",
                    "url": "https://example.com/work1",
                }
            ],
        }

        result = validate_artifact_json(json.dumps(full_artifact))
        assert result.is_valid
        assert result.artifact_id == "test456"
        assert result.artifact_type == "brief"

    def test_invalid_artifact_validation(self):
        """Test validation of invalid artifact."""
        invalid_artifact = {
            "schema_version": "1.0.0",
            "artifact_id": "short",  # Too short
            "artifact_type": "invalid_type",  # Invalid enum
            "project_key": "test_project",
            "source_systems": ["system1"],
            "hash": "short",  # Too short
        }

        result = validate_artifact_json(json.dumps(invalid_artifact))
        assert not result.is_valid
        assert len(result.errors) > 0

    def test_pydantic_model_creation(self):
        """Test creating Pydantic models from valid data."""
        artifact_data = {
            "schema_version": "1.0.0",
            "artifact_id": "test789",
            "artifact_type": "dataset",
            "project_key": "test_project",
            "source_systems": ["system1", "system2"],
            "hash": "1234567890abcdef1234567890abcdef",
            "period": {"from": "2024-01-01", "to": "2024-12-31"},
            "actors": [
                {"name": "Jane Doe", "role": "contributor", "email": "jane@example.com"}
            ],
        }

        # Test Artifact model creation
        artifact = Artifact(**artifact_data)
        assert artifact.artifact_id == "test789"
        assert artifact.artifact_type == ArtifactType.DATASET
        assert artifact.period is not None
        assert artifact.period.from_date == date(2024, 1, 1)
        assert len(artifact.actors) == 1
        assert artifact.actors[0].name == "Jane Doe"
        assert artifact.actors[0].role == ActorRole.CONTRIBUTOR

    def test_pydantic_model_validation(self):
        """Test Pydantic model validation against schema."""
        artifact = Artifact(
            schema_version="1.0.0",
            artifact_id="test999",
            artifact_type=ArtifactType.ACTION_LOG,
            project_key="test_project",
            source_systems=["system1"],
            hash="99999999999999999999999999999999",
            period=Period(from_date=date(2024, 1, 1), to_date=date(2024, 12, 31)),
            actors=[
                Actor(
                    name="Test User",
                    role=ActorRole.STAKEHOLDER,
                    email="test@example.com",
                )
            ],
            decisions=[
                Decision(
                    id="dec1",
                    title="Test Decision",
                    date=date(2024, 1, 15),
                    rationale="Test rationale",
                    status=DecisionStatus.APPROVED,
                )
            ],
        )

        result = validate_artifact_model(artifact)
        assert result.is_valid
        assert result.artifact_id == "test999"
        assert result.artifact_type == "action_log"

    def test_invalid_pydantic_model(self):
        """Test Pydantic model validation with invalid data."""
        with pytest.raises(ValidationError):
            Artifact(
                schema_version="1.0.0",
                artifact_id="short",  # Too short
                artifact_type="invalid_type",  # Invalid enum
                project_key="test_project",
                source_systems=["system1"],
                hash="short",  # Too short
            )

    def test_yaml_validation(self):
        """Test validation of YAML content."""
        yaml_content = """
schema_version: "1.0.0"
artifact_id: "yaml123"
artifact_type: "mapping"
project_key: "test_project"
source_systems: ["system1"]
hash: "yaml1234567890abcdef1234567890abcdef"
"""

        result = validate_artifact_json(yaml_content, format_type="yaml")
        assert result.is_valid
        assert result.artifact_id == "yaml123"
        assert result.artifact_type == "mapping"

    def test_schema_version_pattern(self):
        """Test schema version pattern validation."""
        # Valid versions
        valid_versions = ["1.0.0", "1.1.0", "1.2.3"]
        for version in valid_versions:
            artifact_data = {
                "schema_version": version,
                "artifact_id": "test123",
                "artifact_type": "deck",
                "project_key": "test_project",
                "source_systems": ["system1"],
                "hash": "1234567890abcdef1234567890abcdef",
            }
            result = validate_artifact_json(json.dumps(artifact_data))
            assert result.is_valid, f"Version {version} should be valid"

        # Invalid versions
        invalid_versions = ["2.0.0", "1.0", "1.0.0.0", "v1.0.0"]
        for version in invalid_versions:
            artifact_data = {
                "schema_version": version,
                "artifact_id": "test123",
                "artifact_type": "deck",
                "project_key": "test_project",
                "source_systems": ["system1"],
                "hash": "1234567890abcdef1234567890abcdef",
            }
            result = validate_artifact_json(json.dumps(artifact_data))
            assert not result.is_valid, f"Version {version} should be invalid"
