from dataclasses import dataclass
from pollination_dsl.function import Inputs, Outputs, Function, command


@dataclass
class EnergyUseIntensity(Function):
    """Get information about energy use intensity from energy simulation SQLite files."""

    result_folder = Inputs.folder(
        description='Path to folder containing SQLite files that were generated '
        'by EnergyPlus. This can be a single EnergyPlus simulation folder or '
        'it can be a folder with multiple sql files, in which case EUI will '
        'be computed across all results.', path='result_folder'
    )

    units = Inputs.str(
        description='A switch to indicate whether the data in the resulting JSON '
        'should be in SI or IP units. Valid values are "si" and "ip".', default='si',
        spec={'type': 'string', 'enum': ['si', 'ip']}
    )

    @command
    def energy_use_intensity(self):
        return 'honeybee-energy result energy-use-intensity result_folder ' \
            '--{{inputs.units}} --output-file output.json'

    eui_json = Outputs.file(
        description='A JSON object with the following keys - eui, total_floor_area, '
        'total_energy, end_uses. The eui value is the energy use intensity across '
        'the total floor area. The end_uses value is a dictionary containing a '
        'breakdown of the eui by end use.', path='output.json'
    )


@dataclass
class AvailableResultsInfo(Function):
    """Get information about time-series outputs that can be requested from a SQL file.
    """

    result_sql = Inputs.file(
        description='A SQLite file that was generated by EnergyPlus.',
        path='result.sql', extensions=['sql', 'db', 'sqlite']
    )

    @command
    def available_results_info(self):
        return 'honeybee-energy result all-available-info result.sql ' \
            '--output-file output.json'

    available_outputs = Outputs.file(
        description='A JSON object with two keys - run_periods, outputs. outputs '
        'is a list of EnergyPlus outputs and metadata that can be requested from '
        'the result-sql. run_periods is a list of run periods and metadata contained '
        'within the result-sql.', path='output.json'
    )


@dataclass
class DataByOutput(Function):
    """Get an array of DataCollection JSONs for a specific EnergyPlus output."""

    result_sql = Inputs.file(
        description='A SQLite file that was generated by EnergyPlus.',
        path='result.sql', extensions=['sql', 'db', 'sqlite']
    )

    output_name = Inputs.str(
        description='An EnergyPlus output name to be retrieved from the result-sql. '
        '(eg. Zone Operative Temperature). This can also be a JSON array of names '
        'formatted with [] brackets.'
    )

    @command
    def data_by_output(self):
        return 'honeybee-energy result data-by-output result.sql ' \
            '"{{inputs.output_name}}" --output-file output.json'

    data_jsons = Outputs.file(
        description='A list of DataCollection JSONs that match the requested '
        'output name.', path='output.json'
    )


@dataclass
class ZoneSizes(Function):
    """Get a JSON with the peak heating and cooling values for each zone."""

    result_sql = Inputs.file(
        description='A SQLite file that was generated by EnergyPlus.',
        path='result.sql', extensions=['sql', 'db', 'sqlite']
    )

    @command
    def data_by_output(self):
        return 'honeybee-energy result zone-sizes result.sql --output-file output.json'

    zone_size_json = Outputs.file(
        description='A JSON with two keys for "heating" and "cooling," each of '
        'which contains an array of sizing information for each zone.',
        path='output.json'
    )


@dataclass
class ComponentSizes(Function):
    """Get a JSON with the sizes for various components of the HVAC system."""

    result_sql = Inputs.file(
        description='A SQLite file that was generated by EnergyPlus.',
        path='result.sql', extensions=['sql', 'db', 'sqlite']
    )

    component_type = Inputs.str(
        description='A name of a HVAC component type, which will be used to filter '
        'the output HVAC components. If none, all HVAC component sizes will be output.',
        default=''
    )

    @command
    def data_by_output(self):
        return 'honeybee-energy result component-sizes result.sql --component-type ' \
            '"{{inputs.component_type}}" --output-file output.json'

    component_size_json = Outputs.file(
        description='A JSON array of sizing information for each component of '
        'HVAC equipment.', path='output.json'
    )


@dataclass
class ResultCsvQueryable(Function):
    """Get a CSV of energy simulation results as an easily query-able SQLite table."""

    result_sql = Inputs.file(
        description='A SQLite file that was generated by EnergyPlus.',
        path='result.sql', extensions=['sql', 'db', 'sqlite']
    )

    model = Inputs.file(
        description='Honeybee model in JSON format that will be matched to results '
        'in the result sql and used to account for triangulated sub-faces, room '
        'multipliers, and any area normalization of results.', path='model.json',
        extensions=['hbjson', 'json']
    )

    output_name = Inputs.str(
        description='An EnergyPlus output name to be retrieved from the result-sql. '
        '(eg. Zone Operative Temperature). This can also be a JSON array of names '
        'formatted with [] brackets.'
    )

    run_period = Inputs.str(
        description='The name of the run period from which the CSV data will be '
        'selected. (eg. BOSTON LOGAN INTL AIRPORT ANN CLG .4% CONDNS DB=>MWB).'
    )

    units = Inputs.str(
        description='A switch to indicate whether the data in the resulting CSV '
        'should be in SI or IP units. Valid values are "si" and "ip".', default='si',
        spec={'type': 'string', 'enum': ['si', 'ip']}
    )

    normalization = Inputs.str(
        description='A switch to indicate whether the data in the resulting CSV should '
        'be normalized by floor area (in the case of Zone/System data) or surface '
        'area (in the case of Surface data). This has no effect if the requested '
        'data is not normalizable. Valid values are "normalize" and "no-normalize".',
        default='normalize',
        spec={'type': 'string', 'enum': ['normalize', 'no-normalize']}
    )

    @command
    def result_csv_queryable(self):
        return 'honeybee-energy result output-csv-queryable result.sql model.json ' \
            '"{{inputs.run-period}}" "{{inputs.output_name}}" --{{inputs.units}} ' \
            '--{{inputs.normalization}} --folder output --log-file output/col_names.json'

    column_names = Outputs.file(
        description='A list of DataCollection JSONs that match the requested '
        'output name.', path='output.json'
    )
