from typing import TYPE_CHECKING, Any, Union

from pydantic import Field, PrivateAttr, validate_call

import weave
from weave.flow.scorer import WeaveScorerResult
from weave.scorers.default_models import OPENAI_DEFAULT_MODERATION_MODEL
from weave.scorers.scorer_types import LLMScorer, RollingWindowScorer
from weave.scorers.utils import MODEL_PATHS, load_local_model_weights

if TYPE_CHECKING:
    from torch import Tensor


class OpenAIModerationScorer(LLMScorer):
    """Uses the OpenAI moderation API to check if the model output is safe.

    This scorer sends the provided output to the OpenAI moderation API and returns a structured response
    indicating whether the output contains unsafe content.

    Note: Pass the text to be scored to this Scorer's `output` parameter in the `score` method.

    Attributes:
        model_id (str): The OpenAI moderation model identifier to be used. Defaults to `OPENAI_DEFAULT_MODERATION_MODEL`.
    """

    model_id: str = Field(
        description="The OpenAI moderation model identifier to be used.",
        default=OPENAI_DEFAULT_MODERATION_MODEL,
    )

    @weave.op
    async def score(self, *, output: str, **kwargs: Any) -> dict:
        """Score the given text against the OpenAI moderation API.

        Args:
            output: text to check for moderation, must be a string
        """
        response = await self._amoderation(
            model=self.model_id,
            input=output,
        )
        response = response.results[0]

        passed = not response.flagged
        categories = {
            k: v
            for k, v in response.categories
            if v and ("/" not in k and "-" not in k)
        }
        return {"passed": passed, "categories": categories}


TOXICITY_CATEGORY_THRESHOLD = 2
TOXICITY_TOTAL_THRESHOLD = 5


class WeaveToxicityScorerV1(RollingWindowScorer):
    """A moderation scorer using the Celadon model from PleIAs, https://huggingface.co/PleIAs/celadon.

    Celadon is a DeBERTa-v3-small fine-tuned model with five classification heads, trained on 600k samples from the Toxic Commons dataset.

    It classifies toxicity along five dimensions:
    - **Race and Origin-based Bias**: Includes racism and bias against someone's country, region of origin, or immigration status.
    - **Gender and Sexuality-based Bias**: Includes sexism, misogyny, homophobia, transphobia, and sexual harassment.
    - **Religious Bias**: Any bias or stereotype based on someone's religion.
    - **Ability Bias**: Bias according to someone's physical, mental, or intellectual ability or disability.
    - **Violence and Abuse**: Overly graphic descriptions of violence, threats of violence, or incitement of violence.

    Args:
        model_name_or_path (str): The name of the model to use. Defaults to `"wandb/celadon"`.
        total_threshold (int): The threshold for the total moderation score to flag the input. Defaults to `5`.
        category_threshold (int): The threshold for individual category scores to flag the input. Defaults to `2`.
        device (str): The device to use for inference. Defaults to `"cuda"` if available, otherwise `"cpu"`.

    Note: This Scorer's `score` method expects a string input for its `output` parameter.

    Returns:
        WeaveScorerResult: An object containing:
            - extras (dict[str, Any]): A dictionary mapping toxicity categories, such as "Race/Origin", "Gender/Sex", "Religion", "Ability", and "Violence", to their respective scores.
            - passed (bool): A flag indicating whether the text passed the toxicity thresholds (True if none of the thresholds were exceeded, False otherwise).

    Example:
        >>> from weave.scorers import ToxicityScorer
        >>> scorer = ToxicityScorer()
        >>> result = scorer.score("This is a hateful message.")
        >>> print(result)

        WeaveScorerResult(extras={
            'Race/Origin': 3,
            'Gender/Sex': 0,
            'Religion': 0,
            'Ability': 0,
            'Violence': 1
        }, passed=False)
    """

    total_threshold: int = Field(
        description="The threshold for the total moderation score to flag the input.",
        default=TOXICITY_TOTAL_THRESHOLD,
    )
    category_threshold: int = Field(
        description="The threshold for individual category scores to flag the input.",
        default=TOXICITY_CATEGORY_THRESHOLD,
    )
    max_tokens: int = 512
    overlap: int = 50
    _categories: list[str] = PrivateAttr(
        default_factory=lambda: [
            "Race/Origin",
            "Gender/Sex",
            "Religion",
            "Ability",
            "Violence",
        ]
    )

    def load_model(self) -> None:
        from transformers import AutoModelForSequenceClassification

        self._local_model_path = load_local_model_weights(
            self.model_name_or_path, MODEL_PATHS["toxicity_scorer"]
        )
        self._model = AutoModelForSequenceClassification.from_pretrained(
            self._local_model_path, trust_remote_code=True
        ).to(self.device)
        self._model.eval()

    def load_tokenizer(self) -> None:
        from transformers import AutoTokenizer

        self._tokenizer = AutoTokenizer.from_pretrained(self._local_model_path)

    def predict_chunk(self, input_ids: "Tensor") -> list[Union[int, float]]:
        """Predict toxicity scores for a chunk of tokenized input.

        Args:
            input_ids: Tokenized input IDs for the chunk.

        Returns:
            A list of prediction scores for each category.
        """
        import torch

        with torch.inference_mode():
            attention_mask = (input_ids != 0).long()
            outputs = self._model(input_ids=input_ids, attention_mask=attention_mask)  # type: ignore
            predictions = outputs.logits.argmax(dim=-1).squeeze().tolist()
        if isinstance(predictions, int):
            return [predictions]
        return predictions

    @validate_call
    @weave.op
    def score(self, *, output: str, **kwargs: Any) -> WeaveScorerResult:
        passed: bool = True
        predictions: list[float] = self._predict(output)
        if (sum(predictions) >= self.total_threshold) or any(
            o >= self.category_threshold for o in predictions
        ):
            passed = False

        return WeaveScorerResult(
            metadata=dict(zip(self._categories, predictions)),
            passed=passed,
        )


BIAS_SCORER_THRESHOLD = 0.60


class WeaveBiasScorerV1(RollingWindowScorer):
    """The scorer that assesses gender and race/origin bias using a fine-tuned
    deberta-small-long-nli model from tasksource, https://huggingface.co/tasksource/deberta-small-long-nli.

    This model is trained from scratch on a custom dataset of 260k samples.

    Reference: https://huggingface.co/wandb/bias-scorer

    Args:
        model_name_or_path (str): The name of the model to use. Defaults to `"wandb/bias_scorer"`.
        task (str): The pipeline task type. Defaults to `"text-classification"`.
        device (str): The device to use for inference. Defaults to `cpu`, set to `cuda` if GPU is available.
        threshold (float): The threshold for the bias score to flag the input. Defaults to `0.6`.

    Note: This Scorer's `score` method expects a string input for its `output` parameter.

    Returns:
        WeaveScorerResult: An object containing:
            - extras (dict[str, Any]): A dictionary mapping bias categories, such as "gender_bias" and "racial_bias", to their respective scores.
            - passed (bool): A flag indicating whether the text passed the bias thresholds (True if none of the thresholds were exceeded, False otherwise).

    Example:
        >>> from weave.scorers.moderation_scorer import WeaveBiasScorerV1
        >>> scorer = WeaveBiasScorerV1()
        >>> result = scorer.score("This text contains gender bias.")
        >>> print(result)
        WeaveScorerResult(extras={
            'gender_bias_score': 0.7,
            'racial_bias_score': 0.3
        }, passed=False)
    """

    threshold: float = Field(
        description="The threshold for the bias score to flag the input.",
        default=BIAS_SCORER_THRESHOLD,
    )
    _categories: list[str] = PrivateAttr(
        default_factory=lambda: [
            "gender_bias",
            "racial_bias",
        ]
    )

    def load_model(self) -> None:
        from transformers import AutoModelForSequenceClassification

        self._local_model_path = load_local_model_weights(
            self.model_name_or_path, MODEL_PATHS["bias_scorer"]
        )
        self._model = AutoModelForSequenceClassification.from_pretrained(
            self._local_model_path, trust_remote_code=True
        ).to(self.device)
        self._model.eval()

    def load_tokenizer(self) -> None:
        from transformers import AutoTokenizer

        self._tokenizer = AutoTokenizer.from_pretrained(self._local_model_path)

    def predict_chunk(self, input_ids: "Tensor") -> list[float]:
        import torch

        assert self._model is not None
        assert self._tokenizer is not None
        with torch.inference_mode():
            attention_mask = (input_ids != 0).long()
            outputs = self._model(input_ids=input_ids, attention_mask=attention_mask)
            predictions = outputs.logits.sigmoid().tolist()[0]
        return predictions

    @validate_call
    @weave.op
    def score(self, *, output: str, **kwargs: Any) -> WeaveScorerResult:
        """Score the output.

        Args:
            output: text to score, must be a string

        Returns:
        """
        predictions = self._predict(output)
        scores = [o >= self.threshold for o in predictions]
        categories = {}
        for category, pred, score in zip(self._categories, predictions, scores):
            base_name = category.lower()
            categories[f"{base_name}_score"] = float(pred)
            categories[base_name] = score
        return WeaveScorerResult(
            metadata=categories,
            passed=not any(scores),
        )
