"""Main Gemini client for exc2issue.

This module provides the main GeminiClient class for interacting with the Google
Gemini API to generate intelligent GitHub issue descriptions.
"""

from __future__ import annotations

import time
from typing import TYPE_CHECKING, Any, cast

from pydantic import SecretStr

from exc2issue.config import GeminiConfig

from ._fallback import create_fallback_description
from ._prompt_builder import build_prompt

if TYPE_CHECKING:
    from exc2issue.core.models import ErrorRecord

APIError: type[Exception]
ServerError: type[Exception]
GENAI: Any | None

try:
    from google import genai as genai_module
    from google.genai.errors import APIError as GenaiAPIError
    from google.genai.errors import ServerError as GenaiServerError
except ImportError:
    # Google Gemini is an optional dependency
    GENAI = None
    APIError = type("APIError", (Exception,), {})
    ServerError = type("ServerError", (Exception,), {})
else:
    GENAI = genai_module
    APIError = GenaiAPIError
    ServerError = GenaiServerError


class GeminiClient:
    """Client for interacting with the Gemini API.

    This client handles authentication, prompt generation, and AI-powered
    issue description generation with proper error handling and fallback mechanisms.
    Uses Pydantic configuration for settings management.

    Attributes:
        config: Gemini configuration settings
        api_key: Google Gemini API key for authentication
        model_name: Name of the Gemini model to use
        temperature: Sampling temperature for response generation
        use_fallback: Whether to use fallback descriptions when API fails
        max_retries: Maximum number of retry attempts for API calls
        max_output_tokens: Maximum tokens in generated output
        prompt_template: Custom template for prompt generation
    """

    def __init__(
        self,
        api_key: str | None = None,
        *,
        config: GeminiConfig | None = None,
        **kwargs: Any,
    ):
        """Initialize Gemini client.

        Args:
            api_key: Google Gemini API key. If None, will try to read from env vars.
            config: Gemini configuration settings (keyword-only, advanced usage).
            **kwargs: Additional configuration parameters (model_name, temperature, etc.).

        Raises:
            ValueError: If no API key is provided and GEMINI_API_KEY env var is not set.

        Note:
            For complete configuration control, use the config parameter with a GeminiConfig object.
            Additional kwargs are used only when config is not provided.
        """
        # Priority order: config > parameters > env vars
        self.config: GeminiConfig
        if config is not None:
            self.config = config
        elif api_key is not None:
            # Create config from simplified parameters with kwargs support
            config_params: dict[str, Any] = {"api_key": SecretStr(api_key)}
            # Add any additional kwargs that are valid GeminiConfig parameters
            # Note: GeminiConfig expects plain types, not SecretStr
            config_params.update(kwargs)
            self.config = GeminiConfig(**config_params)
        else:
            # Use GeminiConfig to load from environment variables (including .env file)
            try:
                config_factory = cast("type[Any]", GeminiConfig)
                self.config = cast("GeminiConfig", config_factory())
            except Exception as e:
                # No API key available - GeminiClient will be None in decorator
                # This allows the system to work with fallback descriptions
                raise ValueError(
                    "Gemini API key not found. Set GEMINI_API_KEY environment "
                    "variable or install with 'pip install exc2issue[ai]' for AI "
                    f"support. Error: {e}"
                ) from e

        # Store config and extract critical values for internal use
        # This reduces the number of instance attributes by grouping related settings
        self.prompt_template = kwargs.get("prompt_template")  # Extract from kwargs

        # Initialize the Gemini client
        if GENAI is None:
            raise ValueError(
                "Google Gemini library not available. Install with "
                "'pip install exc2issue[ai]' for AI support."
            )
        self.client = GENAI.Client(api_key=self.config.api_key.get_secret_value())

    @classmethod
    def from_config(cls, config: GeminiConfig) -> GeminiClient:
        """Create GeminiClient from GeminiConfig.

        Args:
            config: Gemini configuration settings

        Returns:
            GeminiClient: Configured client instance
        """
        return cls(config=config)

    @classmethod
    def from_config_optional(cls, config: GeminiConfig | None) -> GeminiClient | None:
        """Create GeminiClient from GeminiConfig if configuration is available.

        Args:
            config: Gemini configuration settings or None

        Returns:
            GeminiClient: Configured client instance if config is available, None otherwise
        """
        if config is None:
            return None

        try:
            return cls(config=config)
        except ValueError:
            # If creation fails (e.g., missing API key), return None
            return None

    def generate_issue_description(
        self, error_record: ErrorRecord, _allow_fallback: bool = True
    ) -> str:
        """Generate a description for a GitHub issue based on an error record.

        Args:
            error_record: ErrorRecord containing error details
            _allow_fallback: Internal parameter to control fallback behavior for retry logic

        Returns:
            Generated issue description as a string

        Raises:
            APIError: If Gemini API call fails and use_fallback is False
        """
        try:
            prompt = build_prompt(error_record, self.prompt_template)

            # Generate content using Gemini API
            response = self.client.models.generate_content(
                model=self.config.model_name, contents=prompt
            )

            if self.validate_response(response):
                if response.text is not None:
                    response_text = str(response.text)
                    return response_text.strip()

                if self.config.use_fallback and _allow_fallback:
                    return create_fallback_description(error_record)

                raise ValueError("Empty response from Gemini API")

            if self.config.use_fallback and _allow_fallback:
                return create_fallback_description(error_record)

            raise ValueError("Invalid response from Gemini API")

        except (APIError, ServerError):
            # Don't use fallback during retry attempts for proper retry logic
            if _allow_fallback and self.config.use_fallback:
                return create_fallback_description(error_record)

            raise

    def generate_issue_description_with_retry(self, error_record: ErrorRecord) -> str:
        """Generate issue description with retry mechanism.

        Args:
            error_record: ErrorRecord containing error details

        Returns:
            Generated issue description as a string

        Raises:
            APIError: If all retry attempts fail
        """
        last_error = None

        for attempt in range(self.config.max_retries):
            try:
                # Disable fallback during retries to allow proper retry logic
                return self.generate_issue_description(
                    error_record, _allow_fallback=False
                )
            except (APIError, ServerError, ValueError) as exc:
                last_error = exc
                # Check for ServerError first since it's a subclass of APIError
                if isinstance(exc, ServerError):
                    # Handle server errors with retry
                    if attempt == self.config.max_retries - 1:
                        # Last attempt failed, use fallback if enabled
                        if self.config.use_fallback:
                            return create_fallback_description(error_record)

                        raise

                    # Exponential backoff for server errors
                    wait_time = 2**attempt
                    time.sleep(wait_time)
                elif isinstance(exc, APIError):
                    # Don't retry API errors (usually client errors)
                    # Use fallback if enabled
                    if self.config.use_fallback:
                        return create_fallback_description(error_record)
                    raise
                else:
                    # Handle ValueError and other exceptions
                    if attempt == self.config.max_retries - 1:
                        # Last attempt failed, use fallback if enabled
                        if self.config.use_fallback:
                            return create_fallback_description(error_record)

                        raise

                    # Exponential backoff for other errors
                    wait_time = 2**attempt
                    time.sleep(wait_time)

        # This should never be reached, but just in case
        if last_error:
            raise last_error

        raise ValueError("All retry attempts failed")

    def validate_response(self, response: Any) -> bool:
        """Validate API response.

        Args:
            response: Response object from Gemini API

        Returns:
            True if response is valid, False otherwise
        """
        if not response:
            return False

        if not hasattr(response, "text"):
            return False

        if response.text is None:
            return False
        return bool(response.text.strip())
