"""Client factory for creating GitHub and Gemini clients.

This module provides factory functions for creating client instances
while avoiding circular import issues in the decorator module.

Note: Imports are done inside functions to prevent circular import issues.
Pylint import-outside-toplevel warning is intentionally accepted here as
moving imports to module level would recreate the circular dependency.
"""

from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from exc2issue.adapters.gemini import GeminiClient
    from exc2issue.adapters.github import GitHubClient


def _get_github_client_class() -> type["GitHubClient"]:
    """Get GitHub client class.

    Import is done here to prevent circular dependencies.
    """
    # pylint: disable=import-outside-toplevel
    from exc2issue.adapters.github import GitHubClient as _GitHubClient

    return _GitHubClient


def _get_gemini_client_class() -> type["GeminiClient"] | None:
    """Get Gemini client class if available.

    Import is done here to prevent circular dependencies.
    """
    try:
        # pylint: disable=import-outside-toplevel
        from exc2issue.adapters.gemini import GeminiClient as _GeminiClient

        return _GeminiClient
    except ImportError:
        return None


def create_github_client(token: str | None = None) -> "GitHubClient":
    """Create GitHub client instance.

    Args:
        token: GitHub API token (optional)

    Returns:
        GitHubClient: Configured GitHub client
    """
    github_client_class = _get_github_client_class()
    if token:
        return github_client_class(token=token)
    return github_client_class()


def create_gemini_client(api_key: str | None = None) -> "GeminiClient | None":
    """Create Gemini client instance if available.

    Args:
        api_key: Gemini API key (optional)

    Returns:
        GeminiClient: Configured Gemini client or None if unavailable
    """
    gemini_client_class = _get_gemini_client_class()

    if gemini_client_class is None:
        # Gemini client is optional - system works with fallback descriptions
        return None
    if api_key:
        try:
            return gemini_client_class(api_key=api_key)
        except ValueError:
            return None
    else:
        try:
            return gemini_client_class()
        except ValueError:
            return None
