"""Main bug hunter decorator class with comprehensive error handling.

This module provides the core BugHunterDecorator class that wraps functions
to automatically create GitHub issues when errors occur. The decorator handles:

- Function wrapping with error collection context
- Exception and SystemExit handling
- Integration with consolidated handlers for log errors
- Signal and exit termination handling
- Configuration management for resilience features

The decorator coordinates with other core modules for background processing,
signal handling, issue creation, and registry management.
"""

import contextlib
import functools
import uuid
from collections.abc import Callable
from typing import TYPE_CHECKING, Any

from exc2issue.core.background_worker import ensure_background_worker
from exc2issue.core.client_factory import create_gemini_client, create_github_client
from exc2issue.core.config_types import AuthConfig, BugHunterConfig, ProcessingConfig
from exc2issue.core.error_collection import ErrorCollection, ErrorCollectionContext
from exc2issue.core.error_handling import (
    handle_exit_cleanup,
    handle_signal_termination,
    handle_system_exit,
)
from exc2issue.core.handlers import ConsolidatedHandlers
from exc2issue.core.issue_creator import process_error_collection
from exc2issue.core.registry import add_active_decorator
from exc2issue.core.signal_handling import setup_exit_handler, setup_signal_handlers

if TYPE_CHECKING:
    from exc2issue.adapters.gemini import GeminiClient
    from exc2issue.adapters.github import GitHubClient


class BugHunterDecorator:
    """Bug hunter decorator with comprehensive error handling and resilience.

    This decorator provides complete error handling capabilities:
    1. Consolidated error handling for multiple related errors
    2. Multi-layered resilience for termination scenarios
    3. Configurable behavior through feature flags
    """

    def __init__(
        self,
        config: BugHunterConfig | None = None,
        repository: str | None = None,
        **kwargs: Any,
    ):
        """Initialize bug hunter decorator with comprehensive error handling.

        Args:
            config: Complete configuration object (preferred)
            repository: GitHub repository in format "owner/repo" (required if config not provided)
            **kwargs: Legacy parameters for backward compatibility
        """
        # Use provided config or create from legacy parameters
        if config is not None:
            self.config = config
        elif repository:
            # Create from repository and kwargs
            all_params = {"repository": repository, **kwargs}
            self.config = BugHunterConfig.create_legacy(**all_params)
        elif kwargs.get("repository"):
            # Create from kwargs only
            self.config = BugHunterConfig.create_legacy(**kwargs)
        else:
            raise ValueError("Repository is required for exc2issue decorator")

        # Initialize client attributes to avoid W0201, but don't create actual clients yet
        # Clients are created lazily when first accessed
        self._github_client: GitHubClient | None = None
        self._gemini_client: GeminiClient | None = None

        # Instance tracking
        self._instance_id = str(uuid.uuid4())
        self._is_active = False
        self._current_error_collection: ErrorCollection | None = None

        # Setup consolidated handlers
        self.consolidated_handlers = ConsolidatedHandlers(self)

        # Setup resilience mechanisms
        self._setup_resilience_mechanisms()

    # Backward compatibility properties
    @property
    def repository(self) -> str:
        """Get repository from config."""
        return self.config.repository

    @property
    def labels(self) -> list[str]:
        """Get labels from config."""
        return self.config.labels

    @property
    def assignees(self) -> list[str]:
        """Get assignees from config."""
        return self.config.assignees

    @property
    def auth_config(self) -> AuthConfig:
        """Get auth config."""
        return self.config.auth_config

    @property
    def processing_config(self) -> ProcessingConfig:
        """Get processing config."""
        return self.config.processing_config

    @property
    def consolidation_threshold(self) -> int:
        """Get consolidation threshold from processing config."""
        return self.config.processing_config.consolidation_threshold

    @property
    def enable_signal_handling(self) -> bool:
        """Get signal handling setting from processing config."""
        return self.config.processing_config.enable_signal_handling

    @property
    def enable_exit_handling(self) -> bool:
        """Get exit handling setting from processing config."""
        return self.config.processing_config.enable_exit_handling

    @property
    def enable_background_processing(self) -> bool:
        """Get background processing setting from processing config."""
        return self.config.processing_config.enable_background_processing

    @property
    def github_client(self) -> "GitHubClient":
        """Get GitHub client, creating it lazily if needed."""
        if self._github_client is None:
            self._github_client = create_github_client(
                self.config.auth_config.github_token
            )
        return self._github_client

    @property
    def gemini_client(self) -> "GeminiClient | None":
        """Get Gemini client, creating it lazily if needed."""
        if self._gemini_client is None:
            self._gemini_client = create_gemini_client(
                self.config.auth_config.gemini_api_key
            )
        return self._gemini_client

    # Public accessors for state management
    def is_active(self) -> bool:
        """Check if decorator is currently active."""
        return self._is_active

    def get_current_error_collection(self) -> ErrorCollection | None:
        """Get current error collection if available."""
        return self._current_error_collection

    def _setup_resilience_mechanisms(self) -> None:
        """Setup signal handlers and exit handlers."""
        # Register this decorator
        add_active_decorator(self)

        # Setup global signal handlers (only once)
        if self.config.processing_config.enable_signal_handling:
            setup_signal_handlers()

        # Setup global exit handler (only once)
        if self.config.processing_config.enable_exit_handling:
            setup_exit_handler()

        # Ensure background worker is running
        if self.config.processing_config.enable_background_processing:
            ensure_background_worker()

    def __call__(self, func: Callable[..., Any]) -> Callable[..., Any]:
        """Decorate the function with comprehensive error handling and resilience."""

        @functools.wraps(func)
        def wrapper(*args: Any, **kwargs: Any) -> Any:
            # Create error collection for this function execution
            error_collection = ErrorCollection(
                function_name=func.__name__, args=args, kwargs=kwargs
            )

            # Mark decorator as active
            self._is_active = True
            self._current_error_collection = error_collection

            # Use error collection context to manage thread-local storage
            with ErrorCollectionContext(error_collection) as collection:
                try:
                    # Wrap in SystemExit handler
                    try:
                        result = func(*args, **kwargs)

                        # Check if any errors were collected (e.g., log errors)
                        # even though the function succeeded
                        if collection.has_errors():
                            process_error_collection(self, collection)  # type: ignore[arg-type]

                        return result

                    except SystemExit as e:
                        # Handle sys.exit() calls - add to error collection
                        handle_system_exit(e, func, collection)

                        # Process all collected errors
                        process_error_collection(self, collection)  # type: ignore[arg-type]

                        # Re-raise SystemExit
                        raise

                except Exception as e:
                    # Handle regular exceptions - add to error collection
                    self.consolidated_handlers.handle_exception(e, func)

                    # Process all collected errors
                    process_error_collection(self, collection)  # type: ignore[arg-type]

                    # Re-raise the original exception
                    raise

                finally:
                    # Mark decorator as inactive
                    self._is_active = False
                    self._current_error_collection = None

        return wrapper

    def handle_signal_termination(self, signum: int, frame: Any) -> None:
        """Handle signal-based termination by delegating to error handling module."""
        handle_signal_termination(self, signum, frame)

    def handle_exit_cleanup(self) -> None:
        """Handle exit cleanup by delegating to error handling module."""
        handle_exit_cleanup(self)

    def cleanup(self) -> None:
        """Clean up handlers and resources."""
        if hasattr(self, "consolidated_handlers"):
            self.consolidated_handlers.cleanup()

    def __del__(self) -> None:
        """Ensure cleanup on garbage collection."""
        with contextlib.suppress(Exception):
            self.cleanup()


def exc2issue(
    config: BugHunterConfig | None = None,
    **kwargs: Any,
) -> Callable[[Callable[..., Any]], Callable[..., Any]]:
    """Bug hunter decorator with comprehensive error handling and resilience.

    This decorator provides comprehensive error handling and resilience:

    **Consolidated Error Handling:**
    - Collects multiple errors from single function execution
    - Creates single comprehensive GitHub issues for related errors
    - HYBRID logic: single errors use deterministic titles, multiple errors use consolidated format

    **Multi-Layered Resilience:**
    - Catches sys.exit() calls and creates GitHub issues
    - Handles signal termination (SIGTERM, SIGINT, SIGHUP)
    - Background processing with retry logic for reliable issue creation
    - Graceful cleanup and shutdown handling

    **Configurable Behavior:**
    - Feature flags to enable/disable specific resilience features
    - Adjustable consolidation threshold
    - Full backward compatibility

    Args:
        config: Complete configuration object (preferred)
        **kwargs: Legacy parameters for backward compatibility including:
            repository, labels, assignee, assignees, github_token, gemini_api_key,
            enable_signal_handling, enable_exit_handling, enable_background_processing,
            consolidation_threshold, auth_config, processing_config

    Returns:
        Decorated function that creates GitHub issues on various error/termination types

    Raises:
        ValueError: If neither config nor repository is provided
    """
    decorator_instance = BugHunterDecorator(
        config,
        **kwargs,
    )

    return decorator_instance
