from functools import lru_cache
from itertools import count
import time
import unittest

from DLMS_SPODES.cosem_interface_classes import Parameter
from src.DLMS_SPODES.types import cdt, cst, ut
from src.DLMS_SPODES.cosem_interface_classes import collection, overview, ln_pattern
from src.DLMS_SPODES.obis import media_id
from src.DLMS_SPODES.exceptions import NeedUpdate, NoObject
from test.collection_init import col as col_


server_1_4_0 = collection.ParameterValue(
        par=bytes.fromhex("0000000201ff02"),
        value=cdt.OctetString(bytearray(b"1.4.0")).encoding)

server_1_5_15 = collection.ParameterValue(
        par=bytes.fromhex("0000000201ff02"),
        value=cdt.OctetString(bytearray(b"1.5.15")).encoding)

server_1_6_2 = collection.ParameterValue(
        par=bytes.fromhex("0000000201ff02"),
        value=cdt.OctetString(bytearray(b"1.6.2")).encoding)

firID_M2M_1 = collection.ParameterValue(
        par=bytes.fromhex("0000600101ff02"),
        value=cdt.OctetString(bytearray(b'M2M_1')).encoding)

firID_M2M_3 = collection.ParameterValue(
        par=bytes.fromhex("0000600101ff02"),
        value=cdt.OctetString(bytearray(b'M2M_3')).encoding)

# col_M2M1_1_5_15 = collection.get_collection(
#     manufacturer=b"KPZ",
#     server_type=serID_M2M_1,
#     server_ver=server_1_5_15)
#
# col_M2M1_1_6_2 = collection.get_collection(
#     manufacturer=b"KPZ",
#     server_type=serID_M2M_1,
#     server_ver=server_1_6_2)


class TestType(unittest.TestCase):

    def setUp(self) -> collection.Collection:
        col = collection.Collection(collection.ID(
            man=b'KPZ',
            f_id=collection.ParameterValue(
                par=b'',
                value=cdt.OctetString("4d324d5f33").encoding),
            f_ver=collection.ParameterValue(
                par=b'',
                value=cdt.OctetString(bytearray(b'1.3.0')).encoding)
        ))
        col.spec_map = "SPODES_3"
        reg = col.add(
            class_id=overview.ClassID.REGISTER,
            version=overview.VERSION_0,
            logical_name=cst.LogicalName("01 00 01 07 00 ff")
        )
        col.add(
            class_id=overview.ClassID.PROFILE_GENERIC,
            version=overview.VERSION_1,
            logical_name=cst.LogicalName("01 00 5e 07 04 ff")
        )
        reg.set_attr(2, cdt.DoubleLongUnsigned(1234567890).encoding)
        reg.set_attr(3, (-8, 33))
        activity_calendar_obj = col.add(
            class_id=overview.ClassID.ACTIVITY_CALENDAR,
            version=overview.VERSION_0,
            logical_name=cst.LogicalName.from_obis("0.0.13.0.0.255")
        )
        activity_calendar_obj.parse_attr(
            9,
            [
                ["1", [["01:00", "00 00 0A 00 64 FF", "1"],]],
                ["2", [["02:00", "00 00 0A 00 64 FF", "2"],]]
            ])
        script_obj = col.add(
            class_id=overview.ClassID.SCRIPT_TABLE,
            version=overview.VERSION_0,
            logical_name=cst.LogicalName("00 00 0a 00 64 ff")
        )
        script_obj.parse_attr(2, [['1', [['2', '3', "01 00 01 07 00 FF", '2', "0:"]]]])
        alarm1_obj = col.add(
            class_id=overview.ClassID.DATA,
            version=overview.VERSION_0,
            logical_name=cst.LogicalName("00 00 61 62 00 FF")
        )
        alarm1_obj.set_attr(2, 11)
        return col

    def test_ParVal(self):
        pv = collection.ParameterValue(
            par=bytes.fromhex("0000000201ff02"),
            value=cdt.OctetString(bytearray(b"1.5.15")).encoding)
        self.assertEqual(b'\x00\x00\x00\x02\x01\xff\x02\x00\t\x061.5.15', bytes(pv))
        pv2 = collection.ParameterValue.parse(b'\x00\x00\x00\x02\x01\xff\x02\x00\t\x061.5.15')
        print(bytes(pv2).hex())

    def test_CollectionID(self):
        ID = collection.ID(
            man=b'KPZ',
            f_id=firID_M2M_1,
            f_ver=server_1_5_15
        )

    def test_has_association(self):
        col = collection.Collection()
        sap = collection.enums.ClientSAP(0x10)
        self.assertRaises(NoObject, col.sap2association, (sap,))
        self.assertEqual(col.has_sap(sap), False)
        col = col_
        self.assertIsInstance(col.sap2association(sap), collection.AssociationLN)
        self.assertEqual(col.has_sap(sap), True)

    def test_get_type_from_class(self):
        ln = cst.LogicalName.from_obis("0.0.1.0.0.255")
        value = collection.get_interface_class(collection.common_interface_class_map, ut.CosemClassId(7), cdt.Unsigned(1))
        v1 = value(ln)
        print(value, v1)

    def test_add(self):
        cont = set()
        col = collection.Collection(
            man=b"KPZ",
            f_id=firID_M2M_1,
            f_ver=server_1_5_15
        )
        col2 = collection.Collection(
            man=b"KPZ",
            f_id=firID_M2M_1,
            f_ver=server_1_5_15
        )
        col.spec_map = col.get_spec()
        cont.add(col)
        cont.add(col2)
        ver_obj = col.add(class_id=ut.CosemClassId(1),
                          version=cdt.Unsigned(0),
                          logical_name=cst.LogicalName.from_obis("0.0.96.1.6.255"))
        ver_obj.set_attr(2, "33 2e 30")

    def test_ClassMap(self):
        print(hash(collection.DataMap))

    def test_get_object_list(self):
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=firID_M2M_1,
            server_ver=server_1_5_15)
        print(col)
        a = col.get_objects_list(collection.enums.ClientSAP(48))
        print(a)
        for i in range(10):
            col_new = collection.get_collection(
                manufacturer=b"KPZ",
                server_type=firID_M2M_3,
                server_ver=server_1_6_2)
            print(col_new)

    def test_save_type(self):
        """for template"""
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=firID_M2M_1,
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.15"))))
        col.save_type("test_save_type.typ")

    def test_to_xml4(self):
        """for template"""
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=firID_M2M_3,
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.4.15"))))
        col2 = collection.get_collection(
            manufacturer=b"102",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_3'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.3.30"))))
        clock_obj = col.get_object("0.0.1.0.0.255")
        clock_obj.set_attr(3, 120)
        act_cal = col.get_object("0.0.13.0.0.255")
        act_cal.day_profile_table_passive.append((1, [("11:00", "1.1.1.1.1.1", 1)]))
        used = {
            clock_obj.logical_name: {3},
            act_cal.logical_name: {9}
        }
        print(used)
        collection.to_xml4(
            collections=[col, col2],
            file_name="test_to_xml4.xml",
            used=used)
        print(used)

    def test_collection_from_xml4(self):
        cols, used, verif = collection.from_xml4("test_to_xml4.xml")
        print(cols, used, verif)

    def test_get_writable_dict(self):
        """use in template"""
        col = col_M2M1_1_5_15
        ret = col.get_writable_attr()
        print(ret)

    def test_AssociationLN(self):
        col = collection.Collection()
        col.set_manufacturer(b"KPZ")
        col.set_firm_ver(server_1_4_0)
        col.spec_map = col.get_spec()
        ass_obj = col.add(class_id=overview.ClassID.ASSOCIATION_LN,
                          version=overview.VERSION_1,
                          logical_name=cst.LogicalName.from_obis("0.0.40.0.3.255"))
        ver_obj = col.add(class_id=overview.ClassID.DATA,
                          version=overview.VERSION_0,
                          logical_name=cst.LogicalName.from_obis("0.0.0.2.1.255"))
        self.assertRaises(NeedUpdate,
                          ver_obj.set_attr,
                          2,
                          bytes.fromhex("0905312e352e30"))
        ass_obj.set_attr(6, bytes.fromhex("090760857405080202"))
        print(ass_obj)

    def test_get_descriptor(self):
        col = col_M2M1_1_6_2
        p = col.get_object("1.0.94.7.4.255")
        desc = p.get_attr_descriptor(2, True)
        print(col)

    def test_get_manufactures(self):
        a = collection.get_manufactures_container()
        print(a)

    def test_hash(self):
        col1 = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.4.13"))))
        col2 = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_3'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.4.12"))))
        c = [col1]
        print(hash(col1), hash(col2), col1 == col2, col2 in c)

    def test_get_all_collection(self):
        """try get all collection"""
        for i in collection.get_manufactures_container().values():
            for j in i.values():
                for k in j.values():
                    print(k.path)
                    col = collection.Collection.from_xml(k)
                    print(col)

    def test_copy_benchmark(self):
        """best:
        0.50.2: 0.1568sec
        0.50.3: 0.0356
        0.50.4: 0.0354
        """
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_3'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.4.13"))))
        n = 100
        s = time.perf_counter()
        for i in range(n):
            col.copy()
        print((time.perf_counter()-s)/n)

    def test_set_date_for_calibrator(self):
        import datetime

        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.15"))))
        print(col)
        manufacture_date_obj = col.get_object('0.0.96.1.4.255')
        manufacture_date_obj.set_attr(2, datetime.date.today().strftime("%d.%m.%y"))
        print(manufacture_date_obj.value)

    def test_change_version(self):
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.15"))))
        col.change_association_version(cdt.Unsigned(2))
        print(col)

    def test_for_change_version_in_not_valid_ass(self):
        arr: collection.ProfileGenericVer1.A_ELEMENTS[1].DATA_TYPE = cdt.Array(b"\x01\x81\x9a\x02\x04\x12\x00\x08\x11\x00\t\x06\x00\x00\x01\x00\x00\xff\x02\x02\x01\t\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x01\x06\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x02\x0f\x03\x16\x01\x02\x02\x0f\x04\x16\x01\x02\x02\x0f\x05\x16\x01\x02\x02\x0f\x06\x16\x01\x02\x04\x12\x00\x08\x11\x00\t\x06\x00\x01\x01\x00\x00\xff\x02\x02\x01\t\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x01\x06\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x02\x0f\x03\x16\x01\x02\x02\x0f\x04\x16\x01\x02\x02\x0f\x05\x16\x01\x02\x02\x0f\x06\x16\x01\x02\x04\x12\x00\x0f\x11\x02\t\x06\x00\x00(\x00\x00\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x06\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x02\x0f\x03\x16\x01\x02\x02\x0f\x04\x16\x01\x02\x02\x0f\x05\x16\x01\x02\x02\x0f\x06\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x01\x00c\x01\x00\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x01\x00^\x07\x04\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x01\x00b\x02\x00\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x01\x00^\x07\x02\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x01\x00b\x01\x00\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x01\x00^\x07\x01\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x01\x00^\x07\x03\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x01\x00^\x07\x00\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x01\x00^\x07\x05\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00F\x11\x00\t\x06\x00\x00`\x03\n\xff\x02\x02\x01\x04\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00G\x11\x00\t\x06\x00\x00\x11\x00\x00\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x00\x02\x04\x12\x00G\x11\x00\t\x06\x00\x00\x11\x00\x01\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x00\x02\x04\x12\x00G\x11\x00\t\x06\x00\x00\x11\x00\x02\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x00\x02\x04\x12\x00G\x11\x00\t\x06\x00\x00\x11\x00\x03\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x00\x02\x04\x12\x00G\x11\x00\t\x06\x00\x00\x11\x00\x04\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x00\x02\x04\x12\x00G\x11\x00\t\x06\x00\x00\x11\x00\x05\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00\x80e\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x03\x11\x00\t\x06\x00\x00`\t\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00[\x07\x84\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\t\x11\x00\t\x06\x00\x00\n\x00j\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00\x80\x96\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x03\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x04\x11\x00\t\x06\x01\x00\x01\x06\x00\xff\x02\x02\x01\x05\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\x00\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\x01\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\x02\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\x03\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\x04\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\x07\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\x1a\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\x1b\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\x05\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\x06\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\t\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\x08\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x00\t\x06\x00\x00cb\r\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00*\x00\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x01\x01\x00\x00\x01\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00^\x07\x01\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x01\x00\x00\x08\x04\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x02\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x06\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x03\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\n\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x04\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x01\x00\x00\x04\x02\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x01\x00\x00\x04\x03\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x01\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x01\x00\x00\x03\x03\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x01\x00\x00\x03\x04\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x07\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x08\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00\x80d\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\t\x11\x00\t\x06\x00\x00\n\x00d\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x16\x11\x00\t\x06\x00\x00\x0f\x00\x00\xff\x02\x02\x01\x04\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x01\x00\x02\x04\x12\x00\x14\x11\x00\t\x06\x00\x00\r\x00\x00\xff\x02\x02\x01\n\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x0b\x11\x00\t\x06\x00\x00\x0b\x00\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x12\x11\x00\t\x06\x00\x00,\x00\x00\xff\x02\x02\x01\x07\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x01\x04\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x02\x0f\x03\x16\x01\x02\x02\x0f\x04\x16\x01\x02\x04\x12\x00\x12\x11\x00\t\x06\x00\x00,\x00\x80\xff\x02\x02\x01\x07\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x01\x04\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x02\x0f\x03\x16\x01\x02\x02\x0f\x04\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x1f\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x003\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00G\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00[\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0b\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00 \x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x004\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00H\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0e\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0c\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0c\x1b\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0c\x07\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0c\x07\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0c\x07\x03\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00Q\x07\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00Q\x07\n\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00Q\x07\x15\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00Q\x07\x04\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00Q\x07\x0f\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00Q\x07\x1a\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0f\x10\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0f\x10\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0f\x06\x80\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x89\x06\x80\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0f\x06\x81\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0f\x06\x82\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\t\x06\x80\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x08\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x08\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x08\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x08\x03\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x08\x04\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x02\x08\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x02\x08\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x02\x08\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x02\x08\x03\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x02\x08\x04\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x03\x08\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x03\x08\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x03\x08\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x03\x08\x03\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x03\x08\x04\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x04\x08\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x04\x08\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x04\x08\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x04\x08\x03\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x04\x08\x04\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00!\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00I\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x005\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\t\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x1d\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x001\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00E\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x15\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00)\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00=\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x03\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x17\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00+\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00?\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\r\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0c'\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x00\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x01\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x02\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x03\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x04\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x05\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x06\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x07\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x08\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\t\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\n\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x0b\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x0c\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\r\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x0e\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x0f\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x10\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x11\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x12\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x13\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x00\x06\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x00\x06\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x00\x06\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x00\x06\x03\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x02\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x00\x06\x04\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0f\x10\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01")
        arr = cdt.Array(b"\x01\x81\x9a\x02\x04\x12\x00\x08\x11\x00\t\x06\x00\x00\x01\x00\x00\xff\x02\x02\x01\t\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x01\x06\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x02\x0f\x03\x16\x01\x02\x02\x0f\x04\x16\x01\x02\x02\x0f\x05\x16\x01\x02\x02\x0f\x06\x16\x01\x02\x04\x12\x00\x08\x11\x00\t\x06\x00\x01\x01\x00\x00\xff\x02\x02\x01\t\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x01\x06\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x02\x0f\x03\x16\x01\x02\x02\x0f\x04\x16\x01\x02\x02\x0f\x05\x16\x01\x02\x02\x0f\x06\x16\x01\x02\x04\x12\x00\x0f\x11\x02\t\x06\x00\x00(\x00\x00\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x06\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x02\x0f\x03\x16\x01\x02\x02\x0f\x04\x16\x01\x02\x02\x0f\x05\x16\x01\x02\x02\x0f\x06\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x01\x00c\x01\x00\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x01\x00^\x07\x04\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x01\x00b\x02\x00\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x01\x00^\x07\x02\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x01\x00b\x01\x00\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x01\x00^\x07\x01\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x01\x00^\x07\x03\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x01\x00^\x07\x00\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x01\x00^\x07\x05\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00F\x11\x00\t\x06\x00\x00`\x03\n\xff\x02\x02\x01\x04\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00G\x11\x00\t\x06\x00\x00\x11\x00\x00\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x00\x02\x04\x12\x00G\x11\x00\t\x06\x00\x00\x11\x00\x01\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x00\x02\x04\x12\x00G\x11\x00\t\x06\x00\x00\x11\x00\x02\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x00\x02\x04\x12\x00G\x11\x00\t\x06\x00\x00\x11\x00\x03\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x00\x02\x04\x12\x00G\x11\x00\t\x06\x00\x00\x11\x00\x04\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x00\x02\x04\x12\x00G\x11\x00\t\x06\x00\x00\x11\x00\x05\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x02\x03\x0f\x0b\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00\x80e\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x03\x11\x00\t\x06\x00\x00`\t\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00[\x07\x84\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\t\x11\x00\t\x06\x00\x00\n\x00j\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00\x80\x96\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x03\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x04\x11\x00\t\x06\x01\x00\x01\x06\x00\xff\x02\x02\x01\x05\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\x00\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\x01\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\x02\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\x03\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\x04\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\x07\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\x1a\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\x1b\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\x05\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\x06\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\t\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\x08\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x07\x11\x01\t\x06\x00\x00cb\r\xff\x02\x02\x01\x08\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00*\x00\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x01\x01\x00\x00\x01\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00^\x07\x01\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x01\x00\x00\x08\x04\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x02\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x06\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x03\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\n\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x04\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x01\x00\x00\x04\x02\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x01\x00\x00\x04\x03\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x01\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x01\x00\x00\x03\x03\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x01\x00\x00\x03\x04\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x07\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00`\x01\x08\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\x01\x11\x00\t\x06\x00\x00\x80d\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x00\x02\x04\x12\x00\t\x11\x00\t\x06\x00\x00\n\x00d\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x16\x11\x00\t\x06\x00\x00\x0f\x00\x00\xff\x02\x02\x01\x04\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x01\x00\x02\x04\x12\x00\x14\x11\x00\t\x06\x00\x00\r\x00\x00\xff\x02\x02\x01\n\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x03\x00\x02\x03\x0f\t\x16\x03\x00\x02\x03\x0f\n\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x0b\x11\x00\t\x06\x00\x00\x0b\x00\x00\xff\x02\x02\x01\x02\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x01\x02\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x04\x12\x00\x12\x11\x00\t\x06\x00\x00,\x00\x00\xff\x02\x02\x01\x07\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x01\x04\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x02\x0f\x03\x16\x01\x02\x02\x0f\x04\x16\x01\x02\x04\x12\x00\x12\x11\x00\t\x06\x00\x00,\x00\x80\xff\x02\x02\x01\x07\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x02\x03\x0f\x04\x16\x03\x00\x02\x03\x0f\x05\x16\x03\x00\x02\x03\x0f\x06\x16\x03\x00\x02\x03\x0f\x07\x16\x03\x00\x01\x04\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x02\x0f\x03\x16\x01\x02\x02\x0f\x04\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x1f\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x003\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00G\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00[\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0b\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00 \x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x004\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00H\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0e\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0c\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0c\x1b\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0c\x07\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0c\x07\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0c\x07\x03\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00Q\x07\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00Q\x07\n\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00Q\x07\x15\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00Q\x07\x04\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00Q\x07\x0f\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00Q\x07\x1a\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0f\x10\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0f\x10\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0f\x06\x80\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x89\x06\x80\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0f\x06\x81\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0f\x06\x82\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\t\x06\x80\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x08\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x08\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x08\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x08\x03\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x08\x04\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x02\x08\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x02\x08\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x02\x08\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x02\x08\x03\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x02\x08\x04\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x03\x08\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x03\x08\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x03\x08\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x03\x08\x03\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x03\x08\x04\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x04\x08\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x04\x08\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x04\x08\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x04\x08\x03\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x04\x08\x04\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00!\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00I\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x005\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\t\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x1d\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x001\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00E\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x15\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00)\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00=\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x03\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x17\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00+\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00?\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\r\x07\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0c'\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x00\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x01\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x02\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x03\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x04\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x05\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x06\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x07\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x08\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\t\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\n\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x0b\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x0c\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\r\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x0e\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x0f\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x10\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x11\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x12\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x80\x00\x13\x00\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x00\x06\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x00\x06\x01\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x00\x06\x02\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x00\x06\x03\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x01\x02\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x00\x06\x04\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01\x02\x04\x12\x00\x03\x11\x00\t\x06\x01\x00\x0f\x10\x00\xff\x02\x02\x01\x03\x02\x03\x0f\x01\x16\x03\x00\x02\x03\x0f\x02\x16\x03\x00\x02\x03\x0f\x03\x16\x03\x00\x01\x01\x02\x02\x0f\x01\x16\x01")
        for i in arr:
            print(i)
            if int(i[0]) == 7 and int(ver := i[1]) == 0:
                ver.set(1)
                print(ver)
            if int(i[0]) == 15:
                for j in i[3][0]:
                    if int(j[0]) in (3, 4, 5, 6, 8):
                        j[1].set(1)
        print(arr.encoding.hex())

        struct = cdt.Structure(b'\x02\x04\x12\x00\x0f\x11\x02\t\x06\x00\x00(\x00\x01\xff\x02\x02\x01\x0b\x02\x03\x0f\x01\x16\x01\x00\x02\x03\x0f\x02\x16\x01\x00\x02\x03\x0f\x03\x16\x01\x00\x02\x03\x0f\x04\x16\x01\x00\x02\x03\x0f\x05\x16\x01\x00\x02\x03\x0f\x06\x16\x01\x00\x02\x03\x0f\x07\x16\x03\x00\x02\x03\x0f\x08\x16\x01\x00\x02\x03\x0f\t\x16\x01\x00\x02\x03\x0f\n\x16\x01\x00\x02\x03\x0f\x0b\x16\x01\x00\x01\x06\x02\x02\x0f\x01\x16\x01\x02\x02\x0f\x02\x16\x01\x02\x02\x0f\x03\x16\x01\x02\x02\x0f\x04\x16\x01\x02\x02\x0f\x05\x16\x01\x02\x02\x0f\x06\x16\x01')
        for i in struct[3]:
            for j in i:
                print(j)

    def test_get_class_names(self):
        type_ = "4d324d5f3354"
        ver = "1.5.7"
        man = b"KPZ"
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_3T'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.7"))))
        for obj in col:
            print(str(obj.CLASS_ID))

    def test_create_manufacture_obj(self):
        col = collection.Collection()
        obj = col.add(
            class_id=overview.ClassID.DATA,
            version=overview.VERSION_0,
            logical_name=cst.LogicalName.from_obis("0.128.25.6.0.255")
        )
        print(obj)

    def test_get_phases_amount(self):
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.7"))))
        self.assertEqual(col.get_n_phases(), 1, "check amount")
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_3'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.7"))))
        col.get_class_version()
        self.assertEqual(col.get_n_phases(), 3, "check amount 3")
        kpz_ip = col.get_object("0.128.154.0.0.255")
        kpz_ip.set_attr(2, (1, ["0.0.0.0", "0.0.0.1"]))
        print(kpz_ip)
        print(overview.ClassID.get_all_id())

    def test_get_object_list_desc(self):
        self.assertEqual(collection.AttrDesc.OBJECT_LIST.contents, b'\x00\x0f\x00\x00(\x00\x00\xff\x02\x00', "check cached object_list")

    def test_get_relation_group(self):
        ln = cst.LogicalName.from_obis("0.0.1.0.0.255")
        self.assertEqual(collection.get_relation_group(ln), collection.media_id.CLOCK_OBJECTS, "check_group")
        b1 = collection.get_media_id(ln)
        ln = cst.LogicalName.from_obis("0.0.94.1.1.255")
        a = collection.get_relation_group(ln)
        b2 = collection.get_media_id(ln)
        print(a)

    def test_group_objects(self):
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.7"))))
        c = col.filter_by_ass(1)
        print(c)
        d = list(collection.get_filtered(c, (collection.ln_pattern.DEVICE_ID,)))
        print(d)
        e = list(collection.class_id_filter(d, collection.ClassID.DATA))
        print(e)
        f = list(collection.get_filtered(c, (collection.ln_pattern.DEVICE_ID, collection.ClassID.CLOCK)))
        print(f)
        g = list(collection.get_filtered(col, (collection.ClassID.CLOCK, collection.media_id.ELECTRICITY)))
        print(g)

    def test_media_id(self):
        a = collection.media_id.MediaId.from_int(1)
        b = collection.media_id.ELECTRICITY
        print(a is b)
        print(b)
        c = media_id.THRESHOLD_VALUES
        print(c)

        import inspect
        import sys

        clsmembers = inspect.getmembers(media_id, inspect.isclass)
        for it in clsmembers:
            print(F'{it[0]} = ""')

    def test_get_tree(self):
        from src.DLMS_SPODES.obis.media_id import MediaId
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.15"))))
        res = collection.get_object_tree(
            objects=col.filter_by_ass(3),
            # mode="gmc"
            mode="c"
        )
        print(res)

    def test_sort_objects(self):
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.15"))))
        res = collection.get_sorted(col.filter_by_ass(3), mode="l")
        res = collection.get_sorted(col.filter_by_ass(3), mode="n")
        res = collection.get_sorted(col.filter_by_ass(3), mode="c")
        print(res)

    def test_get_filtered(self):
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.6.2"))))
        res = collection.get_filtered(
            objects=col.filter_by_ass(3),
            keys=(collection.ClassID.DATA, collection.ClassID.REGISTER, collection.media_id.ELECTRICITY))
        print(res)
        res2 = collection.get_filtered(
                objects=col,
                keys=(ln_pattern.DEVICE_ID,
                      ln_pattern.PROGRAM_ENTRIES))
        print(res2)
        res3 = collection.get_filtered(
            objects=col,
            keys=(
                ln_pattern.TARIFFICATION_SCRIPT_TABLE,
                overview.ClassID.SCHEDULE,
                overview.ClassID.ACTIVITY_CALENDAR,
                overview.ClassID.SPECIAL_DAYS_TABLE,
            )
        )
        print(res3)
        res4 = collection.get_filtered(
            objects=col,
            keys=(overview.ClassID.IEC_HDLC_SETUP, collection.Channel(1),)
        )
        print(res4)

    def test_get_attr_tree(self):
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.15"))))
        res = col.get_attr_tree(
            ass_id=3,
            obj_mode="",
            # obj_filter=(collection.ClassID.REGISTER, collection.media_id.ABSTRACT),
            # obj_filter=(collection.ClassID.REGISTER,),
            obj_filter=(collection.ClassID.PROFILE_GENERIC, collection.ClassID.REGISTER),
            sort_mode="l",
            af_mode="mlr",
            oi_filter=((overview.ClassID.REGISTER, (2,)),)
        )
        print(res)

    def test_LNPattern(self):
        pattern0 = ln_pattern.NON_CURRENT_ASSOCIATION

        pattern = collection.LNPattern.parse("a.2.(1).4.5.9")
        pattern2 = collection.LNPattern.parse("a.2.!(4).4.5.9")
        pattern3 = collection.LNPattern.parse("a.2.(1, 200, 43, 51, 3-8).4.5.9")
        pat3 = ln_pattern.GENERAL_AND_SERVICE_ENTRY
        pattern2 = collection.LNPattern.parse("a.2.31.4.5.9")
        print(pattern)
        print(cst.LogicalName.from_obis("1.2.31.4.5.9") in (pattern, pattern2))
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.15"))))
        res = collection.get_filtered(
            objects=col.filter_by_ass(3),
            # keys=(collection.LNPattern.parse("a.b.(14-20).d.e.f"),)
            # keys=(pattern2,)
            keys=(pat3,)
        )
        print(res)

    def test_get_scaler_unit(self):
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.15"))))

        for obj in col:
            for i, a_val in obj.get_index_with_attributes():
                if i == 1:
                    continue
                res = col.get_scaler_unit(
                    obj=obj,
                    par=bytes((i,)))
                print(F"{obj.CLASS_ID} {obj.logical_name.get_report()}:{i} {res}")

    def test_get_report(self):
        col = self.create_collection()
        rep_count = count()
        # for obj in collection.get_filtered(col, (collection.ClassID.LIMITER,)):
        for obj in col:
            for i, _ in obj.get_index_with_attributes():
                if i == 1:
                    continue
                par = bytearray([i])
                value = obj.get_attr(i)
                stack = [(value, par)]
                while stack:
                    value, par = stack.pop()
                    rep = col.get_report(obj, bytes(par), value)
                    print(F"{obj.logical_name.get_report()} {', '.join(map(str, par))}: {rep=}")
                    next(rep_count)
                    match value:
                        case cdt.Structure():
                            for j, el in enumerate(value):
                                new_par = par.copy()
                                new_par.append(j)
                                stack.append((el, new_par))
                        case cdt.Array():
                            new_par = par.copy()
                            new_par.append(0)
                            for j, el in enumerate(value):
                                stack.append((el, new_par))
        print(F"{col.get_scaler_unit.cache_info()=}")
        print(F"{collection.get_unit.cache_info()=}")
        print(F"{rep_count=}")

    def test_par2rep(self):
        col = self.create_collection()
        par = Parameter.parse("0.0.13.0.0.255:9").extend(0, 0)
        data = col.par2data(par).unwrap()
        rep1 = col.par2rep(par, data)
        print(str(col[par.obis].get_attr_element(1)))
        print(rep1)

    def test_get_profile_s_u(self):
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.5.15"))))
        res = col.get_profile_s_u(
            obj=col.get_object("1.0.99.2.0.255"),
            mask=None
        )
        print(res)

    def test_Spodes_locker_obj(self):
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.6.2"))))
        locker = col.get_object("0.0.96.4.3.255")
        locker.set_attr(2, 0)
        rep = col.get_report(locker, b'\x02', locker.value)
        print(locker, rep)

    def test_report_enums(self):
        from src.DLMS_SPODES.cosem_interface_classes.association_ln.mechanism_id import MechanismIdElement
        col = collection.get_collection(
            manufacturer=b"KPZ",
            server_type=collection.ParameterValue(
                par=bytes.fromhex("0000600102ff02"),
                value=cdt.OctetString(bytearray(b'M2M_1'))),
            server_ver=collection.ParameterValue(
                par=bytes.fromhex("0000000201ff02"),
                value=cdt.OctetString(bytearray(b"1.6.2"))))
        clock_obj = col.get_object('0.0.1.0.0.255')
        clock_obj.set_attr(9, 3)
        print(clock_obj.clock_base)
        print(col.get_report(clock_obj, b'\x09', clock_obj.clock_base))
        ass_obj = col.get_object("0.0.40.0.1.255")
        print(ass_obj.authentication_mechanism_name.authentication_mechanism_name_element)
        m_id = MechanismIdElement(0)
        m_id.set('2')
        print(F"{m_id=}")
        vol_ev_obj = col.get_object("0.0.96.11.0.255")
        vol_ev_obj.set_attr(2, 33)
        print(col.get_report(vol_ev_obj, b'\x02', vol_ev_obj.value))
        col.add(collection.ClassID.DATA, overview.VERSION_0, cst.LogicalName.from_obis("0.0.96.5.1.255"))
        vol_ev_obj = col.get_object("0.0.96.5.1.255")
        vol_ev_obj.set_attr(2, 5)
        print(col.get_report(vol_ev_obj, b'\x02', vol_ev_obj.value))
        vol_ev_obj = col.get_object("0.0.96.5.4.255")
        vol_ev_obj.set_attr(2, 1)
        print(col.get_report(vol_ev_obj, b'\x02', vol_ev_obj.value))

    def test_create(self):
        col = self.create_collection()
        print(col)
        obj = col.get_object("0.0.13.0.0.255")
        # col.get_report(obj, b'\x09', )

    def test_Alarm1(self):
        col = self.create_collection()
        obj = col.get_object("0.0.97.98.0.255")
        self.assertTrue(obj.value.get_report().msg.startswith("(11)"))
        self.assertEqual(tuple(obj.value), (1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0))
        self.assertEqual(obj.value[1], 1)
        obj.value[1] = 0
        obj.value[30] = 1
        print(obj.value)
