# langchain-markitdown

[![Build](https://github.com/nsasto/langchain-markitdown/actions/workflows/publish.yml/badge.svg)](https://github.com/nsasto/langchain-markitdown/actions)
[![License](https://img.shields.io/github/license/nsasto/langchain-markitdown)](https://github.com/nsasto/langchain-markitdown/blob/main/LICENSE)
[![PyPI version](https://img.shields.io/pypi/v/langchain-markitdown)](https://pypi.org/project/langchain-markitdown/)
[![Contributors](https://img.shields.io/github/contributors/nsasto/langchain-markitdown)](https://github.com/nsasto/langchain-markitdown/graphs/contributors)

# Markitdown LangChain Integration

This project provides document loaders that seamlessly integrate the Markitdown library with LangChain. Markitdown excels at converting various document types (DOCX, PPTX, XLSX, and more) into Markdown format. These loaders empower you to effortlessly load, process, and analyze these documents within your LangChain pipelines.

MarkItDown is a lightweight Python utility designed for converting diverse file formats into Markdown, optimized for use with Large Language Models (LLMs) and related text analysis workflows. It shares similarities with `textract` but distinguishes itself by prioritizing the preservation of crucial document structure and content as Markdown. This includes headings, lists, tables, links, and more. While the output is generally readable, its primary purpose is to be consumed by text analysis tools, rather than serving as a high-fidelity document conversion solution for human readers.

Explore the MarkItDown project on GitHub: [https://github.com/microsoft/markitdown](https://github.com/microsoft/markitdown)

Currently, MarkItDown supports:

- PDF
- PowerPoint
- Word
- Excel
- Images (EXIF metadata and OCR)
- Audio (EXIF metadata and speech transcription)
- HTML
- Text-based formats (CSV, JSON, XML)
- ZIP files (iterates over contents)
- YouTube URLs
- EPUBs
- ...and many more!

While this project borrows liberally from the amazing LangChain and Markitdown projects, it is not affiliated with either in any way.

## Installation

Install the package using pip:

```bash
pip install langchain-markitdown
```

## Usage & Features

The loaders expose a consistent interface that mirrors LangChain's built-in ones but leverage the MarkItDown converters under the hood. Highlights:

- Centralized metadata handling: every loader emits `success`, `conversion_success`, `content_type`, and MarkItDown's native metadata (page count, attachments, output type, etc.).
- Flexible splitting modes:
  - `split_by_page=True` partitions per-page/worksheet/slide output when MarkItDown provides page data.
  - `headers_to_split_on=[("#", "Header 1"), ...]` applies `MarkdownHeaderTextSplitter` to create logical sections from the converted Markdown.
- Loader-specific metadata enrichment, e.g. DOCX core properties, XLSX workbook metadata via `openpyxl`, PPTX slide/image counts, and more.

### Usage

### Specific Examples

#### DOCX

```
from langchain_markitdown import DocxLoader, PptxLoader, XlsxLoader

loader = DocxLoader("path/to/your/document.docx")
documents = loader.load()

# Split by pages or headers when needed
documents_by_page = DocxLoader(
    "path/to/your/document.docx",
    split_by_page=True,
).load()

documents_by_headers = loader.load(
    headers_to_split_on=[("#", "H1"), ("##", "H2")],
)
```

#### PPTX

```
from langchain_markitdown import PptxLoader

loader = PptxLoader("path/to/your/presentation.pptx")
documents = loader.load()
```

#### XLSX

```
from langchain_markitdown import XlsxLoader

loader = XlsxLoader("path/to/your/spreadsheet.xlsx")
documents = loader.load()
```

## Metadata

The `Document` objects returned by the loaders include the following metadata:

- `source`: The path to the source file.
- `file_name`: The name of the source file.
- `file_size`: The size of the source file in bytes.
- `success` / `conversion_success`: Boolean flags that include failure details when conversion fails.
- `content_type`: Semantic tags such as `document_full`, `document_section`, `document_page`, `workbook`, `worksheet`, or `presentation_slide`.
- `page_number` / `worksheet`: When splitting by page/worksheet.
- `markitdown_metadata`: Raw metadata surfaced by MarkItDown (attachments, page counts, conversion format, etc.).
- Loader-specific enrichments:
  - DOCX: author/title/subject/keywords/category/revision timestamps.
  - XLSX: workbook properties (author, title, subject, description, keywords, category).
  - PPTX: slide counts plus counts for images, tables, charts, and text boxes.

When header-based splitting is enabled, LangChain's header metadata is injected automatically.

## Development & Testing

Install dependencies (and the package itself in editable mode) before running tests:

```bash
python -m pip install -r requirements.txt
python -m pip install -e .
pytest
```

## Contributing

Contributions are welcome! Please fork the repository and submit a pull request with your changes.

## License

[MIT License](LICENSE)

## Trademarks

Markitdown, and so this project by extension, may contain trademarks or logos for projects, products, or services. Authorized use of Microsoft trademarks or logos is subject to and must follow Microsoft's Trademark & Brand Guidelines. Use of Microsoft trademarks or logos in modified versions of this project must not cause confusion or imply Microsoft sponsorship. Any use of third-party trademarks or logos are subject to those third-party's policies.
