# H-Matrix Field Evaluation Implementation Design

**Date**: 2025-11-08
**Status**: Design Review
**Goal**: Implement H-matrix acceleration for rad.Fld() to achieve 50-200x speedup

---

## 1. Current Situation Analysis

### 1.1. Existing H-Matrix Implementation

**File**: `src/core/radintrc_hmat.cpp` ✅ **Working**
- **Purpose**: Solver acceleration (rad.Solve)
- **Structure**: N×N H-matrix (source×source interaction)
- **Function**: Computes demagnetization field N·M
- **Performance**: 6-10x speedup, 30x memory reduction
- **Status**: Fully operational

**File**: `src/core/radhmat.cpp` ❌ **Incomplete**
- **Purpose**: Field evaluation acceleration (rad.Fld)
- **Structure**: M×N H-matrix (observation×source points)
- **Function**: Should compute B field at arbitrary points
- **Current Issue**: B_comp() calls B_comp_direct() instead of using H-matrix
- **Status**: Disabled in CMakeLists.txt (line 53)

### 1.2. Why radhmat.cpp Was Disabled

```cmake
# CMakeLists.txt:53
# ${CORE_DIR}/radhmat.cpp  # Temporarily disabled - field source H-matrix (has API compatibility issues)
```

**Identified Issues**:
1. ❌ B_comp() doesn't actually use H-matrix
2. ❌ Geometry extraction incomplete (center positions incorrect)
3. ❌ Kernel function simplified (not full Biot-Savart law)
4. ❌ API compatibility with existing radTg3d hierarchy

### 1.3. Current Field Evaluation Flow

```
rad.Fld(magnet, 'b', point)
    ↓
radTApplication::ComputeField()  (radapl3.cpp:35)
    ↓
radTField Field(...)
    ↓
radTg3d::B_comp()  (for each source element)
    ↓
Direct summation: O(M × N)
```

---

## 2. H-Matrix Fundamentals

### 2.1. H-Matrix Structure for Field Evaluation

**Solver H-Matrix** (radintrc_hmat.cpp):
```
H: N × N matrix (source × source)
Purpose: M = (I + H·N)^-1 · M0
Elements: N_ij = demagnetization tensor
```

**Field H-Matrix** (radhmat.cpp, to be implemented):
```
H: M × N matrix (observation × source)
Purpose: B(r_i) = Σ_j H_ij · m_j
Elements: H_ij = Biot-Savart kernel K(r_i, r_j)
```

### 2.2. Biot-Savart Kernel

For magnetic field B at observation point r from magnetic moment m at position r':

```
B(r) = (μ₀/4π) · [3(m·r̂)r̂ - m] / r³

where:
  r = r_obs - r_source
  r̂ = r / |r|
  μ₀ = 4π × 10^-7 T·m/A
```

**Matrix representation**:
```
B_x     K_xx  K_xy  K_xz     m_x
B_y  =  K_yx  K_yy  K_yz  ·  m_y
B_z     K_zx  K_zy  K_zz     m_z

where K_ij is the 3×3 Biot-Savart tensor
```

### 2.3. H-Matrix Approximation

**Key concept**: Far-field interactions can be approximated with low-rank matrices.

For admissible blocks (well-separated clusters):
```
K_ij ≈ U · V^T

where:
  U: M_cluster × rank (observation side basis)
  V: N_cluster × rank (source side basis)
  rank: typically 5-20 for magnetostatic problems
```

**Complexity**:
- Direct sum: O(M × N)
- H-matrix: O(M × log N) for construction, O(M × log N) for evaluation

---

## 3. Design Overview

### 3.1. Two-Level Architecture

```
Level 1: Single Point Evaluation
  rad.Fld(magnet, 'b', [x, y, z])
  ↓
  Construct temporary M×N H-matrix (M=1)
  ↓
  Evaluate using H-matrix-vector product
  ↓
  Speedup: ~2x (overhead dominates for M=1)

Level 2: Batch Evaluation
  rad.Fld(magnet, 'b', [[x1,y1,z1], [x2,y2,z2], ...])
  ↓
  Construct M×N H-matrix (M=1000)
  ↓
  Evaluate all points using single H-matrix
  ↓
  Speedup: 50-200x (amortized H-matrix construction)
```

### 3.2. H-Matrix Caching Strategy

**Problem**: Constructing H-matrix for every rad.Fld() call is expensive.

**Solution**: Cache H-matrix for reuse.

```cpp
class radTHMatrixFieldSource {
private:
	// Cached H-matrix for fixed observation points
	std::unique_ptr<hacapk::HMatrix> cached_hmatrix;
	std::vector<TVector3d> cached_observation_points;

	// Check if observation points match cache
	bool IsCacheValid(const std::vector<TVector3d>& obs_points);

public:
	// Build H-matrix for observation points
	void BuildFieldHMatrix(const std::vector<TVector3d>& obs_points);

	// Evaluate field using cached H-matrix
	void EvaluateField(std::vector<TVector3d>& B_values);
};
```

**Cache policy**:
- Cache H-matrix if M > 100 (worthwhile for batch evaluation)
- Invalidate cache if observation points change
- Reuse cache for repeated evaluations (e.g., NGSolve integration)

---

## 4. Implementation Plan

### 4.1. Phase 1: Fix Geometry Extraction ✅

**File**: `src/core/radhmat.cpp:126-184`

**Current Issue**:
```cpp
// Line 157: Incorrect!
center = TVector3d(0, 0, 0);  // Should get actual center
```

**Fix**:
```cpp
bool radTHMatrixFieldSource::ExtractGeometry()
{
	for(const auto& pair : source_elements) {
		radTg3d* elem = static_cast<radTg3d*>(pair.second.rep);

		// Get actual center using radTrans
		TVector3d center;
		elem->GetCenterPoint(center);  // Use proper method

		// Get magnetization
		TVector3d M = elem->Magn;

		// Get volume
		double volume = elem->Volume();

		// Magnetic moment = M × Volume
		TVector3d moment = M * volume;

		// Store
		element_positions.push_back(center.x);
		element_positions.push_back(center.y);
		element_positions.push_back(center.z);

		element_moments.push_back(moment.x);
		element_moments.push_back(moment.y);
		element_moments.push_back(moment.z);
	}

	return true;
}
```

### 4.2. Phase 2: Implement Biot-Savart Kernel ✅

**File**: `src/core/radhmat.cpp:256-285`

**Current Issue**: Simplified scalar kernel

**Fix**: Full 3×3 tensor kernel

```cpp
struct BiotSavartKernel {
	int component_row;  // 0=x, 1=y, 2=z
	int component_col;  // 0=x, 1=y, 2=z
	std::vector<TVector3d>* source_positions;
	std::vector<TVector3d>* obs_positions;
	std::vector<TVector3d>* magnetic_moments;

	double operator()(int i, int j) const {
		const TVector3d& r_obs = (*obs_positions)[i];
		const TVector3d& r_src = (*source_positions)[j];
		const TVector3d& m = (*magnetic_moments)[j];

		// Distance vector
		TVector3d r = r_obs - r_src;
		double r_mag = r.Abs();

		if(r_mag < 1e-10) return 0.0;  // Self-interaction

		// Normalized distance vector
		TVector3d r_hat = r / r_mag;

		// Biot-Savart tensor: K = (μ₀/4π) · [3·r̂⊗r̂ - I] / r³
		const double mu0_over_4pi = 1e-7;  // T·m/A
		double r3 = r_mag * r_mag * r_mag;

		// K_ij = (μ₀/4π) · [3·r̂_i·r̂_j - δ_ij] / r³
		double K_ij = mu0_over_4pi / r3;
		K_ij *= (3.0 * r_hat[component_row] * r_hat[component_col]);

		if(component_row == component_col) {
			K_ij -= mu0_over_4pi / r3;  // Subtract δ_ij term
		}

		return K_ij * m[component_col];  // Pre-multiply by moment component
	}
};
```

**Note**: We need 9 H-matrices (3×3 tensor), similar to radintrc_hmat.cpp.

### 4.3. Phase 3: Implement B_comp() with H-Matrix ⭐

**File**: `src/core/radhmat.cpp:329-363`

**Current Code**:
```cpp
void radTHMatrixFieldSource::B_comp(radTField* FieldPtr)
{
	// Always falls back to direct calculation
	B_comp_direct(FieldPtr);
}
```

**New Implementation**:

```cpp
void radTHMatrixFieldSource::B_comp(radTField* FieldPtr)
{
	if(!FieldPtr) return;

	// Check if H-matrix is built
	if(!is_built || !hmatrix_data) {
		B_comp_direct(FieldPtr);
		return;
	}

	// Get observation point
	TVector3d P = FieldPtr->P;

	// For single-point evaluation, use direct calculation
	// (H-matrix overhead not worth it for M=1)
	if(num_elements < 500) {
		B_comp_direct(FieldPtr);
		return;
	}

	// Use H-matrix for large problems
	try {
		// Create temporary observation point list
		std::vector<TVector3d> obs_points = {P};

		// Build H-matrix for this observation point
		// (Or use cached H-matrix if available)
		BuildFieldHMatrix(obs_points);

		// Evaluate field using H-matrix-vector product
		TVector3d B = EvaluateFieldHMatrix(0);  // Index 0 = first observation point

		// Store result
		FieldPtr->B += B;

	} catch(...) {
		// Fallback to direct calculation
		B_comp_direct(FieldPtr);
	}
}
```

### 4.4. Phase 4: Implement Batch Evaluation ⭐⭐⭐

**Most Important for Performance**

**File**: `src/core/radhmat.cpp:378-449`

**New Implementation**:

```cpp
void radTHMatrixFieldSource::B_comp_batch(std::vector<radTField*>& fields)
{
	if(fields.empty()) return;

	int M = static_cast<int>(fields.size());

	std::cout << "[HMatrix] Batch field evaluation: " << M << " points" << std::endl;

	// Extract observation points
	std::vector<TVector3d> obs_points;
	obs_points.reserve(M);

	for(auto* field : fields) {
		if(field) {
			obs_points.push_back(field->P);
		}
	}

	// Check if we should use H-matrix
	bool use_hmatrix = (M >= 100 && num_elements >= 100);

	if(!use_hmatrix || !is_built) {
		// Use direct calculation (potentially with OpenMP)
		#ifdef _OPENMP
		#pragma omp parallel for schedule(dynamic) if(config.use_openmp && M > 10)
		for(int i = 0; i < M; i++) {
			if(fields[i]) {
				B_comp_direct(fields[i]);
			}
		}
		#else
		for(auto* field : fields) {
			if(field) B_comp_direct(field);
		}
		#endif
		return;
	}

	// Use H-matrix for batch evaluation
	try {
		auto start_time = std::chrono::high_resolution_clock::now();

		// Build M×N H-matrix
		BuildFieldHMatrix(obs_points);

		auto build_time = std::chrono::high_resolution_clock::now();
		double build_duration = std::chrono::duration<double>(build_time - start_time).count();

		std::cout << "[HMatrix] H-matrix constructed in " << build_duration * 1000 << " ms" << std::endl;

		// Evaluate all fields using H-matrix
		#pragma omp parallel for schedule(static) if(config.use_openmp)
		for(int i = 0; i < M; i++) {
			if(fields[i]) {
				TVector3d B = EvaluateFieldHMatrix(i);
				fields[i]->B += B;
			}
		}

		auto eval_time = std::chrono::high_resolution_clock::now();
		double eval_duration = std::chrono::duration<double>(eval_time - build_time).count();

		std::cout << "[HMatrix] Field evaluation in " << eval_duration * 1000 << " ms" << std::endl;
		std::cout << "[HMatrix] Total time: " << (build_duration + eval_duration) * 1000 << " ms" << std::endl;
		std::cout << "[HMatrix] Time per point: " << (build_duration + eval_duration) / M * 1e6 << " us" << std::endl;

	} catch(const std::exception& e) {
		std::cerr << "[HMatrix] Error during batch evaluation: " << e.what() << std::endl;
		std::cerr << "[HMatrix] Falling back to direct calculation" << std::endl;

		// Fallback
		for(auto* field : fields) {
			if(field) B_comp_direct(field);
		}
	}
}
```

### 4.5. Phase 5: New Helper Functions

```cpp
// Build H-matrix for observation points
bool radTHMatrixFieldSource::BuildFieldHMatrix(const std::vector<TVector3d>& obs_points)
{
	int M = obs_points.size();

	std::cout << "[HMatrix] Building field H-matrix: " << M << " obs × "
	          << num_elements << " sources" << std::endl;

	// Convert to hacapk::Point3D
	std::vector<hacapk::Point3D> target_points;
	target_points.reserve(M);
	for(const auto& p : obs_points) {
		target_points.emplace_back(p.x, p.y, p.z);
	}

	std::vector<hacapk::Point3D> source_points;
	source_points.reserve(num_elements);
	for(int i = 0; i < num_elements; i++) {
		source_points.emplace_back(
			element_positions[3*i],
			element_positions[3*i+1],
			element_positions[3*i+2]
		);
	}

	// Build 9 H-matrices (3×3 tensor)
	field_hmatrices.resize(9);

	#pragma omp parallel for schedule(dynamic) if(config.use_openmp && num_elements > 100)
	for(int idx = 0; idx < 9; idx++) {
		int row = idx / 3;
		int col = idx % 3;

		BiotSavartKernel kernel;
		kernel.component_row = row;
		kernel.component_col = col;
		kernel.source_positions = &source_points;
		kernel.obs_positions = &target_points;
		kernel.magnetic_moments = &element_moments;

		hacapk::ControlParams params;
		params.eps_aca = config.eps;
		params.leaf_size = config.min_cluster_size;

		field_hmatrices[idx] = hacapk::build_hmatrix(
			target_points,  // M observation points
			source_points,  // N source points
			kernel,
			&kernel,
			params
		);
	}

	return true;
}

// Evaluate field at observation point i using H-matrix
TVector3d radTHMatrixFieldSource::EvaluateFieldHMatrix(int obs_index)
{
	// B_i = Σ_j K_ij · m_j
	// Using H-matrix: B = H · m

	TVector3d B(0, 0, 0);

	// For each component (x, y, z)
	for(int component = 0; component < 3; component++) {
		double B_component = 0.0;

		// Sum over 3 moment components
		for(int j_comp = 0; j_comp < 3; j_comp++) {
			int matrix_idx = component * 3 + j_comp;

			// Get row from H-matrix corresponding to observation point
			// This is where H-matrix acceleration happens!
			double contrib = field_hmatrices[matrix_idx]->get_row_contribution(obs_index);

			B_component += contrib;
		}

		B[component] = B_component;
	}

	return B;
}
```

---

## 5. API Integration

### 5.1. Python API Extension

**File**: `src/python/radpy.cpp`

Add new function for batch evaluation:

```python
# Current API
B = rad.Fld(magnet, 'b', [x, y, z])           # Single point
B_list = rad.Fld(magnet, 'b', [[x1,y1,z1], [x2,y2,z2], ...])  # Batch

# New H-matrix enabled API (automatically used if available)
B_list = rad.Fld(magnet, 'b', points, use_hmatrix=True)  # Explicit H-matrix
```

**Implementation**:
```cpp
// radpy.cpp
py::list RadiaFld_Batch(int ElemKey, const std::string& FieldChar, py::list points_list)
{
	// Extract observation points
	std::vector<TVector3d> obs_points;
	for(auto point : points_list) {
		py::list p = point.cast<py::list>();
		obs_points.emplace_back(
			p[0].cast<double>(),
			p[1].cast<double>(),
			p[2].cast<double>()
		);
	}

	// Check if element supports H-matrix field evaluation
	radTHMatrixFieldSource* hmat_source =
		dynamic_cast<radTHMatrixFieldSource*>(elem);

	if(hmat_source && obs_points.size() >= 100) {
		// Use H-matrix batch evaluation
		std::vector<radTField*> fields;
		for(const auto& p : obs_points) {
			fields.push_back(new radTField(FieldKey, CompCriterium, p, ...));
		}

		hmat_source->B_comp_batch(fields);

		// Extract results
		py::list results;
		for(auto* field : fields) {
			results.append(py::make_tuple(field->B.x, field->B.y, field->B.z));
			delete field;
		}
		return results;
	} else {
		// Use standard evaluation
		return RadiaFld_Standard(ElemKey, FieldChar, points_list);
	}
}
```

### 5.2. NGSolve Integration

**File**: `src/python/rad_ngsolve.cpp`

Modify `Evaluate()` to use H-matrix batch evaluation:

```cpp
void Evaluate(const BaseMappedIntegrationRule& mir,
              BareSliceMatrix<> result) const override
{
	size_t npts = mir.Size();

	// Extract all evaluation points
	std::vector<TVector3d> obs_points;
	obs_points.reserve(npts);

	for(size_t i = 0; i < npts; i++) {
		auto pnt = mir[i].GetPoint();
		obs_points.emplace_back(
			pnt[0] * 1000.0,  // m → mm
			pnt[1] * 1000.0,
			pnt[2] * 1000.0
		);
	}

	// Use H-matrix batch evaluation if available
	radTHMatrixFieldSource* hmat_source = GetHMatrixSource(radia_obj);

	if(hmat_source) {
		// H-matrix batch evaluation
		std::vector<TVector3d> B_values = hmat_source->EvaluateBatch(obs_points);

		for(size_t i = 0; i < npts; i++) {
			result(0, i) = B_values[i].x * scale;
			result(1, i) = B_values[i].y * scale;
			result(2, i) = B_values[i].z * scale;
		}
	} else {
		// Standard evaluation (existing code)
		// ...
	}
}
```

---

## 6. Expected Performance

### 6.1. Single Point Evaluation

| N Elements | Direct (ms) | H-Matrix (ms) | Speedup | Note |
|------------|-------------|---------------|---------|------|
| 100 | 0.03 | 0.05 | 0.6x | Overhead dominates |
| 500 | 0.15 | 0.10 | 1.5x | Small benefit |
| 1000 | 0.30 | 0.12 | 2.5x | Worthwhile |
| 5000 | 1.50 | 0.15 | 10x | Good |

**Conclusion**: Single point evaluation benefits only for N > 500

### 6.2. Batch Evaluation (M=1000 points)

| N Elements | Direct (ms) | H-Matrix (ms) | Speedup | Construction | Eval |
|------------|-------------|---------------|---------|--------------|------|
| 100 | 30 | 25 | 1.2x | 15 ms | 10 ms |
| 500 | 150 | 40 | 3.8x | 25 ms | 15 ms |
| 1000 | 300 | 50 | 6.0x | 30 ms | 20 ms |
| 5000 | 1500 | 100 | 15x | 60 ms | 40 ms |
| 10000 | 3000 | 150 | 20x | 90 ms | 60 ms |

**Key insight**: H-matrix construction overhead amortized over M=1000 points

### 6.3. NGSolve Integration (5000 mesh points, N=1000 elements)

| Method | Time (ms) | Details |
|--------|-----------|---------|
| Current (element-wise, 4pts/element) | 140 | 1250 calls × 0.11 ms |
| H-Matrix (batch all vertices) | 50 | 1 call × 50 ms |
| **Speedup** | **2.8x** | |

**With larger problems (N=5000, M=20000)**:
- Current: ~2000 ms
- H-Matrix: ~200 ms
- **Speedup: 10x**

---

## 7. Implementation Risks

### 7.1. Technical Risks

| Risk | Severity | Mitigation |
|------|----------|------------|
| Accuracy degradation | High | Rigorous testing vs direct sum |
| Memory overhead | Medium | Cache management, M×N threshold |
| Build time overhead | Medium | Use for M > 100 only |
| API compatibility | High | Maintain backward compatibility |
| Thread safety | Medium | Proper OpenMP critical sections |

### 7.2. API Compatibility Issues

**Problem**: radTHMatrixFieldSource needs to fit into radTg3d hierarchy

**Current hierarchy**:
```
radTg3d (base class)
  ├─ radTg3dRelax (relaxable elements)
  │   ├─ radTRecMag
  │   ├─ radTPolyhedron
  │   └─ ...
  └─ radTGroup (container)
```

**Question**: Where does radTHMatrixFieldSource fit?

**Option A**: Make it a wrapper (current design)
```cpp
// Create magnet normally
magnet = rad.ObjCnt([blocks...])

// Wrap with H-matrix field source (opt-in)
hmat_magnet = rad.ObjHMatrixFieldSource(magnet)

// Use normally
B = rad.Fld(hmat_magnet, 'b', points)  // Uses H-matrix automatically
```

**Option B**: Transparent integration (automatic)
```cpp
// Just use normally
magnet = rad.ObjCnt([blocks...])
B = rad.Fld(magnet, 'b', points)  // Auto-detects if H-matrix beneficial
```

**Recommendation**: Option B (transparent) for better user experience

### 7.3. Testing Strategy

**Unit tests**:
1. ✅ Accuracy: H-matrix vs direct sum (< 0.1% error)
2. ✅ Edge cases: Single element, self-field, zero magnetization
3. ✅ Memory: No leaks, proper cleanup
4. ✅ Thread safety: Parallel batch evaluation

**Integration tests**:
1. ✅ Python API: rad.Fld() with H-matrix
2. ✅ NGSolve: GridFunction.Set() with H-matrix
3. ✅ Performance: Benchmark suite

**Regression tests**:
1. ✅ Existing examples still work
2. ✅ No performance degradation for small problems

---

## 8. Implementation Timeline

### Week 1: Core Implementation
- ✅ Day 1: Fix geometry extraction
- ✅ Day 2: Implement Biot-Savart kernel (9 H-matrices)
- ✅ Day 3: Implement BuildFieldHMatrix()
- ✅ Day 4: Implement EvaluateFieldHMatrix()
- ✅ Day 5: Testing and debugging

### Week 2: API Integration
- ✅ Day 6: Modify B_comp() and B_comp_batch()
- ✅ Day 7: Python API extension
- ✅ Day 8: NGSolve integration
- ✅ Day 9: Enable in CMakeLists.txt
- ✅ Day 10: Build and test

### Week 3: Optimization & Documentation
- ✅ Day 11: Performance benchmarking
- ✅ Day 12: Memory optimization
- ✅ Day 13: OpenMP tuning
- ✅ Day 14: Documentation
- ✅ Day 15: Final testing

**Total**: ~15 working days

---

## 9. Success Criteria

### Performance Goals

| Problem Size | Speedup Target | Status |
|--------------|----------------|--------|
| M=100, N=500 | 2x | TBD |
| M=1000, N=1000 | 6x | TBD |
| M=5000, N=5000 | 20x | TBD |
| M=10000, N=10000 | 50x | TBD |

### Quality Goals

- ✅ Accuracy: < 0.1% relative error vs direct sum
- ✅ Memory: < 2x overhead vs direct calculation
- ✅ Stability: No crashes, proper error handling
- ✅ Compatibility: All existing tests pass

---

## 10. Decision Points

### Decision 1: When to Use H-Matrix?

**Threshold**:
```cpp
bool use_hmatrix = (M >= 100 && N >= 100);
```

**Rationale**: Below this threshold, overhead exceeds benefit

### Decision 2: Cache Strategy?

**Proposal**: Cache H-matrix for repeated evaluations

**Benefits**:
- NGSolve calls Evaluate() thousands of times
- Reusing cached H-matrix = 100x speedup

**Costs**:
- Memory overhead
- Complexity

**Decision**: Implement caching for M > 1000

### Decision 3: Tensor vs Scalar H-Matrix?

**Option A**: 9 H-matrices (3×3 tensor) ← **Recommended**
- Accurate Biot-Savart law
- Memory: 9× overhead
- Follows radintrc_hmat.cpp design

**Option B**: 1 scalar H-matrix
- Approximate field (magnitude only)
- Memory: 1× overhead
- Less accurate

**Decision**: Use 9 H-matrices for accuracy

---

## 11. Next Steps

### Immediate Actions

1. **Review this design document** ← We are here
2. **Get approval to proceed**
3. **Start Phase 1: Fix geometry extraction**

### Implementation Order

```
Phase 1: Geometry extraction      [1 day]
    ↓
Phase 2: Biot-Savart kernel       [1 day]
    ↓
Phase 3: BuildFieldHMatrix()      [2 days]
    ↓
Phase 4: EvaluateFieldHMatrix()   [1 day]
    ↓
Phase 5: B_comp() integration     [1 day]
    ↓
Phase 6: B_comp_batch()           [2 days]
    ↓
Phase 7: Testing                  [2 days]
    ↓
Phase 8: CMakeLists.txt enable    [1 day]
    ↓
Phase 9: Benchmarking             [2 days]
    ↓
Done! (13 days)
```

---

## 12. Questions for Review

1. **Architecture**: Is the two-level design (single + batch) appropriate?
2. **API**: Should H-matrix be transparent or opt-in?
3. **Caching**: Should we cache H-matrix for repeated evaluations?
4. **Threshold**: Is M≥100, N≥100 the right threshold?
5. **Testing**: What additional tests are needed?

---

**Status**: ✅ Design review complete, ready for implementation approval

**Next**: Await user decision to proceed with implementation
