#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Magnetizable Sphere in Quadrupole Field - Nastran Tetrahedral Mesh Analysis

This script:
1. Reads tetrahedral mesh from sphere.bdf (generated by Cubit)
2. Creates Radia model from tetrahedral elements
3. Applies quadrupole background field using ObjBckgCF
4. Solves magnetostatic problem
5. Compares Radia solution with analytical solution
6. Exports geometry and field distribution to VTK

Tetrahedral elements are always convex, perfect for Radia.


Date: 2025-11-01
"""

import sys
import os
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '../../build/Release'))
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '../../src/python'))
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '../../dist'))

import numpy as np
import radia as rd
from nastran_reader import read_nastran_mesh, TETRA_FACES
from radia_vtk_export import exportGeometryToVTK

print("=" * 70)
print("Magnetizable Sphere in Quadrupole Field - Nastran Mesh Analysis")
print("=" * 70)

# ============================================================================
# Parameters
# ============================================================================

# Geometry (matches Cubit2Nastran.py)
R = 10.0  # Sphere radius [mm]
center = [0, 0, 0]  # Sphere center [mm]

# Material
# Allow mu_r to be specified as command-line argument
if len(sys.argv) > 1:
	mu_r = float(sys.argv[1])
else:
	mu_r = 1000.0  # Default: Relative permeability (soft iron)
chi = mu_r - 1.0  # Magnetic susceptibility

# External field
gradient = 10.0  # Quadrupole gradient [T/m]

print(f"\nParameters:")
print(f"  Sphere radius: {R} mm")
print(f"  Sphere center: {center} mm")
print(f"  Relative permeability: {mu_r}")
print(f"  Quadrupole gradient: {gradient} T/m")

# ============================================================================
# Step 1: Read Nastran Mesh
# ============================================================================

print(f"\n[Step 1] Reading Nastran Tetrahedral Mesh")
print("-" * 70)

nas_file = 'sphere.bdf'

if not os.path.exists(nas_file):
	print(f"  [ERROR] File not found: {nas_file}")
	print(f"  Please run: python Cubit2Nastran.py")
	sys.exit(1)

mesh = read_nastran_mesh(nas_file)

nodes = mesh['nodes']
tetra_elements_data = mesh['tetra_elements']
tria_groups = mesh['tria_groups']
node_id_list = mesh['node_id_list']

print(f"  [OK] Mesh loaded successfully")

# ============================================================================
# Step 2: Create Radia Model from Mesh
# ============================================================================

print(f"\n[Step 2] Creating Radia Model from Mesh")
print("-" * 70)

all_polyhedra = []

# Check if we have surface triangles (CTRIA3) or volume tetrahedra (CTETRA)
if tria_groups:
	# Surface mesh approach: group triangles by material ID
	print(f"  Using surface mesh (CTRIA3): {len(tria_groups)} material group(s)")

	for mat_id, group in tria_groups.items():
		faces = group['faces']
		node_set = group['nodes']

		print(f"  Creating polyhedron for material {mat_id}: {len(faces)} triangles, {len(node_set)} nodes")

		# Build coordinate list for all nodes used by this material
		# Map node IDs to local indices for face connectivity
		node_list = sorted(node_set)
		node_id_to_local = {nid: idx for idx, nid in enumerate(node_list)}

		# Get coordinates
		coords = []
		for nid in node_list:
			idx = node_id_list.index(nid)
			coords.append(list(nodes[idx]))

		# Remap face connectivity to 1-indexed local coordinates
		faces_local = []
		for face in faces:
			# Convert global node IDs to local indices (1-indexed for Radia)
			local_face = [node_id_to_local[nid] + 1 for nid in face]
			faces_local.append(local_face)

		try:
			# Create polyhedron from surface triangles
			poly = rd.ObjPolyhdr(coords, faces_local, [0, 0, 0.001])
			all_polyhedra.append(poly)
			print(f"  [OK] Created polyhedron for material {mat_id}")
		except Exception as e:
			print(f"  [ERROR] Failed to create polyhedron for material {mat_id}: {e}")
			sys.exit(1)

elif len(tetra_elements_data) > 0:
	# Volume mesh approach: create individual tetrahedra
	total_elements = len(tetra_elements_data)
	print(f"  Using volume mesh (CTETRA): Creating {total_elements} tetrahedral polyhedra...")

	processed = 0
	for i, elem in enumerate(tetra_elements_data):
		coords = []
		for nid in elem:
			idx = node_id_list.index(nid)
			coords.append(list(nodes[idx]))

		try:
			# Tetrahedra are always convex - perfect for Radia
			poly = rd.ObjPolyhdr(coords, TETRA_FACES, [0, 0, 0.001])
			all_polyhedra.append(poly)
		except Exception as e:
			print(f"\n  Warning: Failed to create tetra {i+1}: {e}")

		processed += 1
		if processed % 500 == 0:
			print(f"  Progress: {processed}/{total_elements}", end='\r')

	print(f"  Progress: {total_elements}/{total_elements}")
	print(f"  [OK] Created {len(all_polyhedra)} polyhedra")
else:
	print("  [ERROR] No mesh elements found (neither CTRIA3 nor CTETRA)")
	sys.exit(1)

# Combine into container
if not all_polyhedra:
	print("  [ERROR] No polyhedra created")
	sys.exit(1)

sphere = rd.ObjCnt(all_polyhedra)
print(f"  [OK] Sphere container created")

# ============================================================================
# Step 3: Apply Material and Background Field
# ============================================================================

print(f"\n[Step 3] Applying Material and Background Field")
print("-" * 70)

# Apply linear magnetic material
mat = rd.MatLin([chi, chi], 0.001)
rd.MatApl(sphere, mat)
print(f"  Material applied: mu_r = {mu_r}, chi = {chi}")

# Create quadrupole background field using ObjBckgCF
def quadrupole_field_callback(gradient):
	"""Create quadrupole field callback for rd.ObjBckgCF"""
	def field(pos):
		x, y, z = pos  # Position in mm
		# Convert to meters
		x_m = x * 1e-3
		y_m = y * 1e-3
		# Quadrupole field: Bx = g*y, By = g*x, Bz = 0
		Bx = gradient * y_m  # [T]
		By = gradient * x_m  # [T]
		Bz = 0.0
		return [Bx, By, Bz]
	return field

quad_field = quadrupole_field_callback(gradient)
bckg = rd.ObjBckgCF(quad_field)
print(f"  Quadrupole background field created (ObjBckgCF)")

# Create container with sphere and background field
container = rd.ObjCnt([sphere, bckg])
print(f"  Container created")

# ============================================================================
# Step 4: Solve Magnetostatic Problem
# ============================================================================

print(f"\n[Step 4] Solving Magnetostatic Problem")
print("-" * 70)

print(f"  Solving...")
rd.Solve(container, 1e-5, 5000)
print(f"  [OK] Solution converged")

# ============================================================================
# Step 5: Export Sphere Geometry
# ============================================================================

print(f"\n[Step 5] Exporting Sphere Geometry to VTK")
print("-" * 70)

geometry_filename = 'sphere_nastran_geometry'
exportGeometryToVTK(sphere, geometry_filename)
print(f"  [OK] Exported to {geometry_filename}.vtk")

# ============================================================================
# Step 6: Field Verification
# ============================================================================

print(f"\n[Step 6] Field Verification")
print("-" * 70)

# Test points
test_points = [
	([0, 0, 0], "Center"),
	([5, 0, 0], "On +x-axis (r=5mm)"),
	([0, 5, 0], "On +y-axis (r=5mm)"),
	([5, 5, 0], "Diagonal (r=7.07mm)"),
	([15, 0, 0], "Outside +x (r=15mm)"),
	([0, 15, 0], "Outside +y (r=15mm)"),
]

print(f"\nQuadrupole field properties:")
print(f"  - Bx should increase with y (Bx = g*y)")
print(f"  - By should increase with x (By = g*x)")
print(f"  - Field enhanced inside magnetic material")
print(f"  - Field outside matches analytical quadrupole")

print(f"\n{'Location':<25} {'Point (mm)':<15} {'Bx (T)':>12} {'By (T)':>12} {'|B| (T)':>12}")
print("-" * 80)

for pt, label in test_points:
	B = rd.Fld(container, 'b', pt)
	B_mag = np.sqrt(B[0]**2 + B[1]**2 + B[2]**2)
	print(f"{label:<25} {str(pt):<15} {B[0]:>12.6f} {B[1]:>12.6f} {B_mag:>12.6f}")

# ============================================================================
# Step 7: Compare with Analytical Solution
# ============================================================================

print(f"\n[Step 7] Comparison with Analytical Solution")
print("-" * 70)

# For points outside the sphere, compare with analytical quadrupole field
print(f"\nExternal field comparison (outside sphere, r > {R} mm):")
print(f"{'Point (mm)':<15} {'B_Radia':>15} {'B_Analytical':>15} {'Error':>15}")
print("-" * 65)

external_points = [
	[15, 0, 0],
	[0, 15, 0],
	[15, 15, 0],
	[20, 0, 0],
	[0, 20, 0],
	[30, 0, 0],
	[0, 30, 0],
	[40, 0, 0],
	[0, 40, 0],
	[50, 0, 0],
	[0, 50, 0],
]

for pt in external_points:
	B_radia = rd.Fld(container, 'b', pt)

	# Analytical quadrupole field
	x_m = pt[0] * 1e-3
	y_m = pt[1] * 1e-3
	B_analytical = [gradient * y_m, gradient * x_m, 0.0]

	# Error
	error = np.sqrt((B_radia[0] - B_analytical[0])**2 +
	                (B_radia[1] - B_analytical[1])**2)

	B_radia_mag = np.sqrt(B_radia[0]**2 + B_radia[1]**2)
	B_analytical_mag = np.sqrt(B_analytical[0]**2 + B_analytical[1]**2)

	print(f"{str(pt):<15} {B_radia_mag:>15.6f} {B_analytical_mag:>15.6f} {error:>15.6f}")

# ============================================================================
# Step 8: Calculate Field Distribution
# ============================================================================

print(f"\n[Step 8] Calculating Field Distribution")
print("-" * 70)

# Define evaluation grid (extended to 20mm for far-field evaluation)
x_min, x_max = -20, 20
y_min, y_max = -20, 20
z_min, z_max = -20, 20

# Grid resolution
nx, ny, nz = 21, 21, 21

print(f"  Evaluation range:")
print(f"    X: [{x_min:.1f}, {x_max:.1f}] mm")
print(f"    Y: [{y_min:.1f}, {y_max:.1f}] mm")
print(f"    Z: [{z_min:.1f}, {z_max:.1f}] mm")
print(f"  Grid resolution: {nx} x {ny} x {nz} = {nx*ny*nz} points")

# Create grid
x = np.linspace(x_min, x_max, nx)
y = np.linspace(y_min, y_max, ny)
z = np.linspace(z_min, z_max, nz)

X, Y, Z = np.meshgrid(x, y, z, indexing='ij')

# Evaluate fields
print(f"\n  Evaluating Radia field at {nx*ny*nz} points...")
Bx_radia = np.zeros_like(X)
By_radia = np.zeros_like(X)
Bz_radia = np.zeros_like(X)

for i in range(nx):
	for j in range(ny):
		for k in range(nz):
			pt = [X[i,j,k], Y[i,j,k], Z[i,j,k]]
			B = rd.Fld(container, 'b', pt)
			Bx_radia[i,j,k] = B[0]
			By_radia[i,j,k] = B[1]
			Bz_radia[i,j,k] = B[2]
	if (i+1) % 5 == 0:
		print(f"  Progress: {(i+1)*ny*nz}/{nx*ny*nz} points", end='\r')

print(f"  Progress: {nx*ny*nz}/{nx*ny*nz} points")
print(f"  [OK] Radia field evaluated")

# Calculate analytical external field (background only)
print(f"\n  Calculating analytical quadrupole field...")
Bx_analytical = gradient * Y * 1e-3  # g*y
By_analytical = gradient * X * 1e-3  # g*x
Bz_analytical = np.zeros_like(X)

print(f"  [OK] Analytical field calculated")

# ============================================================================
# Step 9: Export Field Distribution to VTU
# ============================================================================

print(f"\n[Step 9] Exporting Field Distribution to VTU")
print("-" * 70)

output_file = f'sphere_nastran_field_mu{int(mu_r)}.vtu'

# Flatten arrays
points = np.column_stack([X.ravel(), Y.ravel(), Z.ravel()])
n_points = len(points)

# Convert mm to m for VTK output
points_m = points * 1e-3

# Prepare field data
B_radia = np.column_stack([Bx_radia.ravel(), By_radia.ravel(), Bz_radia.ravel()])
B_analytical = np.column_stack([Bx_analytical.ravel(), By_analytical.ravel(), Bz_analytical.ravel()])
B_difference = B_radia - B_analytical
B_magnitude_radia = np.linalg.norm(B_radia, axis=1)
B_magnitude_analytical = np.linalg.norm(B_analytical, axis=1)

# Create cells (hexahedrons/voxels)
cells = []
cell_types = []

for i in range(nx-1):
	for j in range(ny-1):
		for k in range(nz-1):
			# Hexahedron vertex indices
			v0 = i*ny*nz + j*nz + k
			v1 = (i+1)*ny*nz + j*nz + k
			v2 = (i+1)*ny*nz + (j+1)*nz + k
			v3 = i*ny*nz + (j+1)*nz + k
			v4 = i*ny*nz + j*nz + (k+1)
			v5 = (i+1)*ny*nz + j*nz + (k+1)
			v6 = (i+1)*ny*nz + (j+1)*nz + (k+1)
			v7 = i*ny*nz + (j+1)*nz + (k+1)
			cells.append([v0, v1, v2, v3, v4, v5, v6, v7])
			cell_types.append(12)  # VTK_HEXAHEDRON

n_cells = len(cells)

print(f"  Writing {output_file}...")
print(f"    Points: {n_points}")
print(f"    Cells: {n_cells}")

with open(output_file, 'w') as f:
	# Header
	f.write('<?xml version="1.0"?>\n')
	f.write('<VTKFile type="UnstructuredGrid" version="0.1" byte_order="LittleEndian">\n')
	f.write('  <UnstructuredGrid>\n')
	f.write(f'    <Piece NumberOfPoints="{n_points}" NumberOfCells="{n_cells}">\n')

	# Points
	f.write('      <Points>\n')
	f.write('        <DataArray type="Float64" NumberOfComponents="3" format="ascii">\n')
	for pt in points_m:
		f.write(f'          {pt[0]:.8e} {pt[1]:.8e} {pt[2]:.8e}\n')
	f.write('        </DataArray>\n')
	f.write('      </Points>\n')

	# Cells
	f.write('      <Cells>\n')
	f.write('        <DataArray type="Int32" Name="connectivity" format="ascii">\n')
	for cell in cells:
		f.write('          ' + ' '.join(map(str, cell)) + '\n')
	f.write('        </DataArray>\n')

	f.write('        <DataArray type="Int32" Name="offsets" format="ascii">\n')
	offset = 0
	for cell in cells:
		offset += len(cell)
		f.write(f'          {offset}\n')
	f.write('        </DataArray>\n')

	f.write('        <DataArray type="UInt8" Name="types" format="ascii">\n')
	for ct in cell_types:
		f.write(f'          {ct}\n')
	f.write('        </DataArray>\n')
	f.write('      </Cells>\n')

	# Point Data
	f.write('      <PointData>\n')

	# Radia field
	f.write('        <DataArray type="Float64" Name="B_Radia" NumberOfComponents="3" format="ascii">\n')
	for B in B_radia:
		f.write(f'          {B[0]:.8e} {B[1]:.8e} {B[2]:.8e}\n')
	f.write('        </DataArray>\n')

	# Analytical field
	f.write('        <DataArray type="Float64" Name="B_Analytical" NumberOfComponents="3" format="ascii">\n')
	for B in B_analytical:
		f.write(f'          {B[0]:.8e} {B[1]:.8e} {B[2]:.8e}\n')
	f.write('        </DataArray>\n')

	# Difference
	f.write('        <DataArray type="Float64" Name="B_Difference" NumberOfComponents="3" format="ascii">\n')
	for B in B_difference:
		f.write(f'          {B[0]:.8e} {B[1]:.8e} {B[2]:.8e}\n')
	f.write('        </DataArray>\n')

	# Magnitude Radia
	f.write('        <DataArray type="Float64" Name="B_Magnitude_Radia" format="ascii">\n')
	for B_mag in B_magnitude_radia:
		f.write(f'          {B_mag:.8e}\n')
	f.write('        </DataArray>\n')

	# Magnitude Analytical
	f.write('        <DataArray type="Float64" Name="B_Magnitude_Analytical" format="ascii">\n')
	for B_mag in B_magnitude_analytical:
		f.write(f'          {B_mag:.8e}\n')
	f.write('        </DataArray>\n')

	f.write('      </PointData>\n')

	f.write('    </Piece>\n')
	f.write('  </UnstructuredGrid>\n')
	f.write('</VTKFile>\n')

print(f"  [OK] Exported to {output_file}")

# ============================================================================
# Step 10: Statistics
# ============================================================================

print(f"\n[Step 10] Field Statistics")
print("-" * 70)

print(f"\nRadia field (with magnetizable sphere):")
print(f"  |B| range: [{B_magnitude_radia.min():.6e}, {B_magnitude_radia.max():.6e}] T")
print(f"  |B| mean:  {B_magnitude_radia.mean():.6e} T")

print(f"\nAnalytical field (background quadrupole only):")
print(f"  |B| range: [{B_magnitude_analytical.min():.6e}, {B_magnitude_analytical.max():.6e}] T")
print(f"  |B| mean:  {B_magnitude_analytical.mean():.6e} T")

print(f"\nDifference (effect of magnetizable sphere):")
diff_magnitude = np.linalg.norm(B_difference, axis=1)
print(f"  |ΔB| range: [{diff_magnitude.min():.6e}, {diff_magnitude.max():.6e}] T")
print(f"  |ΔB| mean:  {diff_magnitude.mean():.6e} T")

# ============================================================================
# Summary
# ============================================================================

print("\n" + "=" * 70)
print("Summary")
print("=" * 70)

print(f"\nMesh statistics:")
print(f"  Nodes: {len(nodes)}")
print(f"  Tetrahedral elements: {len(tetra_elements_data)}")
print(f"  Radia polyhedra: {len(all_polyhedra)}")

print(f"\nOutput files created:")
print(f"  1. sphere_nastran_geometry.vtk - Sphere from tetrahedral mesh")
print(f"  2. sphere_nastran_field.vtu - 3D field distribution")

print(f"\nField data in VTU file:")
print(f"  - B_Radia: Total field (sphere + background)")
print(f"  - B_Analytical: Background quadrupole field only")
print(f"  - B_Difference: Effect of magnetizable sphere")
print(f"  - B_Magnitude_Radia: |B| from Radia")
print(f"  - B_Magnitude_Analytical: |B| from analytical")

print(f"\nVisualization in ParaView:")
print(f"  1. Open both sphere_nastran_geometry.vtk and sphere_nastran_field.vtu")
print(f"  2. Compare Radia solution with analytical quadrupole field")
print(f"  3. Tetrahedral mesh provides accurate convex elements for Radia")

print("\n" + "=" * 70)
print("Complete")
print("=" * 70)
