#!/usr/bin/env python
"""
PyVista-based viewer for Radia objects
This is a modern alternative to rad.ObjDrwOpenGL()
"""

import sys, os
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..', 'dist'))
sys.path.insert(0, os.path.join(os.path.dirname(__file__), '..', '..', 'build', 'lib', 'Release'))

import radia as rad
import numpy as np

def view_radia_object(obj, plotter=None, show=True, color=None, opacity=1.0):
	"""
	View a Radia object using PyVista

	Parameters:
	-----------
	obj : int
		Radia object ID
	plotter : pyvista.Plotter, optional
		Existing plotter to add to (for combining multiple objects)
	show : bool, default True
		Whether to show the plotter immediately
	color : str or list, optional
		Color override (e.g., 'red', [1,0,0])
	opacity : float, default 1.0
		Opacity (0=transparent, 1=opaque)

	Returns:
	--------
	plotter : pyvista.Plotter
		The plotter object (can be reused)
	"""
	try:
		import pyvista as pv
	except ImportError:
		print("ERROR: PyVista is not installed.")
		print("Install it with: pip install pyvista")
		return None

	# Get VTK data from Radia
	vtk_data = rad.ObjDrwVTK(obj, 'Axes->False')

	if 'polygons' not in vtk_data:
		print("ERROR: No polygon data found in Radia object")
		return None

	poly = vtk_data['polygons']

	# Extract vertices (flatten x,y,z triplets)
	vertices = np.array(poly['vertices']).reshape(-1, 3)

	# Extract polygon connectivity
	lengths = poly['lengths']
	n_polys = len(lengths)

	# Build faces array for PyVista
	# PyVista format: [n_points, p0, p1, p2, ..., n_points, p0, p1, ...]
	faces = []
	offset = 0
	for length in lengths:
		faces.append(length)
		for i in range(length):
			faces.append(offset + i)
		offset += length

	faces = np.array(faces)

	# Create PyVista mesh
	mesh = pv.PolyData(vertices, faces)

	# Create or use existing plotter
	if plotter is None:
		plotter = pv.Plotter()
		plotter.add_axes()
		plotter.add_text("Radia Object Viewer (PyVista)", position='upper_left', font_size=10)

	# Determine color
	if color is None:
		# Use Radia's color if available
		colors = poly.get('colors', None)
		if colors is not None and len(colors) > 0:
			# Use first color (all polygons typically same color)
			rgb = colors[0:3]
			# Radia uses -1 for default, convert to actual color
			if rgb[0] < 0:
				color = 'lightblue'
			else:
				color = rgb

	# Add mesh to plotter
	plotter.add_mesh(mesh, color=color, opacity=opacity, show_edges=True,
	                 edge_color='black', line_width=0.5)

	if show:
		plotter.show()

	return plotter


def view_radia_vtk_file(filename):
	"""
	View a VTK file generated by exportGeometryToVTK

	Parameters:
	-----------
	filename : str
		Path to .vtk file
	"""
	try:
		import pyvista as pv
	except ImportError:
		print("ERROR: PyVista is not installed.")
		print("Install it with: pip install pyvista")
		return

	# Read VTK file
	mesh = pv.read(filename)

	# Create plotter
	plotter = pv.Plotter()
	plotter.add_axes()
	plotter.add_text(f"Viewing: {os.path.basename(filename)}",
	                 position='upper_left', font_size=10)

	# Add mesh
	plotter.add_mesh(mesh, show_edges=True, edge_color='black', line_width=0.5)

	# Show
	plotter.show()


if __name__ == '__main__':
	print("="*60)
	print("Radia PyVista Viewer Demo")
	print("="*60)

	# Check if PyVista is available
	try:
		import pyvista as pv
		print(f"\nPyVista version: {pv.__version__}")
	except ImportError:
		print("\nERROR: PyVista is not installed.")
		print("Install it with:")
		print("  pip install pyvista")
		sys.exit(1)

	# Create a simple test object
	print("\nCreating test objects...")

	# Arc current
	g1 = rad.ObjArcCur([0,0,0], [95, 105], [0, 2*np.pi], 20, 36, 1e6/20/10, "man")

	# Rectangular magnet
	g2 = rad.ObjRecMag([0,0,0], [30,30,10], [0,0,1])

	# Container
	g = rad.ObjCnt([g1, g2])

	print("Objects created. Opening PyVista viewer...")
	print("\nControls:")
	print("  - Left click + drag: Rotate")
	print("  - Right click + drag: Pan")
	print("  - Scroll wheel: Zoom")
	print("  - 'q': Quit")

	# View the object
	view_radia_object(g)

	print("\nViewer closed.")
